package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.direct.ytwrapper.utils.FreshestClusterFinder
import ru.yandex.misc.io.ClassPathResourceInputStreamSource

/**
 * Джоба для удаления более не используемых в Директе сайтов для превью из MBI.
 * Выполняется раз в сутки, так как выгрузка текущих сайтов для превью из MBI в YT выполняется с той же периодичностью
 */
@JugglerCheck(ttl = JugglerCheck.Duration(days = 2, hours = 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = ProductionOnly::class)
class DeleteSitePreviewsFromMBIJob @Autowired constructor(private val ytProvider: YtProvider,
                                                          private val mbiService: MbiService) : DirectJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(DeleteSitePreviewsFromMBIJob::class.java)

        private val SITE_PREVIEWS_TO_DELETE_QUERY =
            ClassPathResourceInputStreamSource("uac/absentedSitePreviewsInMBI.yql")
                .readLines()
                .joinToString("\n")

        private const val SHOP_ID = "shop_id"
        private const val FEED_ID = "feed_id"

        private const val YT_SYNC_TABLE_PATH = "//home/direct/mysql-sync/current/ppcdict/straight/ecom_domains"

        private val YT_SYNC_TABLE = YtTable(YT_SYNC_TABLE_PATH)

        private val CLUSTERS = listOf(YtCluster.HAHN, YtCluster.ARNOLD)

        private const val FETCH_SIZE = 50_000
    }

    override fun execute() {
        val cluster = chooseFreshestCluster()
        if (cluster == null) {
            logger.error("No available clusters found")
            return
        }

        // Из YT берутся не более указанного количества записей для удаления
        // В среднем после обновления живости их 20-30 тыс.
        val sitePreviewsToDelete = ytProvider.getOperator(cluster)
            .yqlQuery(SITE_PREVIEWS_TO_DELETE_QUERY, {
                val shopId = it.getLong(SHOP_ID)
                val feedId = it.getLong(FEED_ID)
                EcomDomain()
                    .withMarketShopId(shopId)
                    .withMarketFeedId(feedId)
            }, FETCH_SIZE)

        if (sitePreviewsToDelete.isEmpty()) {
            logger.info("Site previews to delete were not found")
            return
        }

        logger.info("Deleting ${sitePreviewsToDelete.size} site previews from MBI")
        val deleted = mbiService.deleteSitePreviews(sitePreviewsToDelete)
        logger.info("Deleted $deleted site previews from MBI")
    }

    private fun chooseFreshestCluster(): YtCluster? {
        return FreshestClusterFinder.getFreshestCluster(ytProvider, YT_SYNC_TABLE, CLUSTERS)
    }
}
