package ru.yandex.direct.jobs.uac

import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.DELETE_UAC_ASSET_CURRENT_DATE_OFFSET
import ru.yandex.direct.common.db.PpcPropertyNames.DELETE_UAC_ASSET_MIN_LOOKUP_TIME
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.uac.service.DeleteUnusedAssetsFromMDSService
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob

/**
 * Джоба предназначена для удаления созданных в GrUT ассетов,
 * которые не добавлены ни в одну кампанию за установленное время. Количество дней, в течение которых считается,
 *      что ассет новый и его еще могут соединить с кампанией храним в проперти DELETE_UAC_ASSET_CURRENT_DATE_OFFSET.
 * Неиспользованные ассеты будут удалены из аватарницы и таблицы ассетов в GrUT
 * В работе используются следующие параметры:
 * DELETE_UAC_ASSET_CURRENT_DATE_OFFSET - проперти, хранит количество дней,
 *      на которые надо сдвинуться от текущей даты (верхняя граница поиска для ассетов)
 * DELETE_UAC_ASSET_MIN_LOOKUP_TIME - проперти, хранит минимальную дату для поиска ассетов
 *      (нижняя граница поиска)
 * DELETE_UAC_ASSET_IDLE_TIME - проперти(управление доступно из веб-интерфейса),
 *      хранит время бездействия между итерациями удаления(в миллисекундах)
 * Запускается джоба каждый час в ночное время("0 0 0-7,22-23 * * ?"),
 * если на момент запуска предыдущая итерация не завершена - запуск не случится
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 3),
    needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_AREVOLUTION, NotificationRecipient.LOGIN_PAVELKATAYKIN],
        method = [NotificationMethod.TELEGRAM]
    )]
)
@Hourglass(cronExpression = "0 */10 0-10,20-23 * * ?", needSchedule = NonDevelopmentEnvironment::class)
class DeleteUnusedAssetsFromMDSJob @Autowired constructor(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val deleteUnusedAssetsFromMDSService: DeleteUnusedAssetsFromMDSService
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(DeleteUnusedAssetsFromMDSJob::class.java)
    }

    override fun execute() {
        val minLookupTime = ppcPropertiesSupport
            .get(DELETE_UAC_ASSET_MIN_LOOKUP_TIME)
            .getOrDefault(LocalDateTime.MIN)

        val currentDateOffset = ppcPropertiesSupport
            .get(DELETE_UAC_ASSET_CURRENT_DATE_OFFSET)
            .getOrDefault(7)

        var maxLookupTime = LocalDateTime.now().minusDays(currentDateOffset)
        if (maxLookupTime.isBefore(minLookupTime)) {
            logger.info("Minimal lookup time is greater than maximum lookup time." +
                " Setting maximum lookup time to {}", minLookupTime)
            maxLookupTime = minLookupTime
        }

        logger.info("Launching asset deleting job for assets created from {} to {}",
            minLookupTime, maxLookupTime)

        val countDeleted = deleteUnusedAssetsFromMDSService.execute(
            minLookupTime,
            maxLookupTime
        )
        logger.info("{} assets where successfully deleted", countDeleted)
    }

}
