package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.uac.service.EcomCampaignsMonitoringService
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import ru.yandex.direct.solomon.SolomonPushClient
import ru.yandex.direct.solomon.SolomonPushClientException
import ru.yandex.direct.solomon.SolomonUtils

/**
 * Джоба для отправки в Solomon метрик про товарные кампании:
 * - Созданные товарные кампании
 * - Сломанные товарные кампании
 * - Товарные кампании, у которых статусы подкампаний отличаются от статуса родительской кампании
 */
@JugglerCheck(ttl = JugglerCheck.Duration(hours = 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_KOZOBRODOV],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 10, needSchedule = ProductionOnly::class)
class EcomCampaignsMonitoringJob @Autowired constructor(
    private val solomonPushClient: SolomonPushClient,
    private val ecomCampaignsMonitoringService: EcomCampaignsMonitoringService
): DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(EcomDomainStatusesMonitoringJob::class.java)

        private const val SOLOMON_LABEL = "EcomCampaignsMonitoringJob"
    }

    override fun execute() {
        val ecomCampaignsCount = ecomCampaignsMonitoringService.countEcomCampaigns(shard)
        val ecomCampaignsWithMissingSubcampaignsCount =
            ecomCampaignsMonitoringService.countEcomCampaignsWithoutAllSubcampaigns(shard)
        val ecomCampaignWithStatusDifference = ecomCampaignsMonitoringService.countEcomCampaignsWithStatusesDifference(shard)

        sendToSolomon(ecomCampaignsCount, ecomCampaignsWithMissingSubcampaignsCount, ecomCampaignWithStatusDifference)
    }

    private fun sendToSolomon(
        ecomCampaignsCount: Int,
        ecomCampaignsWithMissingSubcampaignsCount: Int,
        ecomCampaignWithStatusDifference: Int
    ) {
        logger.info("Sending ecom campaigns counts to Solomon")

        val registry = SolomonUtils.newPushRegistry("flow", SOLOMON_LABEL, "shard", shard.toString())
        registry.gaugeInt64("ecom_campaigns_count").set(ecomCampaignsCount.toLong())
        registry.gaugeInt64("ecom_campaigns_with_missing_subcampaigns_count")
            .set(ecomCampaignsWithMissingSubcampaignsCount.toLong())
        registry.gaugeInt64("ecom_campaigns_different_statuses_count").set(ecomCampaignWithStatusDifference.toLong())
        try {
            solomonPushClient.sendMetrics(registry)
        } catch (e: SolomonPushClientException) {
            logger.error("Got exception on sending metrics", e)
        }
    }
}
