package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.solomon.SolomonPushClient
import ru.yandex.direct.solomon.SolomonPushClientException
import ru.yandex.direct.solomon.SolomonUtils

/**
 * Джоба для мониторинга статусов еком доменов (```ppcdict.ecom_domains```): считает общее их количество, количество
 * отправленных в MBI, количество ждущих отправки в MBI, количество доменов с полученным превью из bannerland'а.
 * Посчитанные данные отправляются в Соломон с периодом, равным отправке доменов в MBI (2 часа).
 */
@JugglerCheck(ttl = JugglerCheck.Duration(hours = 5),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 2, needSchedule = ProductionOnly::class)
class EcomDomainStatusesMonitoringJob @Autowired constructor(
    private val solomonPushClient: SolomonPushClient,
    private val ecomDomainsRepository: EcomDomainsRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(EcomDomainStatusesMonitoringJob::class.java)

        private const val SOLOMON_LABEL = "EcomDomainStatusesMonitoringJob"

        private const val DEFAULT_DAYS_LIMIT_FOR_UPDATING_PREVIEW = 10L
    }

    override fun execute() {
        val minOffersCountProperty: PpcProperty<Long> =
            ppcPropertiesSupport.get(PpcPropertyNames.MIN_OFFERS_COUNT_BY_HOST_FOR_SEND_TO_MBI)
        val minOffersCount = minOffersCountProperty.getOrDefault(SendEcomDomainToMBIJob.DEFAULT_MIN_OFFERS_COUNT)

        val daysLimitForUpdatingPreviewProperty: PpcProperty<Long> =
            ppcPropertiesSupport.get(PpcPropertyNames.DAYS_LIMIT_FOR_UPDATING_PREVIEW)
        val daysLimitForUpdatingPreview =
            daysLimitForUpdatingPreviewProperty.getOrDefault(DEFAULT_DAYS_LIMIT_FOR_UPDATING_PREVIEW)

        val totalCount = ecomDomainsRepository.getEcomDomainsTotalCount()
        val sentToMbiCount = ecomDomainsRepository.getSentToMbiEcomDomainsCount()
        val notSentToMbiCount = ecomDomainsRepository.getNotSentToMbiEcomDomainsCount(minOffersCount)
        val domainsWithPreviewCount = ecomDomainsRepository.getEcomDomainsWithPreviewCount()
        val domainsWithoutUpdatingPreviewCount =
            ecomDomainsRepository.getEcomDomainsWithoutUpdatingPreviewCount(daysLimitForUpdatingPreview)

        sendToSolomon(totalCount, sentToMbiCount, notSentToMbiCount, domainsWithPreviewCount,
            domainsWithoutUpdatingPreviewCount)
    }

    private fun sendToSolomon(totalCount: Int, sentToMbiCount: Int, notSentToMbiCount: Int,
                              domainsWithPreviewCount: Int, domainsWithoutUpdatingPreviewCount: Int) {
        logger.info("Sending ecom domains statuses to Solomon")

        val registry = SolomonUtils.newPushRegistry("flow", SOLOMON_LABEL)
        registry.gaugeInt64("ecom_domains").set(totalCount.toLong())
        registry.gaugeInt64("ecom_domains_sent_to_MBI").set(sentToMbiCount.toLong())
        registry.gaugeInt64("ecom_domains_not_sent_to_MBI").set(notSentToMbiCount.toLong())
        registry.gaugeInt64("ecom_domains_with_preview").set(domainsWithPreviewCount.toLong())
        registry.gaugeInt64("ecom_domains_without_updating_preview")
            .set(domainsWithoutUpdatingPreviewCount.toLong())

        try {
            solomonPushClient.sendMetrics(registry)
        } catch (e: SolomonPushClientException) {
            logger.error("Got exception on sending metrics", e)
        }
    }
}
