package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.jobs.uac.model.LiveEcomDomainTableRow
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicInteger

/**
 * Джоба берет живые (есть переходы с СЕРПа за месяц) еком домены из yt таблички //home/direct/ecom/live_ecom_domains.
 * При этом новые домены добавляются в ppcdict.ecom_domains.
 * Если домен присутствует в ppcdict.ecom_domains и is_permanent = false, но отсутствует в живых еком доменах - он удаляется
 */
@JugglerCheck(ttl = JugglerCheck.Duration(days = 2, hours = 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = TypicalEnvironment::class)
class ImportEcomDomainsFromYtJob @Autowired constructor(
    private val ytProvider: YtProvider,
    private val ecomDomainsRepository: EcomDomainsRepository) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(ImportEcomDomainsFromYtJob::class.java)

        private const val FETCH_CHUNK_SIZE = 50_000
        private const val UPDATE_CHUNK_SIZE = 1000

        private val TABLE = YtTable("//home/direct/ecom/live_ecom_domains")

        private val CLUSTERS = listOf(YtCluster.HAHN, YtCluster.ARNOLD)
    }

    override fun execute() {
        val existingDomains = ConcurrentHashMap(ecomDomainsRepository.getAllDomainsToIds())
        logger.info("${existingDomains.size} domains found in ecom_domains")

        val addedDomains = AtomicInteger(0)
        val updatedDomains = AtomicInteger(0)

        YtExecutionUtil.executeWithFallback(CLUSTERS, ytProvider::getOperator) { operator ->
            operator.readTableSnapshot(TABLE, LiveEcomDomainTableRow(), LiveEcomDomainTableRow::toEcomDomain, { ecomDomains ->
                addedDomains.addAndGet(insertNewEcomDomains(ecomDomains, existingDomains))
                updatedDomains.addAndGet(updateEcomDomains(ecomDomains, existingDomains))
                existingDomains.keys.removeAll(ecomDomains.map { it.domain })
            }, FETCH_CHUNK_SIZE)
        }

        // Если домены, присутствующие в ecom_domains, не были найдены в таблице с живыми доменами - удаляем их
        val removedDomains = if (existingDomains.size > 0) {
            removeEcomDomains(existingDomains)
        } else 0

        logger.info("Added ${addedDomains.get()} domains to ecom_domains, updated $updatedDomains, removed $removedDomains")
    }

    fun insertNewEcomDomains(ecomDomains: List<EcomDomain>, existingDomains: Map<String, *>): Int {
        return ecomDomains
            .filter { !existingDomains.contains(it.domain) }
            .chunked(UPDATE_CHUNK_SIZE)
            .map {
                logger.info("Inserting ${it.size} domains")
                val inserted = ecomDomainsRepository.insert(it)
                logger.info("Inserted $inserted domains")
                inserted
            }
            .sum()
    }

    fun updateEcomDomains(ecomDomains: List<EcomDomain>, existingDomains: Map<String, Long>): Int {
        return ecomDomains
            .asSequence()
            .filter { existingDomains.containsKey(it.domain) }
            .associate { existingDomains[it.domain]!! to it.offersCount }
            .map {
                ModelChanges(it.key, EcomDomain::class.java)
                    .process(it.value, EcomDomain.OFFERS_COUNT)
                    .applyTo(EcomDomain().withId(it.key))
            }
            .chunked(UPDATE_CHUNK_SIZE)
            .map {
                logger.info("Updating ${it.size} domain offers count")
                val updated = ecomDomainsRepository.update(it)
                logger.info("Updated $updated domains")
                updated
            }
            .sum()
    }

    fun removeEcomDomains(domainsToIds: Map<String, Long>): Int {
        return domainsToIds.values
            .chunked(UPDATE_CHUNK_SIZE)
            .map {
                logger.info("Removing ${it.size} domains")
                val removed = ecomDomainsRepository.deleteNotPermanent(it)
                logger.info("Removed ${it.size} domains")
                removed
            }
            .sum()
    }
}
