package ru.yandex.direct.jobs.uac

import NBannerLand.Banner
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerPropertiesImpl
import ru.yandex.direct.jobs.uac.converter.TBannerConverter
import ru.yandex.direct.jobs.uac.logbroker.EcomDomainsPreviewsLogbrokerReader
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.solomon.SolomonUtils
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.kikimr.persqueue.auth.Credentials
import ru.yandex.monlib.metrics.labels.Labels
import java.time.LocalDateTime
import java.util.function.Supplier

/**
 * Джоба читает офферы из топика ```bannerland/blrt/preview-offers-output```, конвертирует их в json
 * и добавляет в табличку ```ppcdict.ecom_domains``` для соответствующих доменов.
 */
@JugglerCheck(ttl = JugglerCheck.Duration(minutes = 30 + 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH, NotificationRecipient.LOGIN_BUHTER],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 15, needSchedule = TypicalEnvironment::class)
class ReceiveEcomDomainsPreviewsJob @Autowired constructor(
    private val directConfig: DirectConfig,
    private val tvmIntegration: TvmIntegration,
    private val ecomDomainsRepository: EcomDomainsRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(ReceiveEcomDomainsPreviewsJob::class.java)

        private const val UPDATE_CHUNK_SIZE = 250

        private fun createConsumerProperties(bannerLandOffersPreviewConfig: DirectConfig): LogbrokerConsumerProperties {
            // Определяем из каких партиций хотим читать
            val groups = (1..bannerLandOffersPreviewConfig.getInt("groups_count")).toList()

            return LogbrokerConsumerPropertiesImpl.Builder()
                .setHost(bannerLandOffersPreviewConfig.getString("host"))
                .setReadDataTimeoutSec(bannerLandOffersPreviewConfig.getLong("data_timeout"))
                .setInitTimeoutSec(bannerLandOffersPreviewConfig.getLong("init_timeout"))
                .setRetries(bannerLandOffersPreviewConfig.getInt("retries"))
                .setGroups(groups)
                .setReadTopic(bannerLandOffersPreviewConfig.getString("topic"))
                .setConsumerName(bannerLandOffersPreviewConfig.getString("client_id"))
                .build()
        }

        private fun createCredentialsSupplier(bannerLandOffersPreviewConfig: DirectConfig,
                                              tvmIntegration: TvmIntegration): Supplier<Credentials> {
            val logbrokerTvmServiceName = bannerLandOffersPreviewConfig.getString("tvm_service_name")
            val logbrokerTvmService = TvmService.fromStringStrict(logbrokerTvmServiceName)
            return Supplier {
                val serviceTicket = tvmIntegration.getTicket(logbrokerTvmService)
                Credentials.tvm(serviceTicket)
            }
        }

        private fun createSolomonRegistryLabels(label: String, topicName: String): Labels = Labels.of(
            "jobs", label,
            "topic", topicName)
    }

    private var logbrokerReader: LogbrokerBatchReader<Banner.TBanner>? = null
    private var solomonRegistryLabels: Labels? = null

    private val readsCountProperty = ppcPropertiesSupport.get(PpcPropertyNames.PREVIEW_TOPIC_READS_COUNT)

    data class EcomDomainWithNewOffers(
        val ecomDomain: EcomDomain,
        val newOffers: List<Banner.TBanner>
    )

    private fun init() {
        logger.info("Initializing ${ReceiveEcomDomainsPreviewsJob::class.simpleName}")

        val bannerLandOffersPreviewConfig: DirectConfig = directConfig.getBranch("preview-offers-output")

        val credentialsSupplier = createCredentialsSupplier(bannerLandOffersPreviewConfig, tvmIntegration)
        val logbrokerClientFactory = LogbrokerClientFactoryFacade(credentialsSupplier)

        val consumerProperties = createConsumerProperties(bannerLandOffersPreviewConfig)
        val syncConsumerSupplier = logbrokerClientFactory.createConsumerSupplier(consumerProperties)

        solomonRegistryLabels = createSolomonRegistryLabels("ecom_domains_previews_reader",
            consumerProperties.readTopic)
        val metricRegistry = SolomonUtils.SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels)

        logbrokerReader = RetryingLogbrokerBatchReader(
            { EcomDomainsPreviewsLogbrokerReader(syncConsumerSupplier, false, metricRegistry) },
            consumerProperties.retries)
    }

    override fun execute() {
        val readsCount = readsCountProperty.getOrDefault(1)

        init()
        try {
            repeat(readsCount) {
                logger.info("Start reading from topic")
                logbrokerReader?.fetchEvents(this::processEvents)
            }
        } catch (ex: InterruptedException) {
            Thread.currentThread().interrupt()
            throw InterruptedRuntimeException(ex)
        }
    }

    private fun processEvents(events: List<Banner.TBanner>) {
        logger.info("Fetched ${events.size} events from logbroker")

        val ecomDomainsWithNewOffers = getEcomDomainsWithNewOffers(events)

        if (ecomDomainsWithNewOffers.isEmpty()) {
            return
        }

        updatePreviewOffers(ecomDomainsWithNewOffers)
    }

    private fun getEcomDomainsWithNewOffers(offers: List<Banner.TBanner>): List<EcomDomainWithNewOffers> {
        val newOffersByDomains = offers.groupBy { getDomain(it.href) }

        return ecomDomainsRepository.getByDomains(newOffersByDomains.keys)
            .map { EcomDomainWithNewOffers(it, newOffersByDomains[it.domain]!!) }
    }

    private fun updatePreviewOffers(ecomDomainsWithNewOffers: List<EcomDomainWithNewOffers>) {
        val appliedChanges = ecomDomainsWithNewOffers.mapNotNull { (ecomDomain, newOffers) ->
            val mergedOffers = TBannerConverter.mergeWithNewOffers(ecomDomain.previewOffers, newOffers)
                ?: return@mapNotNull null

            val newOffersForPreview = JsonUtils.toJson(mergedOffers)

            ModelChanges(ecomDomain.id, EcomDomain::class.java)
                .process(newOffersForPreview, EcomDomain.PREVIEW_OFFERS)
                .process(LocalDateTime.now(), EcomDomain.LAST_MODIFIED_PREVIEW)
                .applyTo(ecomDomain)
        }

        appliedChanges.chunked(UPDATE_CHUNK_SIZE)
            .forEach { appliedChangesChunk ->
                logger.info("Updating ${appliedChangesChunk.size} domains")
                val chunkUpdated = ecomDomainsRepository.update(appliedChangesChunk)
                logger.info("Updated $chunkUpdated domains")
            }
    }

    private fun getDomain(url: String): String {
        val parts = UrlParts.fromUrl(url)
        return parts.domain
    }

    override fun finish() {
        if (solomonRegistryLabels != null) {
            SolomonUtils.SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels)
        }
        if (logbrokerReader != null) {
            try {
                logbrokerReader?.close()
            } catch (ex: LogbrokerReaderCloseException) {
                logger.error("Error while closing logbrokerReader", ex)
            }
        }
    }
}
