package ru.yandex.direct.jobs.uac

import NCrawl.Feeds
import com.google.protobuf.ByteString
import com.google.protobuf.Timestamp
import java.time.ZoneId
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.samovar.logbroker.ExternalFeedMessagesLogbrokerWriterFactory
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.market.yt.samovar.SamovarContextOuterClass

/**
 * Джоб запршаивает у Самовара обновление превью офферов.
 *
 * Для этого джоба ищет в табличке `ecom_domains` хосты с устаревшим превью, после чего отправляет их в LB
 *
 * @see [Внешние фиды в самоваре](https://wiki.yandex-team.ru/robot/samovar/feedsext/)
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 2, hours = 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_KOZOBRODOV],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = TypicalEnvironment::class)
class RequestUpdateOffersPreviewJob @Autowired constructor(
    private val ecomDomainsService: EcomDomainsService,
    private val externalFeedMessagesLogbrokerWriterFactory: ExternalFeedMessagesLogbrokerWriterFactory,
    private val ppcPropertiesSupport: PpcPropertiesSupport
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(RequestUpdateOffersPreviewJob::class.java)

        private const val DOMAINS_CHUNK_SIZE = 1000
        private const val DEFAULT_EXPIRATION_HOURS = 48L
        private const val DEFAULT_SENDING_SLEEP_COEF = 200.0
        private const val DEFAULT_SENDING_CHUNK_SIZE = 100
        private const val EXTERNAL_FEED_NAME = "check-host-for-smart-preview-ext"
        private const val SAMOVAR_CAMPAIGN_TYPE = "SITE_PREVIEW"
    }

    private var logbrokerWriter: LogbrokerWriter<Feeds.TFeedExt>? = null

    override fun execute() {
        val isJobEnabled = ppcPropertiesSupport.get(PpcPropertyNames.UPDATE_OFFERS_PREVIEW_JOB_ENABLED)
            .getOrDefault(false)
        if (!isJobEnabled) {
            logger.info("Job is disabled, exiting")
            return
        }

        val expirationHours = ppcPropertiesSupport.get(PpcPropertyNames.OFFERS_PREVIEW_EXPIRATION_HOURS)
            .getOrDefault(DEFAULT_EXPIRATION_HOURS)

        val outdatedEcomDomains = ecomDomainsService
            .getEcomDomainsWithOutdatedPreview(expirationHours)

        logger.info("Found ${outdatedEcomDomains.size} ecom domains with outdated preview")

        val envType = getEnvType()

        outdatedEcomDomains.chunked(DOMAINS_CHUNK_SIZE)
            .forEach { ecomDomains ->
                val protoModels = ecomDomains
                    .map { ecomDomainToProto(it, envType) }

                sendToLB(protoModels)
            }
    }

    private fun sendToLB(protoModels: List<Feeds.TFeedExt>) {
        val sleepCoef = ppcPropertiesSupport
            .get(PpcPropertyNames.UPDATE_OFFERS_PREVIEW_SLEEP_COEF)
            .getOrDefault(DEFAULT_SENDING_SLEEP_COEF)
        val sendingChunk = ppcPropertiesSupport
            .get(PpcPropertyNames.UPDATE_OFFERS_PREVIEW_SENDING_CHUNK_SIZE)
            .getOrDefault(DEFAULT_SENDING_CHUNK_SIZE)

        val relaxedWorker = RelaxedWorker(sleepCoef)
        protoModels.chunked(sendingChunk) {
            val hostsToSend = it.map { tFeedExt -> tFeedExt.url }
            logger.info("Sending ${hostsToSend.size} hosts to LB: $hostsToSend")
            relaxedWorker.runAndRelax<RuntimeException> { getLogbrokerWriter().write(it) }
        }
    }

    private fun getLogbrokerWriter(): LogbrokerWriter<Feeds.TFeedExt> {
        if (logbrokerWriter == null) {
            logbrokerWriter = externalFeedMessagesLogbrokerWriterFactory.writer
        }
        return logbrokerWriter!!
    }

    private fun ecomDomainToProto(ecomDomain: EcomDomain, envType: String): Feeds.TFeedExt {
        val url = "${ecomDomain.schema}://${ecomDomain.domain}/"
        val lastUpdateSeconds = ecomDomain.lastModifiedPreview.atZone(ZoneId.systemDefault()).toEpochSecond()

        val feedInfo = SamovarContextOuterClass.FeedInfo
            .newBuilder()
            .setUrl(url)
            .setDirectStandBy(true)
            .setUpdatedAt(Timestamp.newBuilder().setSeconds(lastUpdateSeconds).build())
            .setBusinessId(ecomDomain.marketBusinessId)
            .setShopId(ecomDomain.marketShopId)
            .setFeedId(ecomDomain.marketFeedId)
            .setFeedType(SamovarContextOuterClass.FeedInfo.FeedType.ASSORTMENT) // Фид с полной информацией о товарах
            .setCampaignType(SAMOVAR_CAMPAIGN_TYPE)
            .build()

        val context = SamovarContextOuterClass.SamovarContext
            .newBuilder()
            .addFeeds(feedInfo)
            .setEnvironment(envType)
            .build()
        val contextBytes = ByteString.copyFrom(context.toByteArray())

        return Feeds.TFeedExt
            .newBuilder()
            .setFeedName(EXTERNAL_FEED_NAME)
            .setUrl(url)
            .setFeedContext(
                Feeds.TFeedContext.newBuilder()
                    .setBytesValue(contextBytes)
                    .setUint64Value(10) // Количество офферов для превью по хосту
                    .build()
            )
            .build()
    }

    override fun finish() {
        if (logbrokerWriter != null) {
            logbrokerWriter!!.close()
        }
    }

    private fun getEnvType(): String {
        val envType = Environment.getCached()
        return when {
            envType.isProduction -> "production"
            envType.isTesting -> "testing"
            else -> "development"
        }
    }
}
