package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.core.entity.uac.repository.mysql.EcomDomainsRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob

/**
 * Джоба отправляет домены в MBI для получения превью оферов и проставляет маркетные id.
 * Выбираются только [SendEcomDomainToMBIJob.FETCH_LIMIT] записей из ```ppcdict.ecom_domains```
 * при одном запуске джобы (раз в 2 часа)
 */
@JugglerCheck(ttl = JugglerCheck.Duration(hours = 5),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_1, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 2, needSchedule = TypicalEnvironment::class)
class SendEcomDomainToMBIJob @Autowired constructor(private val ecomDomainsRepository: EcomDomainsRepository,
                                                    private val mbiService: MbiService,
                                                    private val ppcPropertiesSupport: PpcPropertiesSupport) : DirectJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(SendEcomDomainToMBIJob::class.java)

        private const val FETCH_LIMIT = 5_000
        private const val UPDATE_CHUNK_SIZE = 1000

        const val DEFAULT_MIN_OFFERS_COUNT = 150L
    }

    override fun execute() {
        val minOffersCountProperty: PpcProperty<Long> =
            ppcPropertiesSupport.get(PpcPropertyNames.MIN_OFFERS_COUNT_BY_HOST_FOR_SEND_TO_MBI)
        val minOffersCount = minOffersCountProperty.getOrDefault(DEFAULT_MIN_OFFERS_COUNT)

        val domains = ecomDomainsRepository.getWithoutMarketIds(minOffersCount, FETCH_LIMIT)

        if (domains.isEmpty()) {
            logger.info("Domains to send to MBI were not found. Skipping")
            return
        } else {
            logger.info("Fetched ${domains.size} ecom domains to send to MBI")
        }
        val relaxedWorker = RelaxedWorker()
        val appliedChanges: List<AppliedChanges<EcomDomain>> = domains.mapNotNull { domain ->
            try {
                val marketIds = relaxedWorker.callAndRelax<MbiService.AddSitePreviewResult, RuntimeException> {
                    mbiService.addSitePreview(domain)
                }
                ecomDomainToAppliedChanges(domain, marketIds)
            } catch (e: Exception) {
                logger.error("Got market client exception during sending domain ($domain) to MBI", e)
                null
            }
        }

        var updated = 0
        val chunksCount = (appliedChanges.size - 1) / UPDATE_CHUNK_SIZE + 1
        appliedChanges.chunked(UPDATE_CHUNK_SIZE)
            .forEachIndexed { index, appliedChangesChunk ->
                logger.info("Updating ${appliedChangesChunk.size} domains in DB (${index + 1} chunk of $chunksCount)")
                val chunkUpdated = ecomDomainsRepository.update(appliedChangesChunk)
                logger.info("Updated $chunkUpdated domains")

                updated += chunkUpdated
            }

        logger.info("Done. Total of $updated domains have been updated")
    }

    private fun ecomDomainToAppliedChanges(domain: EcomDomain,
                                           marketIds: MbiService.AddSitePreviewResult): AppliedChanges<EcomDomain> {
        return ModelChanges(domain.id, EcomDomain::class.java)
            .process(marketIds.businessId, EcomDomain.MARKET_BUSINESS_ID)
            .process(marketIds.marketFeedId, EcomDomain.MARKET_FEED_ID)
            .process(marketIds.shopId, EcomDomain.MARKET_SHOP_ID)
            .applyTo(domain)
    }
}
