package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.entity.uac.model.ShopInShopBusiness
import ru.yandex.direct.core.entity.uac.model.Source
import ru.yandex.direct.core.entity.uac.repository.mysql.ShopInShopBusinessesRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.jobs.uac.model.ShopInShopBusinessRow
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil.executeWithFallback
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtOperator
import ru.yandex.misc.io.ClassPathResourceInputStreamSource

/**
 * Джоба синхронизирующая бизнесы из маркетплейса маркета (`ppcdict.shop_in_shop_businesses` с `source = "market"`)
 * с выгрузкой бизнессов из YT
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 2, hours = 1),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DMITANOSH, NotificationRecipient.LOGIN_BUHTER],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = TypicalEnvironment::class)
class SyncMarketShopInShopBusinessesJob(
    private val ytProvider: YtProvider,
    private val shopInShopBusinessesRepository: ShopInShopBusinessesRepository,
    directConfig: DirectConfig
) : DirectJob() {

    companion object {
        private val LOGGER = LoggerFactory.getLogger(SyncMarketShopInShopBusinessesJob::class.java)

        private val FETCH_MARKET_SHOP_IN_SHOP_BUSINESSES_QUERY =
            ClassPathResourceInputStreamSource("uac/getMarketShopInShopBusinessesToSync.yql")
                .readLines()
                .joinToString("\n")

        private const val FETCH_SIZE = 500_000
        private const val CHUNK_SIZE = 1000
    }

    private val businessDirectTablePath: String
    private val businessMetrikaTablePath: String
    private val ytClusters: List<YtCluster>

    init {
        val config = directConfig.getBranch("shop_in_shop_businesses")
        businessDirectTablePath = config.getString("business_direct_table_path")
        businessMetrikaTablePath = config.getString("business_metrika_table_path")
        val clusterNames = config.getStringList("clusters")
        ytClusters = clusterNames
            .map { YtCluster.valueOf(it.uppercase()) }
    }

    override fun execute() {
        // Из yt берутся не более указанного числа записей для синхронизации
        val ytShopInShopBusinesses =
            executeWithFallback(ytClusters, ytProvider::getOperator) { ytOperator: YtOperator ->
                ytOperator.yqlQuery(
                    FETCH_MARKET_SHOP_IN_SHOP_BUSINESSES_QUERY, ShopInShopBusinessRow::fromResultSet,
                    businessDirectTablePath, businessMetrikaTablePath, FETCH_SIZE
                )
            }

        if (ytShopInShopBusinesses.size == FETCH_SIZE) {
            LOGGER.warn("Exceeded the count of rows in the table $businessDirectTablePath (max count: $FETCH_SIZE)")
            setJugglerStatus(
                JugglerStatus.WARN,
                "ppcdict.shop_in_shop_businesses not fully synced with yt-table, check log for details"
            )
        }

        LOGGER.info("Fetched ${ytShopInShopBusinesses.size} market businesses from YT")

        val dbBusinesses = shopInShopBusinessesRepository.getBusinessIdsBySource(Source.MARKET)

        LOGGER.info("Fetched ${dbBusinesses.size} market businesses from Direct")

        syncShopInShopBusinesses(ytShopInShopBusinesses, dbBusinesses)
    }

    private fun syncShopInShopBusinesses(
        ytShopInShopBusinesses: List<ShopInShopBusinessRow>,
        dbBusinesses: Collection<Long>
    ) {
        val convertedYtShopInShopBusinesses = ytShopInShopBusinesses
            .map { it.toShopInShopBusiness(Source.MARKET) }
        saveShopInShopBusinesses(convertedYtShopInShopBusinesses)

        val ytBusinessesId = ytShopInShopBusinesses
            .mapTo(HashSet()) { it.businessId }
        val businessesToDelete = dbBusinesses - ytBusinessesId
        deleteShopInShopBusinesses(businessesToDelete)
    }

    private fun saveShopInShopBusinesses(shopInShopBusinesses: Collection<ShopInShopBusiness>) {
        val inserted = shopInShopBusinesses
            .chunked(CHUNK_SIZE) {
                LOGGER.info("Saving ${it.size} market businesses")
                shopInShopBusinessesRepository.saveShopInShopBusinesses(it)
            }
            .sum()

        LOGGER.info("Total saved $inserted market businesses")
    }

    private fun deleteShopInShopBusinesses(businessesToDelete: Collection<Long>) {
        val deleted = businessesToDelete
            .chunked(CHUNK_SIZE) {
                LOGGER.info("Deleting ${it.size} market businesses")
                shopInShopBusinessesRepository.delete(Source.MARKET, it)
            }
            .sum()

        LOGGER.info("Total deleted $deleted market businesses")
    }
}
