package ru.yandex.direct.jobs.uac

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.CPM_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN
import ru.yandex.direct.common.db.PpcPropertyNames.ECOM_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN
import ru.yandex.direct.common.db.PpcPropertyNames.MOBILE_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN
import ru.yandex.direct.common.db.PpcPropertyNames.UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN
import ru.yandex.direct.common.db.PpcPropertyNames.UAC_FAILED_JOBS_MONITORING_ENABLED
import ru.yandex.direct.common.db.PpcPropertyNames.UAC_FAILED_JOBS_MONITORING_LAST_TIME
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toAdvType
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.uac.MonitoringCampaignType.CPM
import ru.yandex.direct.jobs.uac.MonitoringCampaignType.ECOM
import ru.yandex.direct.jobs.uac.MonitoringCampaignType.MOBILE
import ru.yandex.direct.jobs.uac.MonitoringCampaignType.TEXT
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.ALREADY_SYNCED_CAMPAIGN_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.DRAFT_CAMPAIGN_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.NO_CLIENT_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.NO_TYPED_CAMPAIGN_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.NO_YDB_ACCOUNT_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.NO_YDB_CAMPAIGN_ERROR
import ru.yandex.direct.jobs.uac.UpdateAdsJob.Companion.NO_YDB_DIRECT_CAMPAIGN_ERROR
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.IssueCreate
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit

/**
 * Джоба, которая мониторит падения джобы [UpdateAdsJob]
 * и создает тикет со списком упавших кампаний,
 * которые в данный момент не синхронизированы с заявкой в груте
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 4),
    tags = [CheckTag.DIRECT_PRIORITY_1_NOT_READY],
    notifications = [
        OnChangeNotification(
            recipient = [NotificationRecipient.CHAT_UC_API_MONITORING],
            method = [NotificationMethod.TELEGRAM],
        ),
    ],
    needCheck = ProductionOnly::class,
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = NonDevelopmentEnvironment::class)
class UacUpdateAdsFailedJobsMonitoringJob(
    private val shardHelper: ShardHelper,
    private val dbQueueRepository: DbQueueRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val startrekSession: Session,
): DirectJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
        private val EXPECTED_ERRORS = setOf(
            NO_YDB_CAMPAIGN_ERROR, DRAFT_CAMPAIGN_ERROR, ALREADY_SYNCED_CAMPAIGN_ERROR,
            NO_YDB_ACCOUNT_ERROR, NO_CLIENT_ERROR, NO_YDB_DIRECT_CAMPAIGN_ERROR,
            NO_TYPED_CAMPAIGN_ERROR,
        )

        private const val MSMB_BOARD_ID = 15519
        private const val UAC_BOARD_ID = 17823
        private const val CPM_BOARD_ID = 16809
        private const val ECOM_BOARD_ID = 15734

        private const val WHAT_TO_DO_DESCRIPTION = "Требуется посмотреть на пострадавшие кампании и\n" +
            "- **переотправить на генерацию** те кампании, которые упали с временной ошибкой " +
            "(например, из-за недоступности грута) и которые на данный момент не синхронизированы с заявкой.\n" +

            "Проверить синхронизированность с заявкой можно через консольную утилиту **grut-orm**:\n" +
            "Дырки есть с ppcdev. Сначала утилиту нужно собрать (из корня аркадии):\n" +
            "%%ya make --checkout grut/tools/cli/orm/%%. Затем можно делать запрос:\n" +
            "%%./grut/tools/cli/orm/grut-orm get campaign --selector=/spec/campaign_brief/brief_synced --address=stable <cid кампании>%%\n" +
            "Значение `false` означает, что кампания не синхронизирована с заявкой и можно отправлять на генерацию. " +
            "После генерации важно проверить, что значение `brief_synced` стало `true`\n" +

            "Отправить кампании на генерацию можно через " +
            "((https://direct.yandex.ru/internal_tools/#update_ads_deferred_tool отчет)) под супером " +
            "или через ((https://a.yandex-team.ru/arc_vcs/direct/oneshot/src/main/java/ru/yandex/direct" +
            "/oneshot/oneshots/uc/UacUpdateAdsOneshot.kt?rev=r8798285#L52 ваншот));\n" +

            "- **завести баги** на постоянные ошибки " +
            "(после починки бага не забыть переотправить кампании на генерацию).\n\n" +

            "Просмотреть логи упавшей обработки конкретной кампании можно " +
            "((https://direct.yandex.ru/logviewer/short/v2/3680429287536322776 например так)), подставив ее id."
    }

    override fun execute() {
        val enabled = ppcPropertiesSupport[UAC_FAILED_JOBS_MONITORING_ENABLED].getOrDefault(false)
        if (!enabled) {
            logger.info("Skip processing. Job is not enabled")
            return
        }

        val now = LocalDateTime.now()
        val lastTimeProperty = ppcPropertiesSupport[UAC_FAILED_JOBS_MONITORING_LAST_TIME]
        val fromTime = lastTimeProperty.getOrDefault(now.minusWeeks(1))
        val toTime = now.minusHours(1)

        logger.info("Searching failed uac jobs from $fromTime to $toTime")
        val failedCampaignIdsToError = getFailedCampaignIdsToError(fromTime, toTime)
        if (failedCampaignIdsToError.isEmpty()) {
            logger.info("No failed campaigns found")
            lastTimeProperty.set(toTime)
            return
        }

        val failedCampaignIds = failedCampaignIdsToError.keys
        val notSyncedCampaigns = grutUacCampaignService.getCampaigns(failedCampaignIds)
            .filterNot { it.spec.campaignBrief.briefSynced }
        val ecomCampaignIds = notSyncedCampaigns
            .filter { it.spec.campaignBrief.hasEcom() }
            .map { it.meta.id }
            .toSet()
        val campaignIdsByType = notSyncedCampaigns
            .filterNot { ecomCampaignIds.contains(it.meta.id) }
            .filter { it.meta.campaignType.toAdvType() != null }
            .groupBy({ it.meta.campaignType.toAdvType()!! }, { it.meta.id })
            .mapValues { it.value.toSet() }
        if (ecomCampaignIds.isEmpty() && campaignIdsByType.isEmpty()) {
            logger.info("No failed and not synced campaigns found")
            lastTimeProperty.set(toTime)
            return
        }

        logger.info(getLogMessage(ecomCampaignIds, campaignIdsByType))
        val summary = getSummary(fromTime, toTime)
        val campaignIdsToError = failedCampaignIdsToError.mapKeys { it.key.toIdLong() }

        if (ecomCampaignIds.isNotEmpty()) {
            createTicket(summary, ECOM, ecomCampaignIds, campaignIdsToError)
        }
        AdvType.values().forEach {
            if (campaignIdsByType[it]?.isNotEmpty() == true) {
                val campaignIds = campaignIdsByType[it]!!
                createTicket(summary, toMonitoringCampaignType(it), campaignIds, campaignIdsToError)
            }
        }

        lastTimeProperty.set(toTime)
    }

    private fun getFailedCampaignIdsToError(fromTime: LocalDateTime, toTime: LocalDateTime): Map<String, String?> {
        val failedCampaignIdsToError = mutableMapOf<String, String?>()
        shardHelper.forEachShard { shard ->
            val failedJobs = dbQueueRepository.findFailedArchivedJobs(
                shard, DbQueueJobTypes.UAC_UPDATE_ADS, fromTime, toTime
            )
            failedCampaignIdsToError.putAll(
                failedJobs.filter { job ->
                    EXPECTED_ERRORS.none { error ->
                        job.result.error?.contains(error) ?: false
                    }
                }.associateBy({ it.args.uacCampaignId }, { it.result.error })
            )
        }
        return failedCampaignIdsToError
    }

    private fun getLogMessage(ecomCampaignIds: Set<Long>, campaignIdsByType: Map<AdvType, Set<Long>>): String {
        val sb = StringBuilder("Failed and not synced campaigns: ")
        if (ecomCampaignIds.isNotEmpty()) {
            sb.appendLine().append("Ecom: $ecomCampaignIds")
        }
        if (campaignIdsByType[AdvType.TEXT]?.isNotEmpty() == true) {
            sb.appendLine().append("Text: ${campaignIdsByType[AdvType.TEXT]}")
        }
        if (campaignIdsByType[AdvType.MOBILE_CONTENT]?.isNotEmpty() == true) {
            sb.appendLine().append("Mobile: ${campaignIdsByType[AdvType.MOBILE_CONTENT]}")
        }
        if (campaignIdsByType[AdvType.CPM_BANNER]?.isNotEmpty() == true) {
            sb.appendLine().append("Cpm: ${campaignIdsByType[AdvType.CPM_BANNER]}")
        }
        return sb.toString()
    }

    private fun getSummary(fromTime: LocalDateTime, toTime: LocalDateTime) =
        "Uac broken campaigns ${fromTime.truncatedTo(ChronoUnit.SECONDS)}-${toTime.truncatedTo(ChronoUnit.SECONDS)}"

    private fun createTicket(
        summary: String,
        campaignType: MonitoringCampaignType,
        campaignIds: Set<Long>,
        campaignIdsToError: Map<Long, String?>,
    ) {
        val (assigneePpcProperty, boards) = when (campaignType) {
            TEXT -> Pair(UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN, listOf(MSMB_BOARD_ID))
            MOBILE -> Pair(MOBILE_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN, listOf(MSMB_BOARD_ID, UAC_BOARD_ID))
            CPM -> Pair(CPM_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN, listOf(CPM_BOARD_ID))
            ECOM -> Pair(ECOM_UAC_FAILED_JOBS_MONITORING_ASSIGNEE_LOGIN, listOf(ECOM_BOARD_ID))
        }
        val assignee = ppcPropertiesSupport[assigneePpcProperty].get()
        val description = getDescription(campaignType, campaignIds, campaignIdsToError)
        val audienceScale = getAudienceScale(campaignIds.size)
        val ticketKey = createTicket(summary, description, assignee, audienceScale, boards)
        logger.info("${campaignType.displayName} ticket created: $ticketKey")
    }

    private fun getDescription(
        campaignType: MonitoringCampaignType,
        campaignIds: Set<Long>,
        campaignIdsToError: Map<Long, String?>
    ): String {
        val sb = StringBuilder("Список упавших и несинхронизированных с заявкой кампаний: ")
        sb.appendLine().append("**${campaignType.displayName}: $campaignIds**")
        sb.appendLine().append("<{Ошибки:")
        campaignIds.forEach {
            sb.appendLine()
                .append("<{$it").appendLine()
                .append("%%")
                .append(campaignIdsToError[it] ?: "Нет описания ошибки")
                .append("%%")
                .appendLine().append("}>")
        }
        sb.appendLine().append("}>")
        sb.appendLine().append(WHAT_TO_DO_DESCRIPTION)
        return sb.toString()
    }

    private fun createTicket(
        summary: String,
        description: String,
        assignee: String?,
        audienceScale: AudienceScale,
        boards: List<Int>,
    ): String {
        val user = if (assignee != null) {
            val userO = startrekSession.users().getO(assignee)
            if (userO.isPresent) userO.get() else null
        } else {
            null
        }
        val issueCreate = IssueCreate.builder()
            .queue("DIRECT")
            .type("bug")
            .summary(summary)
            .description(description)
            .tags("uac_broken_campaigns")
            .assignee(user)
            .followers(NotificationRecipient.LOGIN_KHUZINAZAT.getName())
            .set("bugDetectionMethod", "Monitoring")
            .set("stage", "Production")
            .set("bugSource", listOf("Команда"))
            .set("typeOfBug", "Functionality blocked")
            .set("audienceScale", audienceScale.value)
            .set("boards", boards)
            .build()
        return startrekSession.issues().create(issueCreate).key
    }

    private fun getAudienceScale(failedCampaignsCount: Int) =
        if (failedCampaignsCount < 3) {
            AudienceScale.SMALL
        } else if (failedCampaignsCount < 10) {
            AudienceScale.NORMAL
        } else {
            AudienceScale.BIG
        }

    private fun toMonitoringCampaignType(advType: AdvType) =
        when (advType) {
            AdvType.TEXT -> TEXT
            AdvType.MOBILE_CONTENT -> MOBILE
            AdvType.CPM_BANNER -> CPM
        }
}

enum class AudienceScale(val value: String) {
    SMALL("Small"),
    NORMAL("Normal"),
    BIG("Big"),
}

enum class MonitoringCampaignType(val displayName: String) {
    TEXT("Text"),
    MOBILE("Mobile"),
    CPM("Cpm"),
    ECOM("E-com"),
}
