package ru.yandex.direct.jobs.uac

import java.time.Duration
import java.time.Instant
import java.time.LocalDateTime.now
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.CpmBannerCampaign
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.service.BaseCampaignService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.model.UpdateAdsJobParams
import ru.yandex.direct.core.entity.uac.model.UpdateAdsJobResult
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAccountRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.CampaignContentUpdateService
import ru.yandex.direct.core.entity.uac.service.GrutCampaignContentUpdateService
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService
import ru.yandex.direct.core.entity.uac.service.UacCampaignServiceHolder
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException
import ru.yandex.direct.dbqueue.model.DbQueueJob
import ru.yandex.direct.dbqueue.service.DbQueueService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.feature.FeatureName.UAC_MULTIPLE_AD_GROUPS_ENABLED
import ru.yandex.direct.jobs.uac.UpdateAdsJobUtil.migratingOperatorUid
import ru.yandex.direct.jobs.uac.model.createUpdateAdsContainers
import ru.yandex.direct.jobs.uac.service.BannerCleanJobService
import ru.yandex.direct.jobs.uac.service.UacJobServiceHolder
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.primitives.GaugeInt64
import ru.yandex.monlib.metrics.primitives.Rate

/**
 * Джоба для асинхронного создания баннеров, групп и условий показа для UC (ТГО) и UAC (РМП).
 * В случае проблем писать [khuzinazat](https://staff.yandex-team.ru/khuzinazat)
 * или [bratgrim](https://staff.yandex-team.ru/bratgrim).
 *
 * Основная цель - синхронизация данных по баннерам из YDB в MySQL, для переданной кампании.
 * Аналог python таски update_direct_ads в tasks.py
 * [https://a.yandex-team.ru/arc/trunk/arcadia/yabs/rmp/backend/src/uac/campaign/tasks.py?rev=r8181254#L15].
 *
 * 1) Джоба ходит в YDB базу UAC для получения необходимых данных (кампания, ее контенты)
 * 2) Подчищает MySQL-баннера для удаленных YDB-контентов (останавливает, а затем архивирует/удаляет)
 * 3) Создает новые или обновляет существующие баннера в MySQL по данным из YDB
 * 4) Обновляет у контентов в YDB статусы, причины отклонения и флаги кампании
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 20), needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_0],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.CHAT_UC_API_MONITORING],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 3, needSchedule = NonDevelopmentEnvironment::class)
class UpdateAdsJob @Autowired constructor(
    private val dbQueueService: DbQueueService,
    private val bannerCleanJobService: BannerCleanJobService,
    private val campaignContentUpdateService: CampaignContentUpdateService,
    private val grutCampaignContentUpdateService: GrutCampaignContentUpdateService,
    private val clientService: ClientService,
    private val baseCampaignService: BaseCampaignService,
    private val uacYdbAccountRepository: UacYdbAccountRepository,
    private val serviceHolder: UacJobServiceHolder,
    private val uacDbDefineService: UacDbDefineService,
    private val uacCampaignServiceHolder: UacCampaignServiceHolder,
    private val grutApiService: GrutApiService,
    private val featureService: FeatureService,
    private val grutUacContentService: GrutUacContentService,
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(UpdateAdsJob::class.java)
        private const val JOB_RETRIES = 3
        private val ITERATION_TIME = Duration.ofMinutes(5)
        const val NO_YDB_CAMPAIGN_ERROR = "Can't find ydb:campaign by campaign id"
        const val DRAFT_CAMPAIGN_ERROR = "Skip draft campaign"
        const val ALREADY_SYNCED_CAMPAIGN_ERROR = "Campaign already synced"
        const val NO_YDB_ACCOUNT_ERROR = "Can't find ydb:account by account id"
        const val NO_CLIENT_ERROR = "Client not found by id"
        const val NO_YDB_DIRECT_CAMPAIGN_ERROR = "Can't find ydb:direct_campaign by id"
        const val NO_TYPED_CAMPAIGN_ERROR = "Can't find mobile_content or text campaign  or cpm_banner campaign"
        const val NO_CAMPAIGN_STATUSES = "Can't find campaign statuses"
    }

    private val supportedTypes = setOf(AdvType.CPM_BANNER, AdvType.MOBILE_CONTENT, AdvType.TEXT)

    override fun execute() {
        val borderTime = now().plus(ITERATION_TIME)
        val metrics = initMetrics(shard)
        do {
            val grabbed = dbQueueService.grabAndProcessJob(
                shard,
                DbQueueJobTypes.UAC_UPDATE_ADS,
                { processGrabbedJobWrapped(it, metrics) },
                JOB_RETRIES,
                { _, stacktrace -> onError(stacktrace) }
            )
            if (!grabbed) {
                logger.info("No free jobs")
                metrics.updateAdsJobDelay.set(0)
                metrics.processedTime.set(0)
            }
        } while (grabbed && now() < borderTime)
    }

    private fun processGrabbedJobWrapped(
        jobInfo: DbQueueJob<UpdateAdsJobParams, UpdateAdsJobResult>,
        metrics: Metrics
    ): UpdateAdsJobResult {

        val start = System.nanoTime()
        try {
            return processGrabbedJob(jobInfo.args.uacCampaignId, jobInfo.args.operatorId)
        } catch (e: Exception) {
            if (e is InterruptedException) {
                Thread.currentThread().interrupt()
            }
            metrics.failedCampaignsCount.inc()
            throw JobFailedWithTryLaterException(Duration.ofMinutes(20), e.message, e)
        } finally {
            val delay = Instant.now().epochSecond - jobInfo.creationTime.atZone(DateTimeUtils.MSK).toEpochSecond()
            metrics.updateAdsJobDelay.set(delay)
            metrics.totalCampaignsCount.inc()
            val finish = System.nanoTime()
            metrics.processedTime.set((finish - start) / 1_000_000_000)
        }
    }

    fun processGrabbedJob(uacCampaignId: String, operatorId: Long): UpdateAdsJobResult {
        val useGrut = uacDbDefineService.useGrut(uacCampaignId)
        val uacCampaignService = uacCampaignServiceHolder.getUacCampaignService(useGrut)
        val uacContentsService = uacCampaignServiceHolder.getUacContentService(useGrut)

        val uacCampaign = uacCampaignService.getCampaignById(uacCampaignId)
            ?: throw IllegalStateException("$NO_YDB_CAMPAIGN_ERROR: $uacCampaignId")

        if (!supportedTypes.contains(uacCampaign.advType)) {
            logger.error("Skip due to not supported campaign. CampaignId = ${uacCampaign.id}")
            return UpdateAdsJobResult(null)
        }
        if (uacCampaign.briefSynced == true) {
            throw IllegalStateException("$ALREADY_SYNCED_CAMPAIGN_ERROR. CampaignId = ${uacCampaign.id}")
        }

        val clientId = if (useGrut) {
            ClientId.fromLong(uacCampaign.account.toLong())
        } else {
            val uacAccount = uacYdbAccountRepository.getAccountById(uacCampaign.account)
                ?: throw IllegalStateException("$NO_YDB_ACCOUNT_ERROR: ${uacCampaign.account}")
            ClientId.fromLong(uacAccount.directClientId)
        }

        val needToMigrate = operatorId == migratingOperatorUid
            && featureService.isEnabledForClientId(clientId, FeatureName.MIGRATING_ECOM_CAMPAIGNS_TO_NEW_BACKEND)
        val adGroupsBriefEnabled = featureService.isEnabledForClientId(clientId, UAC_MULTIPLE_AD_GROUPS_ENABLED)

        val client = clientService.getClient(clientId)
            ?: throw IllegalStateException("$NO_CLIENT_ERROR: $clientId")

        val directCampaignId = uacCampaignService.getDirectCampaignIdById(uacCampaignId)
            ?: throw IllegalStateException("$NO_YDB_DIRECT_CAMPAIGN_ERROR: $uacCampaignId")

        if (uacCampaign.isDraft) {
            throw IllegalStateException("$DRAFT_CAMPAIGN_ERROR. CampaignId = ${uacCampaign.id}")
        }

        logger.info(
            "Processing uac campaign: " +
                "uac campaign id: $uacCampaignId, direct cid: $directCampaignId"
        )

        val campaigns = baseCampaignService.get(clientId, operatorId, listOf(directCampaignId))

        if (campaigns.isEmpty()
            || (campaigns[0] !is MobileContentCampaign
                && campaigns[0] !is TextCampaign
                && campaigns[0] !is CpmBannerCampaign)
        ) {
            throw IllegalStateException("$NO_TYPED_CAMPAIGN_ERROR by id: $directCampaignId")
        }

        val campaignStatuses = uacCampaignService.getCampaignStatuses(clientId, directCampaignId, uacCampaign)
            ?: throw IllegalStateException("$NO_CAMPAIGN_STATUSES: $uacCampaignId")

        // Получение групповых заявок ограничено лимитом [AdGroupBriefGrutApi.DEFAULT_FETCH_LIMIT]
        val uacAdGroupBrief =
            if (adGroupsBriefEnabled) grutApiService.adGroupBriefGrutApi
                .selectAdGroupBriefsByCampaignId(directCampaignId)
            else null

        // Если это заявка на кампанию (без групповых заявок)
        val isItCampaignBrief = uacAdGroupBrief.isNullOrEmpty()

        val uacDirectAdGroups = serviceHolder.getUacAdGroupJobService(useGrut)
            .getAdGroupsByUacCampaignId(uacCampaignId, directCampaignId)

        val uacAssetsByGroupBriefId: Map<Long?, List<UacYdbCampaignContent>> =
            if (isItCampaignBrief) mapOf(null to uacContentsService.getCampaignContents(uacCampaign))
            else grutUacContentService.getAssetsByAdGroups(uacAdGroupBrief)
                .mapKeys { it.key }

        val updateContainers = createUpdateAdsContainers(
            operatorId,
            client,
            uacCampaign,
            uacAdGroupBrief,
            campaigns[0] as CommonCampaign,
        )

        val isGrutStatusArchived = campaignStatuses.status == Status.ARCHIVED
        val isCampaignArchived = (campaigns[0] as CommonCampaign).statusArchived
        if (!isGrutStatusArchived || !isCampaignArchived) {

            // Чистим удаленные
            if (!isGrutStatusArchived) {
                bannerCleanJobService.clean(
                    clientId,
                    uacCampaign,
                    uacAssetsByGroupBriefId,
                    updateContainers,
                    isItCampaignBrief,
                    needToMigrate
                )
            }

            // Создаем новые баннеры
            serviceHolder.getBannerCreateJobService(useGrut).createNewAdsAndUpdateExist(
                client,
                updateContainers,
                uacCampaign,
                uacDirectAdGroups,
                uacAssetsByGroupBriefId,
                isItCampaignBrief,
            )
        }
        if (isGrutStatusArchived || isCampaignArchived) {
            logger.info(
                "Campaign ${uacCampaign.id} archived in GRuT '$isGrutStatusArchived' " +
                    "and in mysql `$isCampaignArchived`"
            )
        }

        // Обновляем статусы и причины отклонения у campaign_contents и флаги у кампании
        if (useGrut) {
            grutCampaignContentUpdateService.updateCampaignContentsAndCampaignFlags(shard, listOf(directCampaignId))
        } else {
            campaignContentUpdateService.updateCampaignContentsAndCampaignFlags(
                uacCampaignId, clientId, uacDirectAdGroups, uacAssetsByGroupBriefId.values.flatten()
            )
        }
        serviceHolder.getBannerCreateJobService(useGrut).updateBriefSynced(uacCampaignId, true)

        return UpdateAdsJobResult(null)
    }

    private fun initMetrics(shard: Int): Metrics {
        val labels = Labels.of("shard", shard.toString())
        val updateAdsJobDelay = SOLOMON_REGISTRY.gaugeInt64("update_ads_job_delay", labels)
        val processedTime = SOLOMON_REGISTRY.gaugeInt64("update_ads_job_processed_time", labels)
        val totalCampaignsCount = SOLOMON_REGISTRY.rate("update_ads_job_total_camps", labels)
        val failedCampaignsCount = SOLOMON_REGISTRY.rate("update_ads_job_failed_camps", labels)

        return Metrics(updateAdsJobDelay, processedTime, totalCampaignsCount, failedCampaignsCount)
    }

    private fun onError(stacktrace: String): UpdateAdsJobResult {
        return UpdateAdsJobResult(stacktrace)
    }

    data class Metrics(
        val updateAdsJobDelay: GaugeInt64,
        val processedTime: GaugeInt64,
        val totalCampaignsCount: Rate,
        val failedCampaignsCount: Rate,
    )
}
