package ru.yandex.direct.jobs.uac

import java.time.Duration
import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.UpdateVideoContentJobParams
import ru.yandex.direct.core.entity.uac.model.UpdateVideoContentJobResult
import ru.yandex.direct.core.entity.uac.service.UacDbDefineService
import ru.yandex.direct.dbqueue.JobFailedWithTryLaterException
import ru.yandex.direct.dbqueue.model.DbQueueJob
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbqueue.service.DbQueueService
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.uac.service.UacJobServiceHolder
import ru.yandex.direct.jobs.uac.service.UacVideoContentJobService
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob

@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 10), needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_TELEPORT],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 3, needSchedule = NonDevelopmentEnvironment::class)
class UpdateVideoContentJob @Autowired constructor(
    private val dbQueueService: DbQueueService,
    private val dbQueueRepository: DbQueueRepository,
    private val uacDbDefineService: UacDbDefineService,
    private val uacJobServiceHolder: UacJobServiceHolder,
    private val uacVideoContentJobService: UacVideoContentJobService,
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(UpdateVideoContentJob::class.java)
        private const val JOB_RETRIES = 3
        private val ITERATION_TIME = Duration.ofMinutes(5)
        private val PERIOD_BETWEEN_TRIES = Duration.ofSeconds(10)
    }

    override fun execute() {
        val borderTime = LocalDateTime.now().plus(ITERATION_TIME)

        do {
            val grabbed = dbQueueService.grabAndProcessJob(
                shard,
                DbQueueJobTypes.UAC_UPDATE_VIDEO_CONTENT,
                { processGrabbedJobWrapped(it) },
                JOB_RETRIES,
                { _, stacktrace -> onError(stacktrace) }
            )
            if (!grabbed) logger.info("No free jobs")
        } while (grabbed && LocalDateTime.now() < borderTime)
    }

    private fun processGrabbedJobWrapped(
        jobInfo: DbQueueJob<UpdateVideoContentJobParams, UpdateVideoContentJobResult>
    ): UpdateVideoContentJobResult {
        try {
            return processGrabbedJob(jobInfo)
        } catch (e: Exception) {
            if (e is InterruptedException) {
                Thread.currentThread().interrupt()
            }
            throw JobFailedWithTryLaterException(Duration.ofMinutes(20), e.message, e)
        }
    }

    private fun processGrabbedJob(
        jobInfo: DbQueueJob<UpdateVideoContentJobParams, UpdateVideoContentJobResult>
    ): UpdateVideoContentJobResult {
        val ydbContentId = jobInfo.args.uacYdbContentId

        val useGrut = uacDbDefineService.useGrutForContentId(ydbContentId)
        val contentJobService = uacJobServiceHolder.getUacContentJobService(useGrut)
        val ydbContent = contentJobService.getContent(ydbContentId)
            ?: return UpdateVideoContentJobResult("Can't find content by id: $ydbContentId")
        if (ydbContent.type != MediaType.VIDEO) {
            return UpdateVideoContentJobResult("Content is not video: $ydbContentId")
        }
        val status: String? = ydbContent.meta["status"] as? String
        if (status == "ready") {
            return UpdateVideoContentJobResult("Video content is already ready: $ydbContentId")
        }

        val updatedMeta = uacVideoContentJobService.createCreativeForContent(jobInfo.clientId.asLong(), ydbContent)
        if (updatedMeta == null) {
            dbQueueRepository.insertDelayedJob(
                shard,
                DbQueueJobTypes.UAC_UPDATE_VIDEO_CONTENT,
                jobInfo.clientId,
                jobInfo.uid,
                jobInfo.args,
                PERIOD_BETWEEN_TRIES
            )
            return UpdateVideoContentJobResult("Video is not yet ready")
        }
        contentJobService.updateVideoMeta(jobInfo.clientId.toString(), ydbContentId, updatedMeta)

        return UpdateVideoContentJobResult(null)

    }

    private fun onError(stacktrace: String): UpdateVideoContentJobResult {
        return UpdateVideoContentJobResult(stacktrace)
    }
}
