package ru.yandex.direct.jobs.uac.converter

import NBannerLand.Banner
import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.core.entity.feed.model.FeedOfferExamples
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.grut.experimental.banner.BannerResources

object TBannerConverter {

    private val logger = LoggerFactory.getLogger(TBannerConverter::class.java)

    fun mergeWithNewOffers(existingOffers: String?, newOffers: List<Banner.TBanner>): FeedOfferExamples? {
        val convertedNewOffers = try {
            convertOffers(newOffers)
        } catch (ex: Exception) {
            logger.warn("Got exception while processing new offers", ex)
            return null
        }
        if (convertedNewOffers.isEmpty()) {
            return null
        }
        if (convertedNewOffers.size >= MbiService.OFFERS_COUNT_FOR_PREVIEW) {
            return FeedOfferExamples(convertedNewOffers)
        }

        val parsedOfferExamples = if (existingOffers != null && existingOffers.isNotBlank()) {
            JsonUtils.fromJson(existingOffers, FeedOfferExamples::class.java)
        } else {
            return FeedOfferExamples(convertedNewOffers)
        }
        val parsedExistingOffers = parsedOfferExamples.offers ?: return FeedOfferExamples(convertedNewOffers)

        val resultOffers = HashMap<String, FeedOfferExamples.Offer>()
        resultOffers.putAll(convertedNewOffers)

        parsedExistingOffers.asSequence()
            .filter { !convertedNewOffers.containsKey(it.key) }
            // Остаются более свежие оффера
            .sortedByDescending { it.value.updateDate ?: LocalDateTime.MIN }
            .take(MbiService.OFFERS_COUNT_FOR_PREVIEW - convertedNewOffers.size)
            .forEach {
                resultOffers[it.key] = it.value
            }

        return FeedOfferExamples(resultOffers)
    }

    fun convertOffers(offers: List<Banner.TBanner>): Map<String, FeedOfferExamples.Offer> {
        return offers
            .mapNotNull(this::convertOffer)
            .take(MbiService.OFFERS_COUNT_FOR_PREVIEW)
            .associate { it }
    }

    fun convertOffer(offer: Banner.TBanner): Pair<String, FeedOfferExamples.Offer>? {
        val shopId = offer.shopId

        if (!offer.hasBannerPrice() || !offer.hasOfferInfo() || !offer.hasHref() || !offer.hasOfferYabsId()) {
            logger.warn("Offer with shop_id = $shopId doesn't contain required fields")
            return null
        }

        val offerPrice = offer.bannerPrice
        val offerInfo = offer.offerInfo

        val text = convertText(offer, offerPrice, offerInfo)
        if (text == null) {
            logger.warn("Offer with shop_id = $shopId contains invalid currency")
            return null
        }

        val price = convertPrice(offerPrice)
        if (price == null) {
            logger.warn("Offer with shop_id = $shopId contains invalid price")
            return null
        }

        val imageInfo = extractImageInfo(offer)
        if (imageInfo == null) {
            logger.warn("Offer with shop_id = $shopId doesn't contain images")
            return null
        }

        return offer.offerYabsId.toString() to FeedOfferExamples.Offer(
            offer.href,
            price,
            text,
            convertClickUrls(offer),
            imageInfo,
            LocalDateTime.now()
        )
    }

    private fun convertPrice(offerPrice: BannerResources.TBannerPrice): FeedOfferExamples.Price? {
        val price = if (offerPrice.hasPrice() && offerPrice.price.toDoubleOrNull() != null) offerPrice.price else return null
        val oldPrice = if (offerPrice.hasOldPrice() && offerPrice.oldPrice.toDoubleOrNull() != null) offerPrice.oldPrice else null
        return FeedOfferExamples.Price(price, oldPrice)
    }

    private fun convertText(offer: Banner.TBanner,
                            offerPrice: BannerResources.TBannerPrice,
                            offerInfo: Banner.TBanner.TOfferInfo): FeedOfferExamples.Text? {
        val description = if (offerInfo.hasDescriptionForDirect() && offerInfo.descriptionForDirect != "null")
            offerInfo.descriptionForDirect else null

        val currency = when {
            offerPrice.currency == "RUR" -> "RUB"
            CurrencyCode.isRealCurrencyCode(offerPrice.currency) -> offerPrice.currency
            else -> return null
        }

        return FeedOfferExamples.Text(description, null, offer.title, currency)
    }

    private fun convertClickUrls(offer: Banner.TBanner): FeedOfferExamples.ClickUrls {
        return FeedOfferExamples.ClickUrls(offer.href, null, null, null, null, null, null, null)
    }

    private fun extractImageInfo(offer: Banner.TBanner): FeedOfferExamples.ImageInfo? {
        val imagesByFormat = HashMap<String, FeedOfferExamples.Image>()

        offer.imagesInfoList?.forEach {
            if (it.hasAvatarsImage()) {
                val imageUrl = makeImageUrl(it.avatarsImage)

                it.imagesList?.forEach { image ->
                    imagesByFormat[image.format] = FeedOfferExamples.Image(image.width.toInt(), image.height.toInt(),
                        "${imageUrl}/${image.format}", null)
                }
            }
        }

        if (imagesByFormat.isEmpty()) {
            return null
        }

        val small = imagesByFormat["small"] ?: return null
        val orig = imagesByFormat["orig"] ?: return null
        val big = imagesByFormat["big"] ?: return null
        val huge = imagesByFormat["huge"] ?: return null

        return FeedOfferExamples.ImageInfo(small, orig, big, huge)
    }

    private fun makeImageUrl(avatarsImage: BannerResources.TImagesInfo.TAvatarsImage): String {
        return "//avatars.mds.yandex.net/get-${avatarsImage.namespace}/${avatarsImage.imageGroupId}/" +
            avatarsImage.imageName
    }
}
