package ru.yandex.direct.jobs.uac.repository

import org.jooq.impl.DSL
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Repository
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.SUBCAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsMetatype
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.time.LocalDateTime

@Repository
class EcomCampaignsMonitoringRepository @Autowired constructor(
    private val dslContextProvider: DslContextProvider
) {

    fun countEcomCampaigns(shard: Int): Int {
        val context = dslContextProvider.ppc(shard)
        return context.fetchCount(
            context
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.METATYPE.eq(CampaignsMetatype.ecom)
                    .and(CAMPAIGNS.TYPE.eq(CampaignsType.text))
                    .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac)))
        )
    }

    fun countEcomCampaignsWithoutAllSubcampaigns(shard: Int): Int {
        val context = dslContextProvider.ppc(shard)
        return context.fetchCount(
            context
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .leftJoin(SUBCAMPAIGNS)
                .on(CAMPAIGNS.CID.eq(SUBCAMPAIGNS.MASTER_CID))
                .where(CAMPAIGNS.METATYPE.eq(CampaignsMetatype.ecom)
                    .and(CAMPAIGNS.TYPE.eq(CampaignsType.text))
                    .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac)))
                .groupBy(CAMPAIGNS.CID)
                .having(DSL.count(SUBCAMPAIGNS.MASTER_CID).lt(2))
        )
    }

    /**
     * Достаёт мап `master_cid -> List<subcampaign_cid>`.
     * При этом для мастер-кампании достаёт только те, которые были изменены более часа назад,
     * чтобы агрегированные статусы с большей вероятностью успели пересчитаться.
     */
    fun getAllEcomCampaignsWithSubcampaigns(shard: Int): Map<Long, List<Long>> {
        val context = dslContextProvider.ppc(shard)
        return context
            .select(CAMPAIGNS.CID, SUBCAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .join(SUBCAMPAIGNS)
            .on(CAMPAIGNS.CID.eq(SUBCAMPAIGNS.MASTER_CID))
            .where(CAMPAIGNS.METATYPE.eq(CampaignsMetatype.ecom)
                .and(CAMPAIGNS.TYPE.eq(CampaignsType.text))
                .and(CAMPAIGNS.SOURCE.eq(CampaignsSource.uac))
                .and(CAMPAIGNS.LAST_CHANGE.lt(LocalDateTime.now().minusHours(1))))
            .fetchGroups(CAMPAIGNS.CID, SUBCAMPAIGNS.CID)
    }
}
