package ru.yandex.direct.jobs.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.jobs.uac.UpdateAdsJobUtil
import ru.yandex.direct.jobs.uac.model.UacBanner
import ru.yandex.direct.jobs.uac.model.UpdateAdsContainer
import ru.yandex.direct.result.MassResult

@Service
class BannerCleanJobService(
    private val uacBannerService: UacBannerService,
    private val serviceHolder: UacJobServiceHolder,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(BannerCleanJobService::class.java)
    }

    /**
     * Удаляем баннеры в mysql и груте, отбирая список для удаления по изменениям в ассетах
     *
     * Аналог python функции clean() в updaters.py
     * https://a.yandex-team.ru/arc/trunk/arcadia/yabs/rmp/backend/src/uac/campaign/updaters.py?rev=r8172954#L541
     */
    fun clean(
        clientId: ClientId,
        uacCampaign: UacYdbCampaign,
        uacAssetsByGroupBriefId: Map<Long?, List<UacYdbCampaignContent>>,
        updateAdsContainers: List<UpdateAdsContainer>,
        isItCampaignBrief: Boolean,
        migrateToNewBackend: Boolean = false
    ) {
        val uacBannerJobService = serviceHolder.getUacBannerJobService(uacCampaign.id)
        val uacAssets = uacAssetsByGroupBriefId.values.flatten()
        val uacBanners = uacBannerJobService
            .getNotDeletedDirectAdsByCampaignContents(uacAssets, uacCampaign.id)

        val pidByBid = uacBanners
            .filter { it.bid != null && it.adGroupId != null }
            .associateBy({ it.bid!! }) { it.adGroupId!! }
        val uacBannersByBriefId: Map<Long?, List<UacBanner>> =
            if (isItCampaignBrief) mapOf(null to uacBanners)
            else UpdateAdsJobUtil.getUacBannersByBriefId(
                uacBanners,
                updateAdsContainers.map { it.brief },
                pidByBid,
            )

        for (container in updateAdsContainers) {
            val operatorUid = container.operatorUid
            val briefId = container.brief.adGroupBriefId

            val uacAssetsInBrief = uacAssetsByGroupBriefId[briefId] ?: emptyList()
            val uacBannersInBrief = uacBannersByBriefId[briefId] ?: emptyList()

            // Если это еще не обработанная групповая заявка или нет ассетов -> удалять нечего
            if (!isItCampaignBrief && (container.brief.adGroupIds.isNullOrEmpty() || uacAssetsInBrief.isEmpty())) {
                logger.info("Group brief $briefId does not contain groups or assets")
                continue
            }

            cleanBannersInGroupBrief(
                operatorUid,
                clientId,
                uacCampaign,
                uacAssetsInBrief,
                uacBannersInBrief,
                uacBannerJobService,
                migrateToNewBackend,
            )
        }
    }

    /**
     * Удаляем баннеры в mysql и груте
     */
    private fun cleanBannersInGroupBrief(
        operatorUid: Long,
        clientId: ClientId,
        uacCampaign: UacYdbCampaign,
        uacAssetsInBrief: List<UacYdbCampaignContent>,
        uacBannersInBrief: List<UacBanner>,
        uacBannerJobService: UacBannerJobService,
        migrateToNewBackend: Boolean = false,
    ) {
        val newContentTypes = uacAssetsInBrief
            .filter { it.removedAt == null }
            .mapNotNull { it.type }
            // отфильтровываем сайтлинки, т.к. они не участвуют в комбинаторной генерации баннеров
            // и не попадают в ассеты баннера (ru.yandex.direct.jobs.uac.model.UacBanner#assetIds)
            .filter { it != MediaType.SITELINK }
            .toSet()
        val currentAssetLinkIds = uacBannersInBrief
            .flatMap { it.assetLinkIds }
            .toSet()
        val currentContentTypes = uacAssetsInBrief
            .filter { currentAssetLinkIds.contains(it.id) }
            .mapNotNull { it.type }
            .toSet()

        val bannersToRemove = if ((newContentTypes - currentContentTypes).isEmpty() && !migrateToNewBackend) {
            // Если все новые типы контентов присутствуют в текущих типах контентов,
            // то удаляем баннера только с удаленными контентами.
            val removedUacCampaignContents = uacAssetsInBrief.filter { it.removedAt != null }
            // Получаем баннеры из Ydb, сгенерированные по удаленным контентам (в том числе староформатные банеры с 3-мя компонентами)
            // todo Постараться убрать запрос в базу DIRECT-155006
            uacBannerJobService.getNotDeletedDirectAdsByCampaignContents(
                removedUacCampaignContents, uacCampaign.id
            )
        } else {
            // Если появился новый тип контента, удаляем все баннеры.
            // Например у баннера был заголовок (TITLE) и текст (TEXT), а потом добавили две картинки (IMAGE)
            // - значит надо удалить старый баннер, чтобы потом сгенерились два новых баннера с картинками.
            uacBannersInBrief
        }

        val bannerIdsToRemove = bannersToRemove
            .mapNotNull { it.bid }
            .toSet()
        if (bannerIdsToRemove.isEmpty()) {
            return
        }
        logger.info("Try to remove [${bannerIdsToRemove.joinToString(separator = ",")}] banners")

        // Останавливаем баннеры в mysql
        val suspendResult = uacBannerService.suspendResumeBanners(clientId, operatorUid, false, bannerIdsToRemove)
        suspendResult.logErrors("banners suspension")

        // Удаляем/архивируем баннеры в mysql
        val (deleteMassResult, archiveMassResult) = uacBannerService
            .deleteAndArchiveBanners(operatorUid, clientId, bannerIdsToRemove)
        deleteMassResult.logErrors("banners deletion")
        archiveMassResult.logErrors("banners archivation")

        val successfulResultBannerIds = mutableSetOf<Long>()
        successfulResultBannerIds.addAll(getSuccessResults(deleteMassResult) + getSuccessResults(archiveMassResult))

        val failedResultBannerIds = bannerIdsToRemove - successfulResultBannerIds

        // Обновляем статусы баннеров в груте
        if (successfulResultBannerIds.isNotEmpty()) {
            uacBannerJobService.updateStatusByDirectAdIds(
                successfulResultBannerIds, DirectAdStatus.DELETED
            )
        }
        if (failedResultBannerIds.isNotEmpty()) {
            uacBannerJobService.updateStatusByDirectAdIds(
                failedResultBannerIds, DirectAdStatus.ERROR_UNKNOWN
            )
        }
    }

    private fun getSuccessResults(massResult: MassResult<Long>?): Set<Long> {
        return if (massResult?.result == null) {
            emptySet()
        } else massResult.result
            .filterNotNull()
            .filter { it.isSuccessful }
            .mapNotNull { it.result }
            .toSet()
    }

    private fun MassResult<Long>.logErrors(action: String = "action") {
        if (!this.isSuccessful) {
            logger.error("Result of $action was unsuccessful: {0}", this.validationResult.flattenErrors())
        }
    }
}
