package ru.yandex.direct.jobs.uac.service

import com.google.common.collect.ImmutableMap
import com.google.common.collect.Lists
import java.time.Duration
import kotlin.math.min
import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.banner.model.Age
import ru.yandex.direct.core.entity.banner.model.Banner
import ru.yandex.direct.core.entity.banner.model.BannerFlags
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.CpmBanner
import ru.yandex.direct.core.entity.banner.model.DynamicBanner
import ru.yandex.direct.core.entity.banner.model.ImageBanner
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner
import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.core.entity.banner.model.TextBanner
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.DEFAULT_MAX_BANNERS_IN_UAC_ADGROUP
import ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.DEFAULT_MAX_BANNERS_IN_UAC_TEXT_ADGROUP
import ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_BY_LISTINGS
import ru.yandex.direct.core.entity.banner.service.validation.BannerConstraints.getLimitBannersInGroup
import ru.yandex.direct.core.entity.bidmodifier.ComplexBidModifier
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.image.container.BannerImageType
import ru.yandex.direct.core.entity.image.model.BannerImageSource
import ru.yandex.direct.core.entity.image.service.ImageService
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService
import ru.yandex.direct.core.entity.sitelink.model.Sitelink
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService
import ru.yandex.direct.core.entity.uac.UacCommonUtils
import ru.yandex.direct.core.entity.uac.UacCommonUtils.getHrefWithTrackingParams
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.UacComplexBidModifier
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAppInfoRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacMeasurer
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectContent
import ru.yandex.direct.core.entity.uac.service.CatalogHrefWithBreadcrumbs
import ru.yandex.direct.core.entity.uac.service.EcomOfferCatalogsService
import ru.yandex.direct.core.entity.uac.service.EcomUcBannerService
import ru.yandex.direct.core.entity.uac.service.EcomUcCampaignService
import ru.yandex.direct.core.entity.uac.service.UacAdGroupService
import ru.yandex.direct.core.entity.uac.service.UacAppInfoService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.UacRetargetingService.toCoreRetargetingCondition
import ru.yandex.direct.core.grut.api.AdGroupBriefGrutModel
import ru.yandex.direct.core.validation.ValidationUtils
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.feature.FeatureName.ECOM_UC_NEW_BACKEND_ENABLED
import ru.yandex.direct.feature.FeatureName.ENABLED_BANNERS_BY_LISTINGS
import ru.yandex.direct.feature.FeatureName.MIGRATING_ECOM_CAMPAIGNS_TO_NEW_BACKEND
import ru.yandex.direct.jobs.uac.UpdateAdsJobUtil
import ru.yandex.direct.jobs.uac.model.BannerWithContents
import ru.yandex.direct.jobs.uac.model.UacBanner
import ru.yandex.direct.jobs.uac.model.UpdateAdsContainer
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.utils.CommonUtils
import ru.yandex.direct.utils.mapToSet

abstract class BannerCreateJobService(
    private val uacYdbAppInfoRepository: UacYdbAppInfoRepository,
    private val uacAppInfoService: UacAppInfoService,
    private val uacAdGroupService: UacAdGroupService,
    private val uacBannerService: UacBannerService,
    private val bannerModerateService: BannerModerateService,
    private val bannerService: BannerService,
    private val sitelinkSetService: SitelinkSetService,
    private val ucRetargetingConditionService: UcRetargetingConditionService,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val uacBannerJobService: UacBannerJobService,
    private val uacAdGroupJobService: UacAdGroupJobService,
    private val ecomUcCampaignService: EcomUcCampaignService,
    private val ecomUcBannerService: EcomUcBannerService,
    private val uacKeywordJobService: UacKeywordJobService,
    private val retargetingConditionService: RetargetingConditionService,
    private val featureService: FeatureService,
    private val ecomOfferCatalogsService: EcomOfferCatalogsService,
    private val imageService: ImageService
) {
    private val maxBannersInUacAdGroupProperty: PpcProperty<Int> = ppcPropertiesSupport
        .get(PpcPropertyNames.MAX_BANNERS_IN_UAC_AD_GROUP, Duration.ofMinutes(5))
    private val maxBannersInUacTextAdGroupProperty: PpcProperty<Int> = ppcPropertiesSupport
        .get(PpcPropertyNames.MAX_BANNERS_IN_UAC_TEXT_AD_GROUP, Duration.ofMinutes(5))

    companion object {
        private val logger = LoggerFactory.getLogger(BannerCreateJobService::class.java)
        private const val MODERATE_BANNERS_CHUNK = 100
        private const val AD_GROUPS_UPDATE_CHUNK = 40
    }

    /**
     * Аналог python функции create_new_ads() в updaters.py
     * https://a.yandex-team.ru/arc/trunk/arcadia/yabs/rmp/backend/src/uac/campaign/updaters.py?rev=r8172954#L545
     * + обновление уже существующих баннеров и групп.
     */
    fun createNewAdsAndUpdateExist(
        client: Client,
        updateAdsContainers: List<UpdateAdsContainer>,
        uacCampaign: UacYdbCampaign,
        uacDirectAdGroups: Collection<UacYdbDirectAdGroup>,
        uacAssetsByGroupBriefId: Map<Long?, List<UacYdbCampaignContent>>,
        isItCampaignBrief: Boolean,
    ) {
        val clientId = ClientId.fromLong(client.clientId)

        val ecomUcNewBackendEnabled = featureService.isEnabledForClientId(clientId, ECOM_UC_NEW_BACKEND_ENABLED)
        val enabledBannersByListings = featureService.isEnabledForClientId(clientId, ENABLED_BANNERS_BY_LISTINGS)
        val migrateToNewBackend = featureService.isEnabledForClientId(clientId, MIGRATING_ECOM_CAMPAIGNS_TO_NEW_BACKEND)

        val uacDirectAdGroupsByPid = uacDirectAdGroups
            .associateBy { it.directAdGroupId }
        val uacDirectAdGroupsByBriefId = updateAdsContainers
            .associateBy { it.brief.adGroupBriefId }
            .mapValues {
                if (isItCampaignBrief) uacDirectAdGroups
                else it.value.brief.adGroupIds
                    ?.mapNotNull { pid ->
                        uacDirectAdGroupsByPid[pid]
                    } ?: emptyList()
            }

        val uacAssets = uacAssetsByGroupBriefId.values.flatten()
        val uacBanners = uacBannerJobService.getNotDeletedDirectAdsByCampaignContents(uacAssets, uacCampaign.id)

        // Получаем все баннеры, сгруппированные по групповым заявкам
        val pidByBid = uacBanners
            .filter { it.bid != null && it.adGroupId != null }
            .associateBy({ it.bid!! }) { it.adGroupId!! }
        val uacBannersByAdGroupBriefId: Map<Long?, List<UacBanner>> =
            if (isItCampaignBrief) mapOf(null to uacBanners)
            else UpdateAdsJobUtil.getUacBannersByBriefId(
                uacBanners,
                updateAdsContainers.map { it.brief },
                pidByBid,
            )

        for (container in updateAdsContainers) {
            val adGroupBriefId = container.brief.adGroupBriefId
            val uacDirectAdGroupsInBrief = uacDirectAdGroupsByBriefId[adGroupBriefId] ?: emptyList()
            val uacCampaignContentsInBrief = uacAssetsByGroupBriefId[adGroupBriefId] ?: emptyList()
            val uacBannersInBrief = uacBannersByAdGroupBriefId[adGroupBriefId] ?: emptyList()

            createNewAdsAndUpdateExist(
                container,
                uacDirectAdGroupsInBrief,
                uacCampaignContentsInBrief,
                uacBannersInBrief,
                ecomUcNewBackendEnabled,
                enabledBannersByListings,
                migrateToNewBackend,
            )
        }
    }

    /**
     * Создает баннеры по переданным ассетам
     */
    private fun createNewAdsAndUpdateExist(
        updateAdsContainer: UpdateAdsContainer,
        uacDirectAdGroupsInBrief: Collection<UacYdbDirectAdGroup>,
        uacCampaignContentsInBrief: List<UacYdbCampaignContent>,
        uacBannersInBrief: List<UacBanner>,
        ecomUcNewBackendEnabled: Boolean,
        enabledBannersByListings: Boolean,
        migrateToNewBackend: Boolean,
    ) {
        val brief = updateAdsContainer.brief
        val directCampaign = updateAdsContainer.campaign
        val client = updateAdsContainer.client
        val clientId = ClientId.fromLong(client.clientId)

        val activeUacAssetsInBrief = uacCampaignContentsInBrief
            .filter { it.removedAt == null }

        // Собираем direct_content и добавляем недостающие
        val uacAssets = getAndCreateContents(clientId, activeUacAssetsInBrief)
        val uacDirectContentsByTypeAndId: Map<DirectContentType, Map<String, UacYdbDirectContent>> =
            uacAssets
                .groupBy { it.type }
                .mapValues {
                    it.value.associateBy { uacDirectContent -> uacDirectContent.id }
                }

        // Группируем CampaignContent по типам и ID для удобства
        val activeUacAssetsInBriefByType: Map<MediaType, List<UacYdbCampaignContent>> =
            UpdateAdsJobUtil.groupCampaignContentByType(activeUacAssetsInBrief)
        val activeUacAssetsInBriefByTypeAndId: Map<MediaType, Map<String, UacYdbCampaignContent>> =
            activeUacAssetsInBriefByType
                .mapValues { it.value.associateBy { uacAsset -> uacAsset.id } }

        // Общие запчасти для создания баннеров
        val uacComplexBidModifier = constructUacComplexBidModifier(client, brief)
        val appInfo = getAppInfoIfApplicable(brief)
        val sitelinkSetId = createSitelinkSet(
            client, directCampaign.id, activeUacAssetsInBriefByType, brief.trackingParams)
        val retargetingCondition = makeRetargetingCondition(client, brief)
        val bizLandingUrl: String? = getBizLandingUrlOrNull(brief)

        // Неполные группы, куда можно добавить ещё баннеров
        val adGroupsWithMinimumAdsByType = uacAdGroupJobService
            .getAdGroupIdsWithMinimumUacAdsByTypes(clientId, uacDirectAdGroupsInBrief.mapToSet { it.id })
        // субкампании товарной компании (типы отличаются от кампании-родителя)
        val subCampaigns = ecomUcCampaignService.getSubCampaigns(clientId, listOf(directCampaign.id))
            .associateBy { it.type }

        // Хранилища для созданных объектов
        val createdBanners = mutableListOf<BannerWithContents>()
        val createdAdGroupIds = mutableSetOf<Long>()

        // функция создания объявлений. Для субкампаний вызывается отдельно
        val createBanners: (
            AdGroupType, List<BannerWithSourceContents>, Long, List<Set<String>>, AdGroupIdToBannersCnt?, Boolean
        ) -> Pair<List<BannerWithContents>, Collection<Long>> =
            { adGroupType, banners, directCampaignId, newContents, adGroupIdToBannersCnt, bannersByListings ->
                val complexBidModifier = UacCommonUtils.getComplexBidModifier(uacComplexBidModifier, directCampaignId)
                val actualAdGroupIdToBannersCnt = if (bannersByListings || adGroupIdToBannersCnt != null) {
                    adGroupIdToBannersCnt
                } else {
                    adGroupsWithMinimumAdsByType[adGroupType]
                }
                val (newBanners, newAdGroups) = createBannersAndGroupsByNewContents(
                    adGroupType,
                    banners,
                    actualAdGroupIdToBannersCnt,
                    updateAdsContainer,
                    subCampaigns,
                    newContents,
                    appInfo,
                    complexBidModifier,
                    retargetingCondition
                )
                createdBanners.addAll(newBanners)
                createdAdGroupIds.addAll(newAdGroups)
                newBanners to newAdGroups
            }

        val getFilledBanners: (List<ContentsByType>, Class<out Banner>) -> List<BannerWithSourceContents> =
            { contents, bannerType ->
                toFilledBanners(
                    contents,
                    brief,
                    appInfo,
                    sitelinkSetId,
                    bannerType,
                    retargetingCondition,
                    bizLandingUrl)
        }

        val contentIdsInBanners = uacBannersInBrief
            .mapToSet { it.assetLinkIds }

        val contentIdsCombinations = generateAllPossibleContentCombinations(
            activeUacAssetsInBriefByType,
            advType = brief.advType
        )
        val newContentIdsCombinations = contentIdsCombinations - contentIdsInBanners

        val contentsByType = groupNewContentsByType(
            newContentIdsCombinations,
            activeUacAssetsInBriefByTypeAndId,
            uacDirectContentsByTypeAndId,
            brief.videosAreNonSkippable
        )

        val filledNonEcomBanners = getFilledBanners(contentsByType, brief.advType.toBannerType())
        var catalogIds : Set<Long>? = null
        if (brief.isValidEcomBrief(subCampaigns, ecomUcNewBackendEnabled)) {
            val newUacDirectDynamicAdContents =
                getNewContentsForDynamicAds(activeUacAssetsInBriefByType, newContentIdsCombinations, contentIdsInBanners)
            val newUacDirectDynamicAdContentsByType = groupNewContentsByType(
                newUacDirectDynamicAdContents,
                activeUacAssetsInBriefByTypeAndId,
                uacDirectContentsByTypeAndId,
                brief.videosAreNonSkippable)
            val dynamicBanners = getFilledBanners(newUacDirectDynamicAdContentsByType, DynamicBanner::class.java)
            if ((subCampaigns.isEmpty() || migrateToNewBackend) && ecomUcNewBackendEnabled) {
                val existingSmartBanners = bannerService.getBannersByCampaignIds(
                    setOf(directCampaign.id),
                    PerformanceBannerMain::class.java,
                )
                val smartBanners = if (existingSmartBanners.isEmpty()) {
                    getFilledBanners(listOf(), PerformanceBannerMain::class.java)
                } else {
                    listOf()
                }
                val bannersToCreate = filledNonEcomBanners + dynamicBanners + smartBanners
                val newContents = newContentIdsCombinations + newUacDirectDynamicAdContents + smartBanners.map { setOf() }
                createBanners(AdGroupType.BASE, bannersToCreate, directCampaign.id, newContents, null, false)
            } else {
                createBanners(AdGroupType.BASE, filledNonEcomBanners, directCampaign.id, newContentIdsCombinations, null, false)
                createBanners(
                    AdGroupType.DYNAMIC,
                    dynamicBanners,
                    subCampaigns[CampaignType.DYNAMIC]!!.id,
                    newUacDirectDynamicAdContents,
                    null,
                    false
                )

                val smartBidModifier = UacCommonUtils
                    .getComplexBidModifier(uacComplexBidModifier, subCampaigns[CampaignType.PERFORMANCE]!!.id)
                val (smartBanners, smartGroups) = createSmartBanners(updateAdsContainer, subCampaigns, smartBidModifier)
                createdBanners.addAll(smartBanners)
                createdAdGroupIds.addAll(smartGroups)
            }

            if (enabledBannersByListings) {
                val url = brief.url
                val campaignId = brief.campaignId.toIdLong()
                val listingsAdGroupBrief = getAdGroupBriefByListings(campaignId)
                val existingCatalogs = listingsAdGroupBrief?.catalogIds?.toSet() ?: emptySet()
                catalogIds = existingCatalogs

                val newCatalogs =
                    ecomOfferCatalogsService.getCatalogsByHosts(listOf(brief.url))[url]
                        ?.filter { !existingCatalogs.contains(it.catalogId) }
                        ?.take(MAX_BANNERS_BY_LISTINGS - existingCatalogs.size)

                if (!newCatalogs.isNullOrEmpty()) {
                    val bannersToCreate = convertListingsToBanners(clientId, newCatalogs, brief, sitelinkSetId)

                    val contents = bannersToCreate.map { emptySet<String>() }

                    val adGroupIdToBannersCnt =
                        if (listingsAdGroupBrief != null && !listingsAdGroupBrief.adGroupIds.isNullOrEmpty()) {
                            val adGroupId = listingsAdGroupBrief.adGroupIds!!.first()
                            AdGroupIdToBannersCnt(adGroupId, existingCatalogs.size)
                        } else null

                    val (_, newAdGroupIds) = createBanners(
                        AdGroupType.BASE, bannersToCreate, directCampaign.id, contents, adGroupIdToBannersCnt, true
                    )

                    catalogIds = newCatalogs
                        .map { it.catalogId }
                        .plus(existingCatalogs)
                        .toSet()

                    // Если групповой заявки еще нет - создаем новую
                    // Если уже есть - будет обновлена ниже
                    if (brief.adGroupBriefId == null) {
                        createAdGroupBriefByListings(
                            directCampaign.id,
                            catalogIds.toList(),
                            newAdGroupIds,
                        )
                    }
                }
            }
        } else {
            createBanners(
                brief.advType.toAdGroupType(),
                filledNonEcomBanners,
                directCampaign.id,
                newContentIdsCombinations,
                null,
                false,
            )
        }

        // Обновляем id групп в групповой заявке (а так же каталоги для листингов)
        if (brief.adGroupBriefId != null) {
            val adGroupIds = (brief.adGroupIds ?: emptySet()) + createdAdGroupIds
            // При выключении листингов каталоги не обновляем в заявке, чтобы не затереть уже существующие
            if (enabledBannersByListings) {
                updateAdGroupBriefCatalogsAndGroups(
                    brief,
                    catalogIds?.toList(),
                    adGroupIds,
                )
            } else {
                updateAdGroupBriefGroupIds(
                    brief,
                    adGroupIds,
                )
            }
        }

        val allCampaigns = mutableMapOf<Long, CommonCampaign>(directCampaign.id to directCampaign)
        allCampaigns.putAll(subCampaigns.values.map { it.id to it })

        updateAdGroups(updateAdsContainer, uacDirectAdGroupsInBrief, allCampaigns, retargetingCondition)
        updateAds(
            uacBannersInBrief,
            brief,
            sitelinkSetId,
            retargetingCondition,
            bizLandingUrl,
            clientId,
            updateAdsContainer.operatorUid,
        )
        sendCreatedBannersToModeration(updateAdsContainer, createdBanners)
        syncMysqlPhrasesToUacStorage(updateAdsContainer, createdAdGroupIds, uacDirectAdGroupsInBrief)
    }

    private fun getAppInfoIfApplicable(brief: UacAdGroupBrief): AppInfo? {
        val isMobileContent = brief.advType == AdvType.MOBILE_CONTENT
        if (!isMobileContent) {
            return null
        }
        val uacYdbAppInfo = uacYdbAppInfoRepository.getAppInfoById(brief.appId!!)
        checkNotNull(uacYdbAppInfo) { "Uac app info not found by app id: ${brief.appId}" }
        return uacAppInfoService.getAppInfo(uacYdbAppInfo)
    }

    open fun getBizLandingUrlOrNull(brief: UacAdGroupBrief): String? {
        return null
    }

    private fun updateAdGroups(
        container: UpdateAdsContainer,
        uacDirectAdGroups: Collection<UacYdbDirectAdGroup>,
        allCampaigns: Map<Long, CommonCampaign>,
        retargetingCondition: RetargetingCondition?,
    ) {
        if (uacDirectAdGroups.isEmpty()) {
            return
        }
        val adGroupIdsForUpdate = uacDirectAdGroups.map { it.directAdGroupId }
        val allAdGroupsByCampaignIds = ecomUcCampaignService.getAdGroupsByCampaignIds(container.getClientId(), allCampaigns.keys)

        val allAdGroupIds = allAdGroupsByCampaignIds.flatMap { it.value }.map { it.id }
        if (!allAdGroupIds.containsAll(adGroupIdsForUpdate)) {
            throw IllegalStateException("Unknown ad groups for update")
        }
        val groupTypeById = allAdGroupsByCampaignIds
            .values
            .flatten()
            .associate { it.id to it.type }

        val keywords = uacKeywordJobService.getKeywordsInUacCampaign(container.brief)

        allCampaigns.map { it.value to allAdGroupsByCampaignIds[it.key] }
            .toMap()
            .filter { !it.value.isNullOrEmpty() }
            .mapValues { it.value!!.map(AdGroup::getId) }
            .mapValues { it.value.filter(adGroupIdsForUpdate::contains) }
            .forEach { (campaignToUpdate, adGroupIds) ->
                val uacComplexBidModifier = constructUacComplexBidModifier(container.client, container.brief)
                val complexBidModifier =
                    UacCommonUtils.getComplexBidModifier(uacComplexBidModifier, campaignToUpdate.id)
                var leftIndex = 0
                while (leftIndex < adGroupIds.size) {
                    val rightIndex = leftIndex + minOf(AD_GROUPS_UPDATE_CHUNK, adGroupIds.size - leftIndex)
                    val adGroupIdsChunk = adGroupIds.subList(leftIndex, rightIndex)
                    val massResult = uacAdGroupService.updateAdGroups(
                        adGroupIdsChunk,
                        groupTypeById,
                        container.operatorUid,
                        container.client,
                        container.brief,
                        keywords,
                        complexBidModifier,
                        retargetingCondition,
                        campaignToUpdate,
                    )
                    if (massResult.validationResult?.hasAnyErrors() == true) {
                        logger.error(
                            "Can't update {}groups with ids: {}: {}",
                            if (container.brief.isEcom == true) "ecom " else "",
                            adGroupIds, massResult.validationResult.flattenErrors()
                        )
                    }
                    leftIndex = rightIndex
                }
            }
    }

    private fun constructUacComplexBidModifier(
        client: Client,
        brief: UacAdGroupBrief,
    ): UacComplexBidModifier {
        return UacComplexBidModifier(
            advType = brief.advType,
            socdem = brief.socdem,
            deviceTypes = brief.deviceTypes,
            inventoryTypes = null,
            isSmartTVEnabled = featureService.isEnabledForClientId(
                ClientId.fromLong(client.clientId), FeatureName.SMARTTV_BID_MODIFIER_ENABLED
            ),
            isTabletModifierEnabled = featureService.isEnabledForClientId(
                ClientId.fromLong(client.clientId), FeatureName.TABLET_BIDMODIFER_ENABLED
            ),
        )
    }

    private fun syncMysqlPhrasesToUacStorage(
        container: UpdateAdsContainer,
        createdAdGroupIds: Collection<Long>,
        oldUacDirectAdGroups: Collection<UacYdbDirectAdGroup>,
    ) {
        val brief = container.brief
        val clientId = ClientId.fromLong(container.client.clientId)

        val adGroupIdToSyncKeywords = if (brief.isEcom == true) {
            val allAdGroupIds = uacAdGroupJobService.getAdGroupsByUacCampaignId(brief.campaignId, container.campaign.id)
                .asSequence()
                .map { it.id }
                .toSet()
            val adGroupsWithMinimumAdsByType =
                uacAdGroupJobService.getAdGroupIdsWithMinimumUacAdsByTypes(clientId, allAdGroupIds)
            adGroupsWithMinimumAdsByType[AdGroupType.BASE]?.adGroupId
        } else {
            createdAdGroupIds.firstOrNull() ?: oldUacDirectAdGroups.firstOrNull()?.directAdGroupId
        }

        if (adGroupIdToSyncKeywords == null) {
            logger.error("No suitable ad group found for campaign ${brief.campaignId}")
            return
        }
        uacAdGroupService.syncMysqlPhrasesToUacStorage(
            adGroupIdToSyncKeywords, brief.advType.toAdGroupType(),
            brief.keywords, brief.minusKeywords
        )
    }

    private fun updateAds(
        uacYdbDirectAds: List<UacBanner>,
        brief: UacAdGroupBrief,
        sitelinkSetId: Long?,
        retargetingCondition: RetargetingCondition?,
        bizLandingUrl: String?,
        clientId: ClientId,
        operatorUid: Long,
    ) {
        val bidsForUpdate = uacYdbDirectAds.mapNotNull { it.bid }
        if (bidsForUpdate.isEmpty()) {
            return
        }
        val assetLinkIdsByBid: Map<Long, Set<String>> = uacYdbDirectAds
            .filter { it.bid != null }.associate { it.bid!! to it.assetLinkIds }

        val bannersByType = bannerService.getBannersByIds(bidsForUpdate)
            .groupBy { it::class.java }
            .mapValues { it.value.map(BannerWithSystemFields::getId) }
        bannersByType.forEach { (bannerType, idsToUpdate) ->
            val changesTemplate = getBannerChanges(bannerType, brief, sitelinkSetId, retargetingCondition, bizLandingUrl)
            val massResult =
                uacBannerJobService.updateBanners(
                    clientId,
                    operatorUid,
                    idsToUpdate,
                    changesTemplate,
                    brief,
                    assetLinkIdsByBid,
                )
            if (massResult.validationResult?.hasAnyErrors() == true) {
                logger.error(
                    "Can't update {}banners with ids: {}: {}",
                    if (brief.isEcom == true) "ecom " else "",
                    idsToUpdate, massResult.validationResult.flattenErrors()
                )
            }
        }
    }

    /**
     * Для создания ДО нам не нужна вся комбинаторика, эти баннеры создаём только по текстам
     */
    private fun getNewContentsForDynamicAds(
        campaignContentsByType: Map<MediaType, List<UacYdbCampaignContent>>,
        newUacDirectAdContents: List<Set<String>>,
        currentContents: Set<Set<String>>
    ): List<Set<String>> {
        val existingAssetsIds = currentContents.flatten().toSet()
        val textAssetLinkIds = campaignContentsByType[MediaType.TEXT]
            ?.mapToSet { it.id }
            ?.filter { !existingAssetsIds.contains(it) }
            ?: emptySet()
        return newUacDirectAdContents.asSequence()
            .map { it.filterTo(mutableSetOf(), textAssetLinkIds::contains) }
            .filter { it.isNotEmpty() }
            .distinct()
            .toList()
    }

    /**
     * Смарт баннеры создаются не по контентам, а по фиду и фильтру с набором стандартных креативов
     * <p>
     * Поскольку из пользовательского ввода для создания смарт баннеров используется только домен из
     * ссылки, которую после создания редактировать нельзя, смарт баннеры создаются лишь единожды.
     */
    private fun createSmartBanners(
        container: UpdateAdsContainer,
        subCampaigns: Map<CampaignType, CommonCampaign>,
        complexBidModifier: ComplexBidModifier?
    ): Pair<List<BannerWithContents>, List<Long>> {
        val (operatorUid, client, brief) = container
        val clientId = ClientId.fromLong(client.id)
        val bannerCampaign = subCampaigns[CampaignType.PERFORMANCE]!!

        // Проверяем, что смарт-баннеры уже есть:
        val existingBanners = bannerService.getBannersByCampaignIds(
            setOf(bannerCampaign.id),
            setOf(PerformanceBanner::class.java, PerformanceBannerMain::class.java))
        if (existingBanners.isNotEmpty()) {
            return Pair(listOf(), listOf())
        }

        val (banners, result) = if (featureService.isEnabledForClientId(clientId, FeatureName.CREATIVE_FREE_ECOM_UC)) {
            val banner = PerformanceBannerMain()
            val result = uacAdGroupService.createSmartGroupWithMainBanner(
                operatorUid, client, bannerCampaign, complexBidModifier, banner, brief)
            Pair(listOf(banner), result)
        } else {
            val banners = ecomUcBannerService.fillEcomPerformanceBanners(ClientId.fromLong(client.id), brief)
            val result = uacAdGroupService.createSmartGroupWithBanner(
                operatorUid, client, bannerCampaign, complexBidModifier, banners, brief)
            Pair(banners, result)
        }
        if (result.validationResult.hasAnyErrors()) {
            logger.error(
                "Can't create a new smart group with banners for ydb:campaign.id: {}, first error is: {}",
                brief.campaignId, result.validationResult.flattenErrors()[0].toString()
            )
        }
        saveBannersAndAdGroupAndUpdateLists(
            List(banners.size) { ContentsByType.emptyContents() }, banners,
            brief.campaignId, mutableListOf(), mutableSetOf(), true
        )

        val createdBanners = banners
            .filter { it.id != null }
            .map { BannerWithContents(it, setOf()) }
        val createdGroups = banners.mapNotNull { it.adGroupId }
        return createdBanners to createdGroups
    }

    private fun createBannersAndGroupsByNewContents(
        adGroupType: AdGroupType,
        banners: List<BannerWithSourceContents>,
        adGroupIdAndAdsCount: AdGroupIdToBannersCnt?,
        container: UpdateAdsContainer,
        subCampaigns: Map<CampaignType, CommonCampaign>,
        newUacDirectAdContents: List<Set<String>>,
        appInfo: AppInfo?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?
    ): Pair<List<BannerWithContents>, Collection<Long>> {
        val bannersAndNewGroups = createBanners(
            container,
            subCampaigns,
            banners,
            appInfo,
            complexBidModifier,
            retargetingCondition,
            adGroupIdAndAdsCount,
            adGroupType
        )

        val newBanners = bannersAndNewGroups.first

        val createdBanners = mutableListOf<BannerWithContents>()
        if (newUacDirectAdContents.size != newBanners.size) {
            logger.error("New contents and created banners have different sizes for campaign ${container.campaign.id}")
        }
        newUacDirectAdContents.zip(newBanners) { contents, banner ->
            if (banner.id != null) {
                createdBanners.add(BannerWithContents(banner, contents))
            }
        }
        val createdAdGroups = bannersAndNewGroups.second

        return createdBanners to createdAdGroups
    }

    // Группируем контенты по типам — заголовки, тексты, картинки, пропускаемые видео, непропускаемые видео
    private fun groupNewContentsByType(
        newUacDirectAdContents: List<Set<String>>,
        campaignContentsByTypeAndId: Map<MediaType, Map<String, UacYdbCampaignContent>>,
        directContentByTypeAndId: Map<DirectContentType, Map<String, UacYdbDirectContent>>,
        videosAreNonSkippable: Boolean?
    ) = newUacDirectAdContents.map { uacDirectAdContents ->
        groupContentsByType(
            uacDirectAdContents,
            campaignContentsByTypeAndId,
            directContentByTypeAndId,
            videosAreNonSkippable == true
        )
    }

    private fun sendCreatedBannersToModeration(
        container: UpdateAdsContainer,
        createdBanners: List<BannerWithContents>
    ) {
        if (createdBanners.isEmpty()) {
            return
        }
        val (operatorUid, client, brief) = container
        val clientId = ClientId.fromLong(client.clientId)
        // Отправляем на модерацию первый баннер
        val firstBanner = createdBanners.first()
        val isModerationSuccessful =
            sendToModeration(operatorUid, clientId, listOf(firstBanner))
        if (isModerationSuccessful) {
            touchCampaign(brief.campaignId)
        }

        // Отправляем на модерацию остальные
        val bannersToModerate = createdBanners
            .filter { b -> b.banner.id != firstBanner.banner.id }
        if (bannersToModerate.isNotEmpty()) {
            sendToModeration(operatorUid, clientId, bannersToModerate)
        }
    }

    private fun createSitelinkSet(
        client: Client,
        campaignId: Long,
        uacCampaignContents: Map<MediaType, List<UacYdbCampaignContent>>,
        trackingParams: String?
    ): Long? {
        val sitelinks = (uacCampaignContents[MediaType.SITELINK] ?: emptyList())
            .sortedBy { it.order }
            .map {
                val href = it.sitelink?.href
                    ?.let { href ->
                        // Добавляем в href трекинговые параметры
                        getHrefWithTrackingParams(href, trackingParams)
                    }

                Sitelink()
                    .withHref(href)
                    .withTitle(it.sitelink?.title)
                    .withDescription(it.sitelink?.description)
                    .withOrderNum(it.order.toLong())
            }
        if (sitelinks.isEmpty()) {
            return null
        }
        val massResult = sitelinkSetService.addSitelinkSetsFull(
            ClientId.fromLong(client.clientId),
            listOf(SitelinkSet().withSitelinks(sitelinks))
        )

        if (!massResult.isSuccessful || massResult.toResultList().isEmpty() || !massResult[0].isSuccessful) {
            logger.error("Can't create sitelinks for campaign $campaignId: ${massResult.validationResult.flattenErrors()}")
            throw IllegalStateException("Can't create sitelinks.")
        }
        return massResult[0].result
    }

    private fun <T> findContentInContentMap(
        contentIds: Set<String>,
        idToContent: Map<String, T>,
    ): T? {
        return contentIds.asSequence()
            .map { idToContent[it] }
            .firstOrNull { it != null }
    }

    private fun makeRetargetingCondition(
        client: Client,
        brief: UacAdGroupBrief,
    ): RetargetingCondition? {
        return if (brief.advType == AdvType.CPM_BANNER) {
            toCoreRetargetingCondition(
                brief.retargetingCondition,
                client.clientId,
                brief.socdem
            )
        } else if (brief.advType == AdvType.TEXT) {
            val ucCustomAudienceEnabled = featureService.isEnabledForClientId(
                ClientId.fromLong(client.clientId), FeatureName.UC_CUSTOM_AUDIENCE_ENABLED
            )
            if (ucCustomAudienceEnabled) {
                if (brief.retargetingCondition == null) null else {
                    toCoreRetargetingCondition(
                        brief.retargetingCondition,
                        client.clientId,
                    )
                }
            } else {
                ucRetargetingConditionService.getAutoRetargetingCondition(
                    ClientId.fromLong(client.clientId),
                    brief.counters,
                    brief.goals?.map { it.goalId },
                    null
                )
            }
        } else {
            retargetingConditionService.getRetargetingConditions(
                ClientId.fromLong(client.clientId),
                listOf(brief.retargetingCondition?.id),
                LimitOffset.maxLimited()
            ).firstOrNull()
        }
    }

    private fun fillBanner(
        contentsByType: ContentsByType,
        brief: UacAdGroupBrief,
        appInfo: AppInfo?,
        sitelinkSetId: Long?,
        bannerClass: Class<out Banner>,
        retargetingCondition: RetargetingCondition?,
        bizLandingUrl: String?,
    ): BannerWithSystemFields {
        val titleCampaignContent = contentsByType.titleCampaignContent
        val textCampaignContent = contentsByType.textCampaignContent
        val imageDirectContent = contentsByType.imageDirectContent
        val html5DirectContent = contentsByType.html5VideoDirectContent
        val videoDirectContent = contentsByType.videoDirectContent
        when (bannerClass) {
            MobileAppBanner::class.java, ImageBanner::class.java -> {
                val ageLimit = toDirectAgeLimit(appInfo)

                return if (html5DirectContent != null) {
                    ImageBanner()
                        .withFlags(if (ageLimit == null) null else BannerFlags().with(BannerFlags.AGE, ageLimit))
                        .withHref(brief.trackingUrl)
                        .withCreativeId(html5DirectContent.directHtml5Id)
                        .withIsMobileImage(true)
                } else {
                    val primaryAction = getPrimaryActionByRetargetingCondition(retargetingCondition)
                    MobileAppBanner()
                        .withFlags(if (ageLimit == null) null else BannerFlags().with(BannerFlags.AGE, ageLimit))
                        .withPrimaryAction(primaryAction)
                        .withTitle(titleCampaignContent!!.text)
                        .withBody(textCampaignContent!!.text)
                        .withHref(brief.trackingUrl)
                        .withImpressionUrl(brief.impressionUrl)
                        .withImageHash(imageDirectContent?.directImageHash)
                        .withCreativeId(videoDirectContent?.directVideoId)
                        .withReflectedAttributes(
                            ImmutableMap.of(
                                NewReflectedAttribute.PRICE, true,
                                NewReflectedAttribute.ICON, true,
                                NewReflectedAttribute.RATING, true,
                                NewReflectedAttribute.RATING_VOTES, true
                            )
                        )
                        .withShowTitleAndBody(brief.showTitleAndBody)
                }
            }
            TextBanner::class.java -> {
                val href = bizLandingUrl ?: getHrefWithTrackingParams(brief.url, brief.trackingParams)
                return TextBanner()
                    .withHref(href)
                    .withTitle(titleCampaignContent!!.text)
                    .withBody(textCampaignContent!!.text)
                    .withImageHash(imageDirectContent?.directImageHash)
                    .withCreativeId(videoDirectContent?.directVideoId)
                    .withPermalinkId(brief.permalinkId)
                    .withPhoneId(brief.phoneId)
                    .withSitelinksSetId(sitelinkSetId)
                    .withZenPublisherId(brief.zenPublisherId)
                    .withShowTitleAndBody(brief.showTitleAndBody)
            }
            CpmBanner::class.java -> {

                if (videoDirectContent != null && html5DirectContent != null) {
                    throw java.lang.IllegalStateException("Both creative types defined for cpm_banner banner")
                }

                val content = (videoDirectContent ?: html5DirectContent)

                val asset = content?.id?.let { brief.cpmAssets?.get(it) }
                val href = if (asset?.bannerHref.isNullOrEmpty()) {
                    brief.url
                } else {
                    asset?.bannerHref
                }

                return CpmBanner()
                    .withLogoImageHash(asset?.logoImageHash)
                    .withButtonAction(asset?.button?.action?.toCoreButtonAction())
                    .withButtonHref(asset?.button?.href)
                    .withTitle(asset?.title)
                    .withBody(asset?.body)
                    .withPixels(asset?.pixels)
                    .withMeasurers(asset?.measurers?.map(this::toBannerMeasurer))
                    .withHref(href)
                    .withCreativeId(content?.directVideoId)
                    .withPermalinkId(brief.permalinkId)
                    .withShowTitleAndBody(brief.showTitleAndBody)
            }
            DynamicBanner::class.java -> {
                return DynamicBanner()
                    .withHref(brief.url)
                    .withBody(textCampaignContent!!.text)
                    .withSitelinksSetId(sitelinkSetId)
            }
            else -> {
                throw IllegalStateException("Unsupported ad type: ${bannerClass.canonicalName}")
            }
        }
    }

    private fun toBannerMeasurer(uacMeasurer: UacMeasurer): BannerMeasurer? {
        return BannerMeasurer()
            .withBannerMeasurerSystem(uacMeasurer.measurerType.toCoreBannerMeasurerSystem())
            .withParams(uacMeasurer.params)
    }

    // Собирает все варианты контента баннеров, для дальнейшего поиска недостающих
    private fun generateAllPossibleContentCombinations(
        grouppedContents: Map<MediaType, List<UacYdbCampaignContent?>>,
        advType: AdvType,
    ): List<Set<String>> {
        val sources: MutableList<List<String>> = mutableListOf()

        sources.addAll(listOf(MediaType.TITLE, MediaType.TEXT, MediaType.IMAGE)
            .map {
                grouppedContents.getOrDefault(it, listOf(null))
                    .map { e -> e?.id ?: "" }
            })

        val videoCreatives = grouppedContents.getOrDefault(MediaType.VIDEO, setOf()).map { e ->
            e?.id ?: ""
        }.toMutableList()
        var html5Creatives = grouppedContents.getOrDefault(MediaType.HTML5, setOf()).map { e -> e?.id ?: "" }
        if (advType != AdvType.MOBILE_CONTENT) {
            videoCreatives.addAll(html5Creatives)
            html5Creatives = listOf("")
        }
        if (videoCreatives.isEmpty()) {
            videoCreatives.add("")
        }

        sources.addAll(listOf(videoCreatives))
        sources.addAll(listOf(listOf("")))

        val combinations: MutableList<MutableList<String>> = mutableListOf()
        combinations.addAll(Lists.cartesianProduct(sources))
        combinations.addAll(html5Creatives.map { mutableListOf("", "", "", "", it) })

        return combinations.map {
            setOfNotNull(
                if (it[0] == "") null else it[0],
                if (it[1] == "") null else it[1],
                if (it[2] == "") null else it[2],
                if (it[3] == "") null else it[3],
                if (it[4] == "") null else it[4],
            )
        }.filter {
            it.isNotEmpty()
        }
    }

    private fun getBannerChanges(
        bannerType: Class<out BannerWithSystemFields>,
        brief: UacAdGroupBrief,
        sitelinkSetId: Long?,
        retargetingCondition: RetargetingCondition?,
        bizLandingUrl: String?,
    ): ModelChanges<BannerWithSystemFields> {
        when (bannerType) {
            MobileAppBanner::class.java -> {
                val primaryAction = getPrimaryActionByRetargetingCondition(retargetingCondition)
                return ModelChanges(0L, MobileAppBanner::class.java)
                    .processNotNull(brief.trackingUrl, MobileAppBanner.HREF)
                    .process(brief.impressionUrl, MobileAppBanner.IMPRESSION_URL)
                    .process(primaryAction, MobileAppBanner.PRIMARY_ACTION)
                    .process(brief.showTitleAndBody, MobileAppBanner.SHOW_TITLE_AND_BODY)
                    .castModel(BannerWithSystemFields::class.java)
            }
            TextBanner::class.java -> {
                val href = bizLandingUrl ?: getHrefWithTrackingParams(brief.url, brief.trackingParams)
                return ModelChanges(0L, TextBanner::class.java)
                    .processNotNull(href, TextBanner.HREF)
                    .processNotNull(brief.permalinkId, TextBanner.PERMALINK_ID)
                    .processNotNull(brief.phoneId, TextBanner.PHONE_ID)
                    .processNotNull(brief.zenPublisherId, TextBanner.ZEN_PUBLISHER_ID)
                    .processNotNull(sitelinkSetId, TextBanner.SITELINKS_SET_ID)
                    .process(brief.showTitleAndBody, TextBanner.SHOW_TITLE_AND_BODY)
                    .castModel(BannerWithSystemFields::class.java)
            }
            CpmBanner::class.java -> {
                return ModelChanges(0L, CpmBanner::class.java)
                    .processNotNull(brief.url, CpmBanner.HREF)
                    .processNotNull(brief.permalinkId, CpmBanner.PERMALINK_ID)
                    .process(null, CpmBanner.LOGO_IMAGE_HASH)
                    .process(null, CpmBanner.BUTTON_ACTION)
                    .process(null, CpmBanner.BUTTON_HREF)
                    .process(null, CpmBanner.BUTTON_CAPTION)
                    .process(null, CpmBanner.TITLE)
                    .process(null, CpmBanner.TITLE_EXTENSION)
                    .process(null, CpmBanner.BODY)
                    .process(null, CpmBanner.PIXELS)
                    .process(null, CpmBanner.MEASURERS)
                    .process(brief.showTitleAndBody, CpmBanner.SHOW_TITLE_AND_BODY)
                    .castModel(BannerWithSystemFields::class.java)
            }
            DynamicBanner::class.java -> {
                return ModelChanges(0L, DynamicBanner::class.java)
                    .processNotNull(brief.url, DynamicBanner.HREF)
                    .processNotNull(sitelinkSetId, DynamicBanner.SITELINKS_SET_ID)
                    .castModel(BannerWithSystemFields::class.java)
            }
            ImageBanner::class.java -> {
                return ModelChanges(0L, ImageBanner::class.java)
                    .processNotNull(brief.trackingUrl, ImageBanner.HREF)
                    .castModel(BannerWithSystemFields::class.java)
            }
            /* Смарт баннеры никогда не обновляются, поэтому тут нет отдельной ветки для них */
            else -> {
                throw IllegalStateException("Unsupported type: $bannerType")
            }
        }
    }

    private fun getPrimaryActionByRetargetingCondition(
        retargetingCondition: RetargetingCondition?
    ): NewMobileContentPrimaryAction {
        val isMobileGoalsRetargeting = retargetingCondition
            ?.collectGoalsSafe()
            ?.all { Goal.computeType(it.id) != GoalType.LAL_SEGMENT }
            ?: false
        return if (isMobileGoalsRetargeting) {
            NewMobileContentPrimaryAction.OPEN
        } else {
            NewMobileContentPrimaryAction.GET
        }
    }

    private fun getDefaultMaxBannersInUacAdGroup(
        brief: UacAdGroupBrief,
    ): Int = when (brief.advType) {
        AdvType.TEXT -> {
            // Лимит не должен превышать тот что в валидации у операции добавления
            min(
                maxBannersInUacTextAdGroupProperty.getOrDefault(DEFAULT_MAX_BANNERS_IN_UAC_TEXT_ADGROUP),
                getLimitBannersInGroup(false, true, true)
            )
        }
        else -> {
            min(
                maxBannersInUacAdGroupProperty.getOrDefault(DEFAULT_MAX_BANNERS_IN_UAC_ADGROUP),
                getLimitBannersInGroup(false, true, false)
            )
        }
    }

    // каждому входному элементу contentsByTypeList соответствует по порядку один баннер в левой части ответа
    fun createBanners(
        container: UpdateAdsContainer,
        subCampaigns: Map<CampaignType, CommonCampaign>,
        banners: List<BannerWithSourceContents>,
        appInfo: AppInfo?,
        complexBidModifier: ComplexBidModifier?,
        retargetingCondition: RetargetingCondition?,
        adGroupIdAndAdsCount: AdGroupIdToBannersCnt?,
        adGroupType: AdGroupType
    ): Pair<List<BannerWithSystemFields>, Set<Long>> {
        val (operatorUid, client, brief, campaign) = container
        val maxBannersInUacGroup = getDefaultMaxBannersInUacAdGroup(brief)

        val returnBanners = mutableListOf<BannerWithSystemFields>()
        val returnAdgroups = mutableSetOf<Long>()
        var leftIndex = 0
        // первый чанк баннеров может влезть в имеющуюся группу
        if (adGroupIdAndAdsCount != null
            && maxBannersInUacGroup > adGroupIdAndAdsCount.bannersCnt
            && banners.isNotEmpty()
        ) {
            val rightIndex = minOf(maxBannersInUacGroup - adGroupIdAndAdsCount.bannersCnt, banners.size)
            val bannersChunk = banners.subList(leftIndex, rightIndex)

            val bannerCampaign = when (adGroupType) {
                AdGroupType.DYNAMIC -> subCampaigns[CampaignType.DYNAMIC] ?: campaign
                else -> campaign
            }
            val massResult = uacBannerService.createBanner(
                operatorUid,
                ClientId.fromLong(client.clientId),
                bannerCampaign,
                bannersChunk.banners(),
                adGroupIdAndAdsCount.adGroupId
            )
            if (massResult != null && massResult.validationResult.hasAnyErrors()) {
                for (error in massResult.validationResult.flattenErrors()) {
                    logger.error(
                        "Can't create a new {}banner for the ydb:campaign.id: {}, an error is: {}",
                        if (brief.isEcom == true) "ecom " else "", brief.campaignId, error.toString()
                    )
                }
            }

            saveBannersAndAdGroupAndUpdateLists(
                bannersChunk.contents(),
                bannersChunk.banners(),
                brief.campaignId,
                returnBanners,
                returnAdgroups,
                false
            )

            leftIndex = rightIndex
        }

        while (leftIndex < banners.size) {
            val rightIndex = leftIndex + minOf(maxBannersInUacGroup, banners.size - leftIndex)
            val bannersChunk = banners.subList(leftIndex, rightIndex)
            // Для группы важны факты наличия видеоконтента нужного типа или html5
            val cpmDirectContentType: DirectContentType? = getCpmDirectContentType(bannersChunk.contents())

            val massResult = uacAdGroupService.createMissingOrNonexistentGroups(
                operatorUid, client, brief, campaign,
                subCampaigns,
                cpmDirectContentType,
                appInfo,
                complexBidModifier,
                retargetingCondition,
                bannersChunk.banners(),
                uacKeywordJobService.getKeywordsInUacCampaign(brief),
                adGroupType
            )

            if (massResult.validationResult.hasAnyErrors()) {
                for (error in massResult.validationResult.flattenErrors()) {
                    logger.error(
                        "Can't create a new {}{} group with banners for the ydb:campaign.id: {}, an error is: {}",
                        if (brief.isEcom == true) "ecom " else "",
                        adGroupType.toString(), brief.campaignId, error.toString()
                    )
                }
            }

            saveBannersAndAdGroupAndUpdateLists(
                bannersChunk.contents(),
                bannersChunk.banners(),
                brief.campaignId,
                returnBanners,
                returnAdgroups,
                true
            )

            leftIndex = rightIndex
        }

        return Pair(returnBanners, returnAdgroups)
    }

    private fun toFilledBanners(
        contentsByTypeList: List<ContentsByType>,
        brief: UacAdGroupBrief,
        appInfo: AppInfo?,
        sitelinkSetId: Long?,
        bannerType: Class<out Banner>,
        retargetingCondition: RetargetingCondition?,
        bizLandingUrl: String?
    ) = if (PerformanceBannerMain::class.java == bannerType) {
        listOf(BannerWithSourceContents(PerformanceBannerMain(), ContentsByType.emptyContents()))
    } else {
        contentsByTypeList.map { contentsByType ->
            val banner = fillBanner(
                contentsByType,
                brief,
                appInfo,
                sitelinkSetId,
                bannerType,
                retargetingCondition,
                bizLandingUrl,
            )
            BannerWithSourceContents(banner, contentsByType)
        }
    }

    abstract fun saveBannersAndAdGroupAndUpdateLists(
        contentsByTypeChunk: List<ContentsByType>,
        banners: List<BannerWithSystemFields>,
        directCampaignId: String,
        returnBanners: MutableList<BannerWithSystemFields>,
        returnAdgroups: MutableSet<Long>,
        isNewGroup: Boolean
    )

    abstract fun touchCampaign(campaignId: String)

    abstract fun updateBriefSynced(campaignId: String, briefSynced: Boolean)

    private fun sendToModeration(
        operatorUid: Long,
        clientId: ClientId,
        banners: List<BannerWithContents>
    ): Boolean {
        val bannerIds = banners.map { it.banner.id }

        var isModerationSuccessful = true
        for (chunk in Lists.partition(bannerIds, MODERATE_BANNERS_CHUNK)) {
            val result = bannerModerateService.moderateBanners(clientId, operatorUid, chunk)
            if (result == null || result.validationResult.hasAnyErrors())
                logger.error(
                    "Can't send to moderation banner ids: {}, first error is: {}",
                    chunk, result.validationResult.flattenErrors()[0].toString()
                )
            isModerationSuccessful = false
        }

        // Обновляем статус у баннеров в ydb
        uacBannerJobService.updateStatusByDirectAdIds(
            bannerIds, if (isModerationSuccessful) DirectAdStatus.MODERATING else DirectAdStatus.ERROR_UNKNOWN
        )

        return isModerationSuccessful
    }

    abstract fun getAndCreateContents(
        clientId: ClientId,
        uacCampaignContents: List<UacYdbCampaignContent>,
    ): List<UacYdbDirectContent>

    protected open fun createAdGroupBriefByListings(
        campaignId: Long,
        catalogIds: List<Long>,
        adGroupIds: Collection<Long>
    ) {}

    protected open fun getAdGroupBriefByListings(
        campaignId: Long
    ): AdGroupBriefGrutModel? = null

    abstract fun updateAdGroupBriefCatalogsAndGroups(
        brief: UacAdGroupBrief,
        catalogIds: List<Long>?,
        adGroupIds: Set<Long>
    )

    abstract fun updateAdGroupBriefGroupIds(
        brief: UacAdGroupBrief,
        adGroupIds: Set<Long>
    )

    /**
     * Конвертируем appInfo.ageLimit в Age. Если такого нет - возвращаем следующее меньшее значение
     * Например, для ageLimit = 3 вернет AGE_0
     */
    private fun toDirectAgeLimit(appInfo: AppInfo?): Age? {
        return if (appInfo?.ageLimit == null) {
            null
        } else {
            val ageLimit = appInfo.ageLimit!!
            for (i in Age.values().size - 1 downTo 0) {
                if (ageLimit >= Age.values()[i].value.toInt()) {
                    return Age.values()[i]
                }
            }
            return null
        }
    }

    private fun groupContentsByType(
        uacDirectAdContents: Set<String>,
        campaignContentsByTypeAndId: Map<MediaType, Map<String, UacYdbCampaignContent>>,
        directContentByTypeAndId: Map<DirectContentType, Map<String, UacYdbDirectContent>>,
        videosAreNonSkippable: Boolean
    ): ContentsByType {
        val titleCampaignContent =
            campaignContentsByTypeAndId[MediaType.TITLE]?.let { findContentInContentMap(uacDirectAdContents, it) }
        val textCampaignContent =
            campaignContentsByTypeAndId[MediaType.TEXT]?.let { findContentInContentMap(uacDirectAdContents, it) }
        val imageCampaignContent =
            campaignContentsByTypeAndId[MediaType.IMAGE]?.let { findContentInContentMap(uacDirectAdContents, it) }
        val videoCampaignContent =
            campaignContentsByTypeAndId[MediaType.VIDEO]?.let { findContentInContentMap(uacDirectAdContents, it) }
        val html5CampaignContent =
            campaignContentsByTypeAndId[MediaType.HTML5]?.let { findContentInContentMap(uacDirectAdContents, it) }

        val html5VideoDirectContent =
            directContentByTypeAndId[DirectContentType.HTML5]?.let { findContentInContentMap(uacDirectAdContents, it) }
        val imageDirectContent =
            directContentByTypeAndId[DirectContentType.IMAGE]?.let { findContentInContentMap(uacDirectAdContents, it) }

        // Когда будет строгая валидация данных при входе, можно будет использовать для проверки флаг
        // videosAreNonSkippable. Но пока он противоречит действительным ассетам, то его не проверяем.
        val videoContentById = if (!directContentByTypeAndId[DirectContentType.NON_SKIPPABLE_VIDEO].isNullOrEmpty()) {
            directContentByTypeAndId[DirectContentType.NON_SKIPPABLE_VIDEO]
        } else {
            directContentByTypeAndId[DirectContentType.VIDEO]
        }
        val videoContent = videoContentById?.let { findContentInContentMap(uacDirectAdContents, it) }

        return ContentsByType(
            titleCampaignContent = titleCampaignContent,
            textCampaignContent = textCampaignContent,
            imageCampaignContent = imageCampaignContent,
            videoCampaignContent = videoCampaignContent,
            html5CampaignContent = html5CampaignContent,

            html5VideoDirectContent = html5VideoDirectContent,
            imageDirectContent = imageDirectContent,
            videoDirectContent = videoContent,
        )
    }

    private fun convertListingsToBanners(
        clientId: ClientId,
        catalogs: List<CatalogHrefWithBreadcrumbs>,
        brief: UacAdGroupBrief,
        sitelinkSetId: Long?
    ): List<BannerWithSourceContents> {
        return catalogs.map { catalog ->
            val title = catalog.title ?: catalog.breadcrumbs.last()
            val body = if (catalog.breadcrumbs.isNotEmpty()) {
                catalog.breadcrumbs.joinToString(separator = " / ")
            } else {
                catalog.title
            }
            val imageHash = catalog.imageUrl?.let { uploadImage(it, clientId) }

            val banner = TextBanner()
                .withHref(catalog.href)
                .withTitle(title)
                .withBody(body)
                .withImageHash(imageHash)
                .withPermalinkId(brief.permalinkId)
                .withPhoneId(brief.phoneId)
                .withSitelinksSetId(sitelinkSetId)
                .withZenPublisherId(brief.zenPublisherId)
                .withShowTitleAndBody(brief.showTitleAndBody)

            // Передаем пустой ContentsByType, так как для баннеров по листингам отсутствуют ассеты в Груте
            BannerWithSourceContents(banner, ContentsByType.emptyContents())
        }
    }

    private fun uploadImage(url: String, clientId: ClientId): String? {
        val operationResult = imageService.saveImageFromUrl(
            clientId,
            url,
            BannerImageType.BANNER_TEXT,
            BannerImageSource.UAC,
            null
        )

        if (!ValidationUtils.hasValidationIssues(operationResult)) {
            return operationResult.result.imageHash
        } else {
            logger.error("Error while uploading image ($url) for clientId = $clientId: ${operationResult.errors}")
        }

        return null
    }

    private fun getCpmDirectContentType(
        contentsByTypeList: List<ContentsByType>,
    ) : DirectContentType? {
        if (contentsByTypeList.any { content -> content.html5VideoDirectContent != null })
            return DirectContentType.HTML5
        return contentsByTypeList
            .filter { it.videoDirectContent != null }
            .map { it.videoDirectContent?.type }
            .firstOrNull()
    }
}

data class ContentsByType(
    val titleCampaignContent: UacYdbCampaignContent?,
    val textCampaignContent: UacYdbCampaignContent?,
    val imageCampaignContent: UacYdbCampaignContent?,
    val html5CampaignContent: UacYdbCampaignContent?,
    val videoCampaignContent: UacYdbCampaignContent?,

    val imageDirectContent: UacYdbDirectContent?,
    val html5VideoDirectContent: UacYdbDirectContent?,
    val videoDirectContent: UacYdbDirectContent?
) {
    val campaignContents = listOfNotNull(
        titleCampaignContent, textCampaignContent, imageCampaignContent,
        html5CampaignContent, videoCampaignContent
    )

    companion object {
        fun emptyContents() =
            ContentsByType(null, null, null, null, null, null, null, null)
    }
}

data class BannerWithSourceContents(val banner: BannerWithSystemFields, val contentsByType: ContentsByType)

private fun List<BannerWithSourceContents>.banners() = this.map { it.banner }

private fun List<BannerWithSourceContents>.contents() = this.map { it.contentsByType }

    private fun UacAdGroupBrief.isValidEcomBrief(
        subCampaigns: Map<CampaignType, CommonCampaign>,
        ecomUcNewBackendEnabled: Boolean
    ) = CommonUtils.nvl(this.isEcom, false)
        && (
            (subCampaigns.containsKey(CampaignType.DYNAMIC) && subCampaigns.containsKey(CampaignType.PERFORMANCE))
            || ecomUcNewBackendEnabled
        )

private fun AdvType.toAdGroupType() = when (this) {
    AdvType.MOBILE_CONTENT -> AdGroupType.MOBILE_CONTENT
    AdvType.TEXT -> AdGroupType.BASE
    AdvType.CPM_BANNER -> AdGroupType.CPM_BANNER
}

private fun AdvType.toBannerType() = when (this) {
    AdvType.MOBILE_CONTENT -> MobileAppBanner::class.java
    AdvType.TEXT -> TextBanner::class.java
    AdvType.CPM_BANNER -> CpmBanner::class.java
}
