package ru.yandex.direct.jobs.uac.service

import java.time.Duration
import java.time.LocalDateTime
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.uac.model.UacGrutPage
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService
import ru.yandex.direct.core.entity.uac.service.GrutUacDeleteImageContentService
import ru.yandex.grut.objects.proto.MediaType

@Service
class DeleteUnusedAssetsFromMDSService(
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutUacDeleteImageContentService: GrutUacDeleteImageContentService,
    private val grutUacContentService: GrutUacContentService,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(DeleteUnusedAssetsFromMDSService::class.java)
        private const val DELIMITER = "/"
        private const val LIMIT = 2000L
        private const val LIMIT_ITERATION = 500
    }

    fun execute(
        minLookupTime: LocalDateTime,
        maxLookupTime: LocalDateTime
    ): Int {
        var localMinLookupTime = minLookupTime
        var totalDeleted = 0
        var currentIteration = 0

        while (localMinLookupTime.isBefore(maxLookupTime)) {

            val totalDeleteLimit = getTotalDeleteLimit()
            val idleTime = getIdleTime()
            val localMaxLookupTime = minOf(maxLookupTime, localMinLookupTime.plusHours(1))

            logger.info("Iteration #$currentIteration has been initiated. Lookup period: " +
                "$localMinLookupTime - $localMaxLookupTime")

            totalDeleted += runDelete(
                localMinLookupTime,
                localMaxLookupTime,
                idleTime,
                totalDeleteLimit - totalDeleted)

            if (totalDeleted >= totalDeleteLimit) {
                logger.info("Aborting operation! Total delete limit has been reached for the current launch.")
                break
            }

            ppcPropertiesSupport.set(PpcPropertyNames.DELETE_UAC_ASSET_MIN_LOOKUP_TIME, localMaxLookupTime.toString())
            localMinLookupTime = localMaxLookupTime

            if (currentIteration++ >= getIterationLimit()) {
                logger.info("Aborting operation! Too many iterations! Operation stopped at $localMaxLookupTime")
                break
            }

            logger.info("Transitioning to idle state for $idleTime milliseconds")
            Thread.sleep(idleTime)
        }

        return totalDeleted
    }

    private fun runDelete(
        localMinLookupTime: LocalDateTime,
        localMaxLookupTime: LocalDateTime,
        idleTime: Long,
        currentDeleteLimit: Long
    ): Int {
        var totalDeleted = 0
        var assetIterator = 0
        var continuationToken: String? = null

        while (true) {
            val result = getAssetsByClient(
                UacYdbUtils.toEpochSecond(localMinLookupTime),
                UacYdbUtils.toEpochSecond(localMaxLookupTime), LIMIT, continuationToken)
            val assetsByClient = result.content
                .filter { it.accountId != null }
                .groupBy { it.accountId!! }

            logger.info("Obtaining assets for ${assetsByClient.size} clients. " +
                "Iteration #$assetIterator has been initiated. Lookup period: " +
                "$localMinLookupTime - $localMaxLookupTime")

            assetsByClient.forEach clientsAssetLoop@{ (clientId, assets) ->
                val assetIds = assets.map { it.id }.toSet()
                val deleteCandidates = try {
                    val clientCampaignAssetIds = grutUacCampaignService.getClientCampaignAssets(clientId, assetIds)
                    getAssetIdToAvatarId(assets, clientCampaignAssetIds)
                } catch (ex: Throwable) {
                    logger.error("Fatal error! Unable to confirm status for assets: $assetIds", ex)
                    emptyList()
                }

                deleteCandidates.forEach {
                    if (grutUacDeleteImageContentService.deleteUnusedAsset(it.first, it.second, idleTime) &&
                        ++totalDeleted >= currentDeleteLimit) {
                        return@clientsAssetLoop
                    }
                }
            }

            continuationToken = result.continuationToken
            if (result.content.size < LIMIT || totalDeleted >= currentDeleteLimit) {
                break
            } else if (assetIterator++ == LIMIT_ITERATION) {
                logger.error("Fatal error! Too many assets. Processing for $localMinLookupTime - " +
                    "$localMaxLookupTime has been stopped")
                break
            }
        }
        return totalDeleted
    }

    private fun getAssetIdToAvatarId(
        assets: List<UacYdbContent>,
        clientAssetIds: Set<String>
    ): List<Pair<String, String>> {
        return assets
            .filter { it.id !in clientAssetIds }
            .mapNotNull {
                val avatarId = grutUacDeleteImageContentService.uacThumbToAvatarId(it.thumb, DELIMITER)
                if (avatarId == null) {
                    logger.warn("Invalid uri (${it.thumb}) for asset ${it.id}")
                    null
                } else {
                    it.id to avatarId
                }
            }
    }

    private fun getAssetsByClient(
        minLookupInstant: Long,
        maxLookupInstant: Long,
        limit: Long,
        continuationToken: String? = null
    ): UacGrutPage<List<UacYdbContent>> {
        return try {
            grutUacContentService.getAssetsUpperHalfCreationTimeInterval(
                minLookupInstant * 1_000_000,
                maxLookupInstant * 1_000_000,
                MediaType.EMediaType.MT_IMAGE,
                continuationToken,
                limit)
        } catch (ex: Throwable) {
            logger.error("Fatal error! Unable to fetch assets for half-interval " +
                "[$minLookupInstant, $maxLookupInstant)", ex)
            UacGrutPage(emptyList(), null)
        }
    }

    private fun getIterationLimit(): Int = ppcPropertiesSupport
        .get(PpcPropertyNames.DELETE_UAC_ASSET_ITERATION_LIMIT, Duration.ofMinutes(1))
        .getOrDefault(500)

    private fun getIdleTime(): Long = ppcPropertiesSupport
        .get(PpcPropertyNames.DELETE_UAC_ASSET_IDLE_TIME, Duration.ofMinutes(1))
        .getOrDefault(1000)

    private fun getTotalDeleteLimit(): Long = ppcPropertiesSupport
        .get(PpcPropertyNames.DELETE_UAC_ASSET_TOTAL_DELETE_LIMIT, Duration.ofMinutes(1))
        .getOrDefault(Long.MAX_VALUE)
}
