package ru.yandex.direct.jobs.uac.service

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.ARCHIVED
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.DRAFT
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.ON_MODERATION
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.PAUSE_OK
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.PAUSE_WARN
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_OK
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_PROCESSING
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_WARN
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.allRun
import ru.yandex.direct.jobs.uac.repository.EcomCampaignsMonitoringRepository

/**
 * Сервис для сбора данных для метрик, отправляемых в Соломон.
 */
@Service
class EcomCampaignsMonitoringService @Autowired constructor(
    private val ecomCampaignsMonitoringRepository: EcomCampaignsMonitoringRepository,
    private val aggregatedStatusesViewService: AggregatedStatusesViewService
) {

    /**
     * Возвращает общее количество еком кампаний.
     */
    fun countEcomCampaigns(shard: Int): Int {
        return ecomCampaignsMonitoringRepository.countEcomCampaigns(shard)
    }

    /**
     * Возвращает количество еком кампаний, у которых по каким-то причинам не хватает
     * подкампаний (в норме должно быть две — смарты и ДО).
     */
    fun countEcomCampaignsWithoutAllSubcampaigns(shard: Int): Int {
        return ecomCampaignsMonitoringRepository.countEcomCampaignsWithoutAllSubcampaigns(shard)
    }

    /**
     * Возвращает количество еком кампаний, у которых имеются расхождения в статусах
     * между мастер-кампанией и подкампаниями (вероятный признак проблем в товарной кампании).
     */
    fun countEcomCampaignsWithStatusesDifference(shard: Int): Int {
        val masterToSubs = ecomCampaignsMonitoringRepository.getAllEcomCampaignsWithSubcampaigns(shard)
        val allCids = mutableSetOf<Long>()
        allCids.addAll(masterToSubs.keys)
        allCids.addAll(masterToSubs.values.flatten())

        val statusesByCids = aggregatedStatusesViewService.getCampaignStatusesByIds(shard, allCids)
            .mapValues { it.value.status }
            .filterValues { it.isPresent }
            .mapValues { it.value.get() }
        val toCidWithStatus: (Long) -> CidWithStatus = { cid -> CidWithStatus(cid, statusesByCids[cid]) }
        return masterToSubs
            .mapKeys{ toCidWithStatus(it.key) }
            .mapValues { it.value.map(toCidWithStatus) }
            .filter { !it.hasOkStatuses() }
            .count()
    }

    /**
     * Проверяет согласованность статусов мастер-кампании и подкампаний. Возвращает `true`, если статусы согалсованы.
     * В следующих ситуациях мы считаем статусы согласованными:
     * - Статусы отсутствуют у всех кампаний
     * - Хотя бы одна из кампаний находится в статусе `ON_MODERATION` (этот статус временный, а по двум кампаниям нельзя
     * сделать никаких выводов о том, какой именно статус правильный)
     * - Все кампании находятся в статусе `RUN`, `PAUSE` или `STOP` уровней `OK`, `WARN`, `PROCESSING`
     * - Все кампании находятся в статусе `DRAFT`
     * - Все кампании находятся в стутсе `ARCHIVED`
     */
    private fun Map.Entry<CidWithStatus, List<CidWithStatus>>.hasOkStatuses(): Boolean {
        val (master, subs) = this
        // Отсутствующие статусы у всех
        if (master.status == null && subs.all { it.status == null }) {
            return true
        }
        if (master.status == ON_MODERATION || subs.any { it.status == ON_MODERATION }) {
            // Модерация — особый случай. Если какие-то из кампаний на модерации, пока считаем, что всё ок
            return true
        }

        // Дальше проверяем согласованность статусов
        val checkAcceptable: (Set<GdSelfStatusEnum>) -> Boolean = { acceptableStatuses ->
            acceptableStatuses.contains(master.status) && subs.all { acceptableStatuses.contains(it.status) }
        }
        if (checkAcceptable(runStatuses())
            || checkAcceptable(pauseStatuses())
            || checkAcceptable(stopStatuses())
            || checkAcceptable(draftStatuses())
            || checkAcceptable(archivedStatuses())) {
            return true
        }
        return false
    }

    private fun runStatuses(): Set<GdSelfStatusEnum> = allRun()
    private fun pauseStatuses(): Set<GdSelfStatusEnum> = setOf(PAUSE_OK, PAUSE_WARN)
    private fun stopStatuses(): Set<GdSelfStatusEnum> = setOf(STOP_OK, STOP_WARN, STOP_PROCESSING)
    private fun draftStatuses(): Set<GdSelfStatusEnum> = setOf(DRAFT)
    private fun archivedStatuses(): Set<GdSelfStatusEnum> = setOf(ARCHIVED)

    private data class CidWithStatus(val cid: Long, val status: GdSelfStatusEnum?)
}
