package ru.yandex.direct.jobs.uac.service

import java.time.Instant
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.image.service.ImageService
import ru.yandex.direct.core.entity.landing.model.BizLanding
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService
import ru.yandex.direct.core.entity.uac.converter.toAdGroupBriefGrutModel
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentStatus
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAppInfoRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectContent
import ru.yandex.direct.core.entity.uac.service.EcomOfferCatalogsService
import ru.yandex.direct.core.entity.uac.service.EcomUcBannerService
import ru.yandex.direct.core.entity.uac.service.EcomUcCampaignService
import ru.yandex.direct.core.entity.uac.service.GrutUacBannerService
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.entity.uac.service.UacAdGroupService
import ru.yandex.direct.core.entity.uac.service.UacAppInfoService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.grut.api.AdGroupBriefGrutModel
import ru.yandex.direct.core.grut.api.BriefAdGroup
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.utils.fromJson
import ru.yandex.grut.objects.proto.AdGroup
import ru.yandex.grut.objects.proto.Banner.EBannerSource
import ru.yandex.grut.objects.proto.client.Schema

@Service
class GrutBannerCreateJobService(
    uacYdbAppInfoRepository: UacYdbAppInfoRepository,
    uacAppInfoService: UacAppInfoService,
    uacAdGroupService: UacAdGroupService,
    uacBannerService: UacBannerService,
    bannerModerateService: BannerModerateService,
    bannerService: BannerService,
    sitelinkSetService: SitelinkSetService,
    ucRetargetingConditionService: UcRetargetingConditionService,
    ppcPropertiesSupport: PpcPropertiesSupport,
    grutUacBannerJobService: GrutUacBannerJobService,
    grutUacAdGroupJobService: GrutUacAdGroupJobService,
    ecomUcCampaignService: EcomUcCampaignService,
    ecomUcBannerService: EcomUcBannerService,
    uacKeywordJobService: UacKeywordJobService,
    retargetingConditionService: RetargetingConditionService,
    featureService: FeatureService,
    ecomOfferCatalogsService: EcomOfferCatalogsService,
    imageService: ImageService,
    private val grutUacCampaignService: GrutUacCampaignService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutUacBannerService: GrutUacBannerService,
    private val grutApiService: GrutApiService,
) : BannerCreateJobService(
    uacYdbAppInfoRepository,
    uacAppInfoService,
    uacAdGroupService,
    uacBannerService,
    bannerModerateService,
    bannerService,
    sitelinkSetService,
    ucRetargetingConditionService,
    ppcPropertiesSupport,
    grutUacBannerJobService,
    grutUacAdGroupJobService,
    ecomUcCampaignService,
    ecomUcBannerService,
    uacKeywordJobService,
    retargetingConditionService,
    featureService,
    ecomOfferCatalogsService,
    imageService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutBannerCreateJobService::class.java)
    }

    override fun getAndCreateContents(
        clientId: ClientId,
        uacCampaignContents: List<UacYdbCampaignContent>
    ): List<UacYdbDirectContent> {
        return uacCampaignContents
            .mapNotNull { contentToDirectContent(it) }
    }

    override fun saveBannersAndAdGroupAndUpdateLists(
        contentsByTypeChunk: List<ContentsByType>,
        banners: List<BannerWithSystemFields>,
        directCampaignId: String,
        returnBanners: MutableList<BannerWithSystemFields>,
        returnAdgroups: MutableSet<Long>,
        isNewGroup: Boolean,
    ) {
        if (banners.isEmpty()) {
            return
        }
        val adGroupId = banners.find { it.adGroupId != null }?.adGroupId
        if (adGroupId == null) {
            logger.error("No pid found, can't create any banner in GRuT")
            return
        }

        grutTransactionProvider.runInRetryableTransaction(retries = 10, sleepBetweenTriesMs = 5000, codeForRevert = {
            // Пока просто пишем в лог, если будет часто стрелять - стоит по-честному архивировать (или удалять)
            // баннеры и группы в mysql
            logger.error("Saving data in GRuT failed, need sync campaign $directCampaignId")
        }) {
            if (isNewGroup) {
                grutApiService.briefAdGroupGrutApi.createOrUpdateBriefAdGroup(
                    BriefAdGroup(
                        id = adGroupId,
                        briefId = directCampaignId.toIdLong(),
                        status = AdGroup.TAdGroupSpec.EDirectAdGroupStatus.DAGS_CREATED
                    )
                )
            }

            val briefBanners = mutableListOf<BriefBanner>()
            contentsByTypeChunk.zip(banners)
                .forEach forEachBanner@{ (assets, banner) ->
                    if (banner.id == null) {
                        logger.error("Bid is absent, can't create banner in GRuT")
                        return@forEachBanner
                    }
                    briefBanners.add(
                        BriefBanner(
                            id = banner.id,
                            adGroupId = adGroupId,
                            briefId = directCampaignId.toIdLong(),
                            source = EBannerSource.BS_DIRECT,
                            assetIds = assets.campaignContents.map { it.contentId!!.toIdLong() },
                            assetLinksIds = assets.campaignContents.map { it.id.toIdLong() },
                        )
                    )
                }
            grutApiService.briefBannerGrutApi
                .createOrUpdateBriefBanners(briefBanners)
        }

        if (isNewGroup) {
            returnAdgroups.add(adGroupId)
        }
        returnBanners.addAll(banners.filter { it.id != null })
    }

    override fun touchCampaign(campaignId: String) {
        grutApiService.briefGrutApi.updateBrief(
            Schema.TCampaign.newBuilder().apply {
                metaBuilder.apply {
                    id = campaignId.toIdLong()
                }
                specBuilder.apply {
                    updateTime = Instant.now().epochSecond.toInt()
                }
            }.build(),
            setPaths = listOf("/spec/update_time"),
        )
    }

    override fun updateBriefSynced(campaignId: String, briefSynced: Boolean) {
        grutUacBannerService.updateBriefSynced(campaignId, briefSynced)
    }

    private fun contentToDirectContent(content: UacYdbCampaignContent): UacYdbDirectContent? {
        val asset = content.asset ?: return null
        if (asset.spec.hasImage()) {
            val assetMeta: Map<String, Any?> = fromJson(asset.spec.image.mdsInfo.meta)
            return UacYdbDirectContent(
                id = content.id,
                type = DirectContentType.IMAGE,
                directImageHash = assetMeta["direct_image_hash"].toString(),
                status = DirectContentStatus.CREATED,
                directVideoId = null,
                directHtml5Id = null,
            )
        }
        if (asset.spec.hasVideo()) {
            val assetMeta: Map<String, Any?> = fromJson(asset.spec.video.mdsInfo.meta)
            val type = if (assetMeta["creative_type"]?.toString()?.lowercase() == "non_skippable_cpm")
                DirectContentType.NON_SKIPPABLE_VIDEO
            else DirectContentType.VIDEO
            return UacYdbDirectContent(
                id = content.id,
                type = type,
                directImageHash = null,
                status = DirectContentStatus.CREATED,
                directVideoId = assetMeta["creative_id"]?.toString()?.toLongOrNull(),
                directHtml5Id = null,
            )
        }
        if (asset.spec.hasHtml5()) {
            val assetMeta: Map<String, Any?> = fromJson(asset.spec.html5.mdsInfo.meta)
            return UacYdbDirectContent(
                id = content.id,
                type = DirectContentType.HTML5,
                directImageHash = null,
                status = DirectContentStatus.CREATED,
                directVideoId = assetMeta["creative_id"]?.toString()?.toLongOrNull(),
                directHtml5Id = assetMeta["creative_id"]?.toString()?.toLongOrNull(),
            )
        }
        logger.info("Unsupported asset ${asset.meta.id}")
        return null
    }

    override fun getBizLandingUrlOrNull(brief: UacAdGroupBrief): String? {
        val bizLandingId: Long = brief.bizLandingId ?: return null
        val bizLanding: BizLanding? = grutApiService.bizLandingGrutApi.getBizLanding(bizLandingId)
        if (bizLanding == null) {
            logger.warn("Can't find BizLanding by ID $bizLandingId")
            return null
        }
        return bizLanding.url
    }

    override fun createAdGroupBriefByListings(
        campaignId: Long,
        catalogIds: List<Long>,
        adGroupIds: Collection<Long>
    ) {
        val adGroupBrief = AdGroupBriefGrutModel(
            campaignId = campaignId,
            catalogIds = catalogIds,
            adGroupIds = adGroupIds.toList(),
            isCpmBrief = false,
        )
        grutApiService.adGroupBriefGrutApi.createAdGroupBriefs(listOf(adGroupBrief))
    }

    override fun getAdGroupBriefByListings(campaignId: Long): AdGroupBriefGrutModel? {
        return grutApiService.adGroupBriefGrutApi.selectAdGroupBriefsByCampaignId(campaignId)
            .firstOrNull { it.catalogIds?.isNotEmpty() ?: false }
    }

    override fun updateAdGroupBriefCatalogsAndGroups(
        brief: UacAdGroupBrief,
        catalogIds: List<Long>?,
        adGroupIds: Set<Long>
    ) {
        val adGroupBriefToUpdate = brief.toAdGroupBriefGrutModel().copy(
            catalogIds = catalogIds,
            adGroupIds = adGroupIds.toList(),
        )
        grutApiService.adGroupBriefGrutApi.updateCatalogsAndGroups(adGroupBriefToUpdate)
    }

    override fun updateAdGroupBriefGroupIds(
        brief: UacAdGroupBrief,
        adGroupIds: Set<Long>
    ) {
        val adGroupBriefToUpdate = brief.toAdGroupBriefGrutModel().copy(
            adGroupIds = adGroupIds.toList(),
        )
        grutApiService.adGroupBriefGrutApi.updateAdGroupIds(adGroupBriefToUpdate)
    }
}
