package ru.yandex.direct.jobs.uac.service


import com.fasterxml.jackson.databind.ObjectMapper
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.banner.model.ImageType
import ru.yandex.direct.core.entity.image.container.BannerImageType
import ru.yandex.direct.core.entity.image.model.BannerImageSource
import ru.yandex.direct.core.entity.image.model.ImageMdsMeta
import ru.yandex.direct.core.entity.image.model.ImageSizeMeta
import ru.yandex.direct.core.entity.image.service.ImageConstants.SUPPORTED_FORMATS_BY_IMAGE_TYPE
import ru.yandex.direct.core.entity.image.service.ImageService
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentStatus
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentType
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectContent
import ru.yandex.direct.core.validation.ValidationUtils
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.KtModelChanges

@Service
class ImageContentCreateJobService(
    private val uacYdbContentRepository: UacYdbContentRepository,
    private val imageService: ImageService,
    private val uacYdbDirectContentRepository: UacYdbDirectContentRepository,
) {

    fun createUacImageContents(
        clientId: ClientId,
        campaignContents: List<UacYdbCampaignContent>,
    ) {
        val contentIds = campaignContents
            .mapNotNull { it.contentId }
            .toSet()

        val contentIdToUacContent = uacYdbContentRepository.getContents(contentIds)
            .associateBy { it.id }

        val uacContentModelChanges = mutableListOf<KtModelChanges<String, UacYdbContent>>()

        val uacDirectContents = campaignContents.map { uacCampaignContent ->
            val uacContent = contentIdToUacContent[uacCampaignContent.contentId]
                ?: throw IllegalStateException("Uac content not found by content id: $uacCampaignContent.contentId")

            getDirectContentWithCollectContentChanges(clientId, uacCampaignContent, uacContent, uacContentModelChanges)
        }

        if (uacContentModelChanges.isNotEmpty()) {
            uacYdbContentRepository.updateMetaAndImageHash(uacContentModelChanges)
        }

        uacYdbDirectContentRepository.addDirectContent(uacDirectContents)
    }

    /**
     * Возвращает DirectContent картинки для добавления в базу и собирает необходимые изменения для таблицы content
     *
     * @param clientId id клиента
     * @param uacCampaignContent данные по баннеру
     * @param uacContent контент картинки
     * @param uacContentModelChanges список изменений контента, для таблицы ydb:content
     */
    private fun getDirectContentWithCollectContentChanges(
            clientId: ClientId,
            uacCampaignContent: UacYdbCampaignContent,
            uacContent: UacYdbContent,
            uacContentModelChanges: MutableList<KtModelChanges<String, UacYdbContent>>,
    ): UacYdbDirectContent {
        var imageHash: String? = null
        var status = DirectContentStatus.CREATED

        // Hash картинки в ydb:content всегда должен присутствовать, но в Python версии он проверяется на
        // наличие и в случае отсутствия - создается. Пока оставили эту часть, но возможно она лишняя
        if (uacContent.meta.containsKey("direct_image_hash")) {
            imageHash = uacContent.directImageHash
        } else {
            val operationResult = imageService.saveImageFromUrl(
                clientId,
                uacContent.thumb,
                BannerImageType.BANNER_TEXT,
                BannerImageSource.UAC, null)
            if (ValidationUtils.hasValidationIssues(operationResult)) {
                status = DirectContentStatus.ERROR_UNKNOWN
            } else {
                imageHash = operationResult.result.imageHash
            }

            val newMeta: Map<String, Any?>
            if (imageHash != null) {
                newMeta = uacContent.meta.toMutableMap()

                val sizeToImageSizeMeta =
                    getSizeToImageSizeMeta(operationResult.result.mdsMeta, operationResult.result.imageType)
                newMeta["direct_mds_meta"] = ObjectMapper().writeValueAsString(sizeToImageSizeMeta)
                newMeta["direct_image_hash"] = imageHash
            } else {
                newMeta = uacContent.meta
            }

            // Добавляем в список для последующего обновления конетне в ydb:content
            val modelChanges: KtModelChanges<String, UacYdbContent> = KtModelChanges(uacContent.id)
            modelChanges.process(UacYdbContent::directImageHash, imageHash)
            modelChanges.process(UacYdbContent::meta, newMeta)

            uacContentModelChanges.add(modelChanges)
        }

        return UacYdbDirectContent(
            id = uacCampaignContent.id,
            status = status,
            type = DirectContentType.IMAGE,
            directImageHash = imageHash,
            directVideoId = null,
            directHtml5Id = null,
        )
    }

    private fun getSizeToImageSizeMeta(
        imageMdsMeta: ImageMdsMeta,
        imageType: ImageType,
    ): Map<String, ImageSizeMeta> {
        return imageMdsMeta.sizes
            .filterKeys { o -> SUPPORTED_FORMATS_BY_IMAGE_TYPE[imageType]!!.contains(o) }
    }
}
