package ru.yandex.direct.jobs.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService
import ru.yandex.direct.core.entity.uac.converter.getProtoEnumValueName
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.jobs.uac.converter.DirectAdGroupStatusConverter.toUacDirectAdGroupStatus
import ru.yandex.grut.objects.proto.Banner

abstract class UacAdGroupJobService(
    private val adGroupService: AdGroupService
) {
    abstract fun getAdGroupsByUacCampaignId(uacCampaignId: String, directCampaignId: Long): List<UacYdbDirectAdGroup>

    abstract fun getAdGroupsIdWithBannersCnt(adGroupIds: Collection<String>): List<AdGroupIdToBannersCnt>

    fun getAdGroupIdsWithMinimumUacAdsByTypes(
        clientId: ClientId,
        adGroupIds: Collection<String>
    ): Map<AdGroupType, AdGroupIdToBannersCnt?> {
        val adGroupsToBannersCntList = getAdGroupsIdWithBannersCnt(adGroupIds)
        val directAdGroupIds = adGroupsToBannersCntList.map { it.adGroupId }
        val directAdGroups = adGroupService.getSimpleAdGroups(clientId, directAdGroupIds)

        // Получение из adGroupsCounts группы с минимальным количеством баннеров по подмножеству идентификаторов групп
        val groupWithMinimumAds: (List<Long>) -> AdGroupIdToBannersCnt? = { groupIds ->
            adGroupsToBannersCntList
                .filter { groupIds.contains(it.adGroupId) }
                .minByOrNull { it.bannersCnt }

        }
        return directAdGroups
            .values
            .groupBy { it.type }
            .mapValues { it.value.map(AdGroupSimple::getId) }
            .mapValues { groupWithMinimumAds(it.value) }
    }
}

@Service
class YdbUacAdGroupJobService(
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    adGroupService: AdGroupService
) : UacAdGroupJobService(adGroupService) {
    override fun getAdGroupsByUacCampaignId(uacCampaignId: String, directCampaignId: Long): List<UacYdbDirectAdGroup> {
        return uacYdbDirectAdGroupRepository.getDirectAdGroupsByCampaignId(uacCampaignId)
    }

    override fun getAdGroupsIdWithBannersCnt(adGroupIds: Collection<String>): List<AdGroupIdToBannersCnt> {
        if (adGroupIds.isEmpty()) {
            return emptyList()
        }

        val uacAdGroupIdToDirectId = uacYdbDirectAdGroupRepository.getDirectAdGroupIdByUacAdGroupId(adGroupIds)
        val uacAdGroupIdToAdsCount = uacYdbDirectAdRepository.getCountAdsByAdGroupIds(uacAdGroupIdToDirectId.keys)

        // в [uacAdGroupIdToAdsCount] может не быть adGroupId если у нее все баннеры удаленные
        return uacAdGroupIdToDirectId.keys
            .map {
                AdGroupIdToBannersCnt(
                    uacAdGroupIdToDirectId[it]!!,
                    uacAdGroupIdToAdsCount.getOrDefault(it, 0),
                )
            }
    }
}

@Service
class GrutUacAdGroupJobService(
    adGroupService: AdGroupService,
    private val grutApiService: GrutApiService,
) : UacAdGroupJobService(adGroupService) {
    override fun getAdGroupsByUacCampaignId(uacCampaignId: String, directCampaignId: Long): List<UacYdbDirectAdGroup> {
        return grutApiService.briefAdGroupGrutApi.selectAdGroups(
            filter = "[/meta/campaign_id] = $uacCampaignId"
        ).map {
            UacYdbDirectAdGroup(
                id = it.meta.id.toString(),
                directCampaignId = directCampaignId.toString(),
                directAdGroupId = it.meta.id.toLong(),
                status = it.spec.directAdGroupStatus.toUacDirectAdGroupStatus()!!
            )
        }.toList()
    }

    override fun getAdGroupsIdWithBannersCnt(adGroupIds: Collection<String>): List<AdGroupIdToBannersCnt> {
        if (adGroupIds.isEmpty()) {
            return listOf()
        }

        val adGroupToAdsCount = grutApiService.briefBannerGrutApi.selectBanners(
            filter = "[/meta/ad_group_id] IN (${adGroupIds.joinToString { it }})" +
                " AND [/spec/status] != \"${getProtoEnumValueName(Banner.TBannerSpec.EBannerStatus.BSS_DELETED)}\"",
            attributeSelector = listOf("/meta/ad_group_id")
        )
            .map { it.meta.adGroupId }
            .groupingBy { it }
            .eachCount()

        return adGroupIds
            .map { it.toIdLong() }
            .map {
                AdGroupIdToBannersCnt(
                    it,
                    adGroupToAdsCount.getOrDefault(it, 0)
                )
            }
    }
}

data class AdGroupIdToBannersCnt(
    val adGroupId: Long,
    val bannersCnt: Int,
)
