package ru.yandex.direct.jobs.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.core.entity.banner.model.BannerMeasurer
import ru.yandex.direct.core.entity.banner.model.BannerMeasurerSystem
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.ButtonAction
import ru.yandex.direct.core.entity.banner.model.CpmBanner
import ru.yandex.direct.core.entity.banner.service.BannersUpdateOperationFactory
import ru.yandex.direct.core.entity.banner.service.DatabaseMode
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCpmAsset
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.grut.GrutUtils.Companion.getDatabaseMode
import ru.yandex.direct.core.grut.api.BriefBanner
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.jobs.uac.converter.DirectAdStatusConverter.toEBannerStatus
import ru.yandex.direct.jobs.uac.model.UacBanner
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.mapToSet
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqSelectObjects
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TBanner

abstract class UacBannerJobService(
    private val bannersUpdateOperationFactory: BannersUpdateOperationFactory,
    private val featureService: FeatureService,
) {

    abstract fun updateStatusByDirectAdIds(
        bannerIds: Collection<Long>,
        directAdStatus: DirectAdStatus,
    )

    abstract fun getNotDeletedDirectAdsByCampaignContents(
        campaignContents: List<UacYdbCampaignContent>,
        campaignId: String,
    ): List<UacBanner>

    abstract fun getContentIdByDirectBid(
        brief: UacAdGroupBrief,
        bannerIds: List<Long>,
        assetLinkIdsByBid: Map<Long, Set<String>>,
    ) : Map<Long, String?>

    fun updateBanners(
        clientId: ClientId,
        operatorUid: Long,
        bannerIds: List<Long>,
        changesTemplate: ModelChanges<BannerWithSystemFields>,
        brief: UacAdGroupBrief,
        assetLinkIdsByBid: Map<Long, Set<String>>,
    ): MassResult<Long> {

        val contentIdByDirectId =
            if (brief.advType == AdvType.CPM_BANNER)
                getContentIdByDirectBid(brief, bannerIds, assetLinkIdsByBid)
            else emptyMap()

        val changes = bannerIds
            .map {
                val contentId = contentIdByDirectId[it]
                val cpmAssets = brief.cpmAssets
                copyChanges(
                    it,
                    changesTemplate,
                    contentId?.let { e -> cpmAssets?.get(e) },
                    brief
                )
            }

        val databaseMode: DatabaseMode = featureService.getDatabaseMode(clientId)
        val operation = bannersUpdateOperationFactory.createFullUpdateOperation(
            changes, operatorUid, clientId, databaseMode
        )

        return operation.prepareAndApply()
    }

    private fun copyChanges(
        bannerId: Long,
        changesTemplate: ModelChanges<BannerWithSystemFields>,
        uacCpmAssets: UacCpmAsset?,
        brief: UacAdGroupBrief,
    ): ModelChanges<BannerWithSystemFields> {
        val changes = ModelChanges(bannerId, changesTemplate.modelType as Class<BannerWithSystemFields>)

        changesTemplate.changedPropsNames.forEach {

            val value = if (uacCpmAssets != null) {
                when (it) {
                    CpmBanner.LOGO_IMAGE_HASH -> uacCpmAssets.logoImageHash
                    CpmBanner.BUTTON_ACTION -> uacCpmAssets.button?.let { cpmAssetButton ->
                        ButtonAction.valueOf(cpmAssetButton.action.name)
                    }
                    CpmBanner.BUTTON_CAPTION -> uacCpmAssets.button?.customText?.let {
                        //должен быть null, когда приходит от пользователя согласно BannerWithButtonValidatorProvider
                        customText -> customText.ifEmpty { null }
                    }
                    CpmBanner.BUTTON_HREF -> uacCpmAssets.button?.href
                    CpmBanner.TITLE -> uacCpmAssets.title
                    CpmBanner.TITLE_EXTENSION -> uacCpmAssets.titleExtension
                    CpmBanner.BODY -> uacCpmAssets.body
                    CpmBanner.PIXELS -> uacCpmAssets.pixels
                    CpmBanner.HREF -> uacCpmAssets.bannerHref ?: brief.url
                    CpmBanner.MEASURERS -> uacCpmAssets.measurers?.map { uacMeasurer ->
                        BannerMeasurer().withBannerMeasurerSystem(BannerMeasurerSystem.valueOf(uacMeasurer.measurerType.name))
                            .withParams(uacMeasurer.params)
                    }
                    else -> changesTemplate.getChangedProp(it)
                }
            } else {
                changesTemplate.getChangedProp(it)
            }

            changes.process(
                value,
                it as ModelProperty<BannerWithSystemFields, Any>
            )
        }
        return changes
    }
}

@Service
class YdbUacBannerJobService(
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacBannerService: UacBannerService,
    bannersUpdateOperationFactory: BannersUpdateOperationFactory,
    featureService: FeatureService,
) : UacBannerJobService(bannersUpdateOperationFactory, featureService) {

    override fun updateStatusByDirectAdIds(
        bannerIds: Collection<Long>,
        directAdStatus: DirectAdStatus
    ) {
        uacYdbDirectAdRepository.updateStatusByDirectAdIds(bannerIds, directAdStatus)
    }

    override fun getNotDeletedDirectAdsByCampaignContents(
        campaignContents: List<UacYdbCampaignContent>,
        campaignId: String
    ): List<UacBanner> {
        val campaignContentIdsByType: Map<MediaType, List<String>> = getAssetIdsByType(campaignContents)

        val creatives: MutableSet<String> = mutableSetOf()
        creatives.addAll(campaignContentIdsByType.getOrDefault(MediaType.VIDEO, setOf()))
        creatives.addAll(campaignContentIdsByType.getOrDefault(MediaType.HTML5, setOf()))

        return getNotDeletedDirectAdsByContentIds(
            campaignContentIdsByType.getOrDefault(MediaType.TITLE, setOf()).toCollection(mutableSetOf()),
            campaignContentIdsByType.getOrDefault(MediaType.TEXT, setOf()).toCollection(mutableSetOf()),
            campaignContentIdsByType.getOrDefault(MediaType.IMAGE, setOf()).toCollection(mutableSetOf()),
            creatives,
            campaignId,
        )
    }

    override fun getContentIdByDirectBid(
        brief: UacAdGroupBrief,
        bannerIds: List<Long>,
        assetLinkIdsByBid: Map<Long, Set<String>>
    ): Map<Long, String?> {
        val uacYdbDirectAdsByDirectId = uacYdbDirectAdRepository.getByDirectAdId(bannerIds)
            .associateBy { it.directAdId!! }
        return bannerIds.associateWith {
            uacYdbDirectAdsByDirectId[it]?.directVideoContentId
                ?: uacYdbDirectAdsByDirectId[it]?.directHtml5ContentId
        }
    }

    private fun getAssetIdsByType(campaignContents: List<UacYdbCampaignContent>): Map<MediaType, List<String>> {
        return campaignContents
            .filter { it.type != null }
            .groupBy({ it.type!! }) { it.id }
    }

    fun getNotDeletedDirectAdsByContentIds(
        titleIds: Set<String>,
        textIds: Set<String>,
        imageIds: Set<String>,
        creativeIds: Set<String>,
        campaignId: String,
    ): List<UacBanner> {
        val adGroups = uacYdbDirectAdGroupRepository.getDirectAdGroupsByCampaignId(campaignId)

        val uacBanners = uacBannerService.getDirectAdsByContentIds(adGroups.map { it.id })
            .filter {
                it.status != DirectAdStatus.DELETED
                    && (titleIds.contains(it.titleContentId)
                    || textIds.contains(it.textContentId)
                    || imageIds.contains(it.directImageContentId)
                    || creativeIds.contains(it.directVideoContentId)
                    || creativeIds.contains(it.directHtml5ContentId))
            }
            .toMutableList()

        val uacBannersWithoutAdGroup = uacYdbDirectAdRepository.getNotDeletedDirectAdsWithoutAdGroupsByContentIds(
            titleIds, textIds, imageIds, creativeIds,
        )
        uacBanners += uacBannersWithoutAdGroup
        val uacYdbOldDirectAdsToRemove =
            uacYdbDirectAdRepository.getOldNotDeletedDirectAdsByContentIds(imageIds, creativeIds)
        uacBanners += uacYdbOldDirectAdsToRemove

        return uacBanners
            .map {
                UacBanner(
                    uacBannerId = it.id,
                    bid = it.directAdId,
                    assetLinkIds = setOfNotNull(
                        it.textContentId, it.titleContentId, it.directVideoContentId,
                        it.directImageContentId, it.directHtml5ContentId
                    ),
                )
            }
            .toList()
    }
}

@Service
class GrutUacBannerJobService(
    val ppcPropertiesSupport: PpcPropertiesSupport,
    private val grut: GrutContext,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService,
    bannersUpdateOperationFactory: BannersUpdateOperationFactory,
    featureService: FeatureService,
) : UacBannerJobService(bannersUpdateOperationFactory, featureService) {

    companion object {
        private const val SELECT_BANNERS_MAX_PAGES = 1000
    }

    override fun getNotDeletedDirectAdsByCampaignContents(
        campaignContents: List<UacYdbCampaignContent>,
        campaignId: String,
    ): List<UacBanner> {
        var continuationToken: String? = null
        val banners = mutableListOf<TBanner>()
        val limit = 2000L
        var it = 0
        while (true) {
            val response = grutTransactionProvider.runRetryable(3) {
                grut.client.selectObjects(TReqSelectObjects.newBuilder().apply {
                    objectType = Schema.EObjectType.OT_BANNER
                    filter = "[/meta/campaign_id] = $campaignId and [/spec/status] != \"deleted\""
                    index = "banners_by_campaign"
                    addAllAttributeSelector(listOf("/meta", "/spec"))
                    this.limit = limit
                    continuationToken?.let { this.continuationToken = continuationToken }
                }.build())
            }
            continuationToken = response.continuationToken
            banners.addAll(response.payloadsList.map { TBanner.parseFrom(it.protobuf) })

            if (response.payloadsCount < limit) {
                break
            }
            if (it++ == SELECT_BANNERS_MAX_PAGES) {
                throw IllegalStateException("Too many banners in campaign $campaignId")
            }
        }

        val allNonDeletedBanners = banners.map {
            UacBanner(
                uacBannerId = it.meta.id.toString(),
                bid = it.meta.id,
                assetLinkIds = if (it.spec.assetLinkIdsCount == 0 && it.spec.assetIdsCount != 0) {
                    it.spec.assetIdsList.map { it.toIdString() }.toSet()
                } else {
                    it.spec.assetLinkIdsList.map { it.toIdString() }.toSet()
                },
                adGroupId = it.meta.adGroupId,
            )
        }
        val assetLinkIds = campaignContents.mapToSet { it.id }

        return allNonDeletedBanners.filter {
            it.assetLinkIds.any { id -> id in assetLinkIds }
        }
    }

    override fun getContentIdByDirectBid(
        brief: UacAdGroupBrief,
        bannerIds: List<Long>,
        assetLinkIdsByBid: Map<Long, Set<String>>
    ): Map<Long, String?> {
        brief.assetLinks.let { assetLinks ->
            return bannerIds.associateWith {bid ->
                assetLinks?.firstOrNull { it.id == assetLinkIdsByBid[bid]?.firstOrNull()}?.contentId
            }
        }
    }

    override fun updateStatusByDirectAdIds(
        bannerIds: Collection<Long>,
        directAdStatus: DirectAdStatus,
    ) {
        // todo #частичный_update
        grutTransactionProvider.runInRetryableTransaction(3) {
            val bannerById = grutApiService.briefBannerGrutApi.getBanners(bannerIds)
                .associateBy { it.meta.id }
            val briefBanners = bannerIds
                .map {
                    val oldBanner = bannerById[it]!!
                    BriefBanner(
                        id = it,
                        adGroupId = oldBanner.meta.adGroupId,
                        briefId = oldBanner.meta.campaignId,
                        source = oldBanner.meta.source,
                        assetIds = oldBanner.spec.assetIdsList,
                        assetLinksIds = oldBanner.spec.assetLinkIdsList,
                        status = directAdStatus.toEBannerStatus()

                    )
                }
            grutApiService.briefBannerGrutApi.createOrUpdateBriefBanners(briefBanners)
        }
    }
}
