package ru.yandex.direct.jobs.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.mailnotification.model.CampaignEvent.campaignStopped
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.model.TargetStatus
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.AudienceSegmentsService
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper

@Service
class UacCheckAudienceSegmentsJobService(
    private val grutUacCampaignService: GrutUacCampaignService,
    private val audienceSegmentsService: AudienceSegmentsService,
    private val userService: UserService,
    private val mailNotificationEventService: MailNotificationEventService,
    private val shardHelper: ShardHelper,
) {
    fun checkCampaignsAudienceSegments(campaignId: String, clientId: ClientId, operatorUid: Long): UacYdbCampaign? {
        val campaign = grutUacCampaignService.getCampaignById(campaignId) ?: return null
        val campaignStatusShow = grutUacCampaignService
            .getCampaignStatusShow(shardHelper.getShardByClientId(clientId), campaignId.toIdLong())

        if (campaign.audienceSegmentsSynchronized == true) {
            return campaign
        }
        if (campaignStatusShow != true) {
            return updateCampaignWithAudienceSynced(campaign)
        }
        val auditionSegmentIds = audienceSegmentsService
            .getAudienceSegmentIdsFromRetargetingCondition(campaign.retargetingCondition)
        if (auditionSegmentIds.isEmpty()) {
            return updateCampaignWithAudienceSynced(campaign)
        }

        val user = userService.getUser(operatorUid)
        val audienceSegments = audienceSegmentsService.getRetargetingSegmentsByIds(user!!.login, auditionSegmentIds)
        if (audienceSegmentsService.hasFailedSegments(audienceSegments)) {
            grutUacCampaignService.updateDirectCampaignStatusShow(user, clientId, campaign.id.toIdLong(), campaign, false)
            val updatedCampaign = campaign.updatedCampaign()
            grutUacCampaignService.updateAudienceSegmentsSyncedFields(updatedCampaign)
            notifyUser(campaignId, clientId, operatorUid)
            return updatedCampaign
        }
        if (!audienceSegmentsService.hasProcessedSegments(audienceSegments)) {
            return updateCampaignWithAudienceSynced(campaign)
        }
        return campaign
    }

    private fun notifyUser(campaignId: String, clientId: ClientId, operatorUid: Long) {
        val user = userService.getChiefUserByClientIdMap(listOf(clientId))[clientId]!!
        val campaignEvents = campaignStopped(operatorUid, user.chiefUid, campaignId.toIdLong())
        mailNotificationEventService.queueEvents(operatorUid, clientId, listOf(campaignEvents))
    }

    private fun updateCampaignWithAudienceSynced(campaign: UacYdbCampaign): UacYdbCampaign {
        val updatedCampaign = campaign.updatedCampaign()
        grutUacCampaignService.updateAudienceSegmentsSyncedFields(updatedCampaign)
        return updatedCampaign
    }

    private fun UacYdbCampaign.updatedCampaign(): UacYdbCampaign {
        return copy(
            audienceSegmentsSynchronized = true,
        )
    }
}
