package ru.yandex.direct.jobs.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgeneration.KeywordGenerationService
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.core.entity.retargeting.Constants
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils

@Service
class UacKeywordJobService(
    private val keywordGenerationService: KeywordGenerationService,
) {
    companion object {
        private const val KEYWORDS_LIMIT = 200;
    }
    /**
     * Аналог python функции get_keywords_for_campaign из campaign/helpers.py
     *
     * Возвращает список сгенерированных фраз
     */
    fun getKeywordsForCampaign(useRecommendedKeywords: Boolean, appInfo: AppInfo): List<Keyword> {

        if (!useRecommendedKeywords) {
            return emptyList()
        }

        // При отправке длинного текста фразы не возвращаются. Подробности в тикете:
        // https://st.yandex-team.ru/RMP-1615#6076b3535fc6fb6d3dd39300
        val title = reduceTextLength(appInfo.title, UacYdbUtils.TITLE_MAX_LENGTH)
        val text = reduceTextLength(appInfo.description, UacYdbUtils.TEXT_MAX_LENGTH)
        val url = appInfo.url

        val response = keywordGenerationService.generateKeywords(url, title, text, emptyList())
        val keywords = if (response.isSuccessful) response.result.take(KEYWORDS_LIMIT) else emptyList()

        keywords.forEach { it.autobudgetPriority = Constants.DEFAULT_AUTOBUDGET_PRIORITY }
        return keywords
    }

    private fun reduceTextLength(text: String?, maxLength: Int): String {
        if (text.isNullOrEmpty()) {
            return ""
        }

        val firstSentenceInd = listOf(text.indexOf(". "), text.indexOf("! "), text.indexOf("? "))
            .filter { it > 0 }
            .minOfOrNull { it }
        val firstSentence =
            if (firstSentenceInd != null && firstSentenceInd > 0) text.substring(0, firstSentenceInd + 1)
            else text

        // Python функция reduce_text_length для текста 'hello' c maxLength=3 вернет 'hell'. Решили что это бага
        return if (firstSentence.length > maxLength) firstSentence.substring(0, maxLength) else firstSentence
    }

    fun getKeywordsInUacCampaign(
        brief: UacAdGroupBrief
    ): List<Keyword>? = when (brief.advType) {
        AdvType.MOBILE_CONTENT, AdvType.TEXT -> {
            brief.keywords?.map { phrase ->
                Keyword()
                    .withPhrase(phrase)
                    .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
            }
        }
        AdvType.CPM_BANNER -> null
    }
}
