package ru.yandex.direct.jobs.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.canvas.client.model.video.AdditionResponse
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse
import ru.yandex.direct.core.entity.uac.UacCommonUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.service.UacCanvasService
import ru.yandex.direct.utils.JsonUtils

@Service
class UacVideoContentJobService(
    private val uacCanvasService: UacCanvasService
) {
    fun createCreativeForContent(clientId: Long, uacYdbContent: UacYdbContent): Map<String, Any?>? {
        val canvasVideoId = uacYdbContent.meta["id"] as String
        val canvasVideoResult = uacCanvasService.getVideoById(clientId, canvasVideoId)
        if (!canvasVideoResult.isSuccessful) {
            throw RuntimeException("Cannot get canvas video info: ${canvasVideoResult.validationResult}")
        }
        val canvasVideo = canvasVideoResult.result!!

        if (!allowedToCreateCreative(canvasVideo)) {
            return null
        }

        val createCreativeResult = uacCanvasService.createCreative(
            clientId, getPresetId(uacYdbContent), canvasVideoId
        )
        if (!createCreativeResult.isSuccessful) {
            throw RuntimeException("Cannot create creative: ${createCreativeResult.validationResult}")
        }

        return fillContentMeta(uacYdbContent.meta, canvasVideo, createCreativeResult.result!!)
    }

    private fun allowedToCreateCreative(canvasVideo: VideoUploadResponse): Boolean {
        return canvasVideo.status == VideoUploadResponse.FileStatus.READY || canvasVideo.createEarlyCreative
    }

    private fun getPresetId(uacYdbContent: UacYdbContent): Long {
        return uacYdbContent.meta["preset_id"].let {
            (it as? Long) ?: (it as Int).toLong()
        }
    }

    private fun fillContentMeta(
        previousMeta: Map<String, Any?>, canvasVideo: VideoUploadResponse, creative: AdditionResponse
    ): Map<String, Any?> {
        return previousMeta + mapOf(
            "creative_id" to creative.creativeId,
            "vast" to creative.vast,
            "status" to "ready",
            "id" to canvasVideo.id,
            "formats" to canvasVideo.formats
                .map { UacCommonUtils.mapFromJson(JsonUtils.toJson(it)) },
            "thumb" to UacCommonUtils.mapFromJson(JsonUtils.toJson(canvasVideo.thumbnail)),
        )
    }
}
