package ru.yandex.direct.jobs.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.image.service.ImageService
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.model.direct_ad.DirectAdStatus
import ru.yandex.direct.core.entity.uac.model.direct_ad_group.DirectAdGroupStatus
import ru.yandex.direct.core.entity.uac.model.direct_content.DirectContentStatus
import ru.yandex.direct.core.entity.uac.model.request.UacAdGroupBrief
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAppInfoRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdGroupRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectAdRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAd
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectAdGroup
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbDirectContent
import ru.yandex.direct.core.entity.uac.service.EcomOfferCatalogsService
import ru.yandex.direct.core.entity.uac.service.EcomUcBannerService
import ru.yandex.direct.core.entity.uac.service.EcomUcCampaignService
import ru.yandex.direct.core.entity.uac.service.UacAdGroupService
import ru.yandex.direct.core.entity.uac.service.UacAppInfoService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.jobs.uac.UpdateAdsJobUtil

@Service
class YdbBannerCreateJobService(
    uacYdbAppInfoRepository: UacYdbAppInfoRepository,
    uacAppInfoService: UacAppInfoService,
    uacAdGroupService: UacAdGroupService,
    uacBannerService: UacBannerService,
    bannerModerateService: BannerModerateService,
    bannerService: BannerService,
    sitelinkSetService: SitelinkSetService,
    ucRetargetingConditionService: UcRetargetingConditionService,
    ppcPropertiesSupport: PpcPropertiesSupport,
    ydbUacBannerJobService: YdbUacBannerJobService,
    ydbUacAdGroupJobService: YdbUacAdGroupJobService,
    ecomUcCampaignService: EcomUcCampaignService,
    ecomUcBannerService: EcomUcBannerService,
    uacKeywordJobService: UacKeywordJobService,
    retargetingConditionService: RetargetingConditionService,
    featureService: FeatureService,
    ecomOfferCatalogsService: EcomOfferCatalogsService,
    imageService: ImageService,
    private val uacYdbDirectContentRepository: UacYdbDirectContentRepository,
    private val uacYdbDirectAdGroupRepository: UacYdbDirectAdGroupRepository,
    private val uacYdbDirectAdRepository: UacYdbDirectAdRepository,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
    private val imageContentCreateJobService: ImageContentCreateJobService,
    private val videoContentCreateJobService: VideoContentCreateJobService,
    private val html5ContentCreateJobService: Html5ContentCreateJobService,
) : BannerCreateJobService(
    uacYdbAppInfoRepository,
    uacAppInfoService,
    uacAdGroupService,
    uacBannerService,
    bannerModerateService,
    bannerService,
    sitelinkSetService,
    ucRetargetingConditionService,
    ppcPropertiesSupport,
    ydbUacBannerJobService,
    ydbUacAdGroupJobService,
    ecomUcCampaignService,
    ecomUcBannerService,
    uacKeywordJobService,
    retargetingConditionService,
    featureService,
    ecomOfferCatalogsService,
    imageService,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
    }

    private fun saveBannerAndAdGroup(
        banner: BannerWithSystemFields,
        isAdGroupFullOrNotExist: Boolean,
        directCampaignId: String,
        titleContentId: String?,
        textContentId: String?,
        directImageContentId: String?,
        directVideoContentId: String?,
        directHtml5ContentId: String?,
    ) {
        val isBannerCreated = banner.id != null

        // Создаем группу в ydb:direct_ad_group
        if (isAdGroupFullOrNotExist && banner.adGroupId != null) {
            uacYdbDirectAdGroupRepository.saveDirectAdGroup(
                UacYdbDirectAdGroup(
                    id = UacYdbUtils.generateUniqueRandomId(),
                    directCampaignId = directCampaignId,
                    directAdGroupId = banner.adGroupId,
                    status = DirectAdGroupStatus.CREATED
                )
            )
        }

        val uacYdbDirectAdGroup = uacYdbDirectAdGroupRepository
            .getDirectAdGroupByDirectAdGroupId(banner.adGroupId)

        // создаем баннер в ydb.direct_ad
        uacYdbDirectAdRepository.saveDirectAd(
            UacYdbDirectAd(
                id = UacYdbUtils.generateUniqueRandomId(),
                status = if (isBannerCreated) DirectAdStatus.CREATED else DirectAdStatus.ERROR_UNKNOWN,
                titleContentId = titleContentId,
                textContentId = textContentId,
                directContentId = null,
                directAdGroupId = if (isBannerCreated) uacYdbDirectAdGroup?.id else null,
                directAdId = if (isBannerCreated) banner.id else null,
                directImageContentId = directImageContentId,
                directVideoContentId = directVideoContentId,
                directHtml5ContentId = directHtml5ContentId,
            )
        )
    }

    override fun saveBannersAndAdGroupAndUpdateLists(
        contentsByTypeChunk: List<ContentsByType>,
        banners: List<BannerWithSystemFields>,
        directCampaignId: String,
        returnBanners: MutableList<BannerWithSystemFields>,
        returnAdgroups: MutableSet<Long>,
        isNewGroup: Boolean,
    ) {
        var saveNewGroup = isNewGroup
        for (i in contentsByTypeChunk.indices) {
            val contentsByType = contentsByTypeChunk.get(i)
            val banner = banners.get(i)
            saveBannerAndAdGroup(
                banner,
                saveNewGroup,
                directCampaignId,
                contentsByType.titleCampaignContent?.id,
                contentsByType.textCampaignContent?.id,
                contentsByType.imageDirectContent?.id,
                (contentsByType.videoDirectContent ?: contentsByType.html5VideoDirectContent)?.id,
                contentsByType.html5VideoDirectContent?.id
            )
            returnBanners.add(banner)
            saveNewGroup = false // во всей пачке может быть только одна новая группа
            if (isNewGroup && banner.adGroupId != null) {
                returnAdgroups.add(banner.adGroupId)
            }
        }
    }

    override fun touchCampaign(campaignId: String) {
        uacYdbCampaignRepository.updateCampaignStatusAndTimes(campaignId, Status.MODERATING)
    }

    override fun updateBriefSynced(campaignId: String, briefSynced: Boolean) {
        uacYdbCampaignRepository.updateBriefSynced(campaignId, briefSynced)
    }

    /**
     * Отдаем все direct_content по campaign_content, c проверкой их существования в таблице
     * Если таких direct_content нет - создаем
     * Аналог python функции ensure_contents_exist в updaters.py
     */
    override fun getAndCreateContents(
        clientId: ClientId,
        uacCampaignContents: List<UacYdbCampaignContent>,
    ): List<UacYdbDirectContent> {
        val idToUacCampaignContent = UpdateAdsJobUtil
            .getIdToCampaignContentByTypes(
                uacCampaignContents,
                setOf(MediaType.IMAGE, MediaType.VIDEO, MediaType.HTML5)
            )

        val uacYdbDirectContents = uacYdbDirectContentRepository.getDirectContentByIds(idToUacCampaignContent.keys)
        val existedUacDirectContentIds = uacYdbDirectContents.map { it.id }.toSet()

        val notExistedUacDirectContentIds = idToUacCampaignContent.keys
            .filter { !existedUacDirectContentIds.contains(it) }

        if (notExistedUacDirectContentIds.isEmpty()) {
            return uacYdbDirectContents
                .filter { it.status == DirectContentStatus.CREATED }
        }

        val uacImageCampaignContents = mutableListOf<UacYdbCampaignContent>()
        val uacVideoCampaignContents = mutableListOf<UacYdbCampaignContent>()
        val uacHtml5CampaignContents = mutableListOf<UacYdbCampaignContent>()

        // Создаем в direct_content недостающие
        notExistedUacDirectContentIds.forEach {
            val uacCampaignContent = idToUacCampaignContent[it]

            when (uacCampaignContent!!.type) {
                MediaType.IMAGE -> {
                    uacImageCampaignContents.add(uacCampaignContent)
                }
                MediaType.VIDEO -> {
                    uacVideoCampaignContents.add(uacCampaignContent)
                }
                MediaType.HTML5 -> {
                    uacHtml5CampaignContents.add(uacCampaignContent)
                }
                else -> {
                    // ничего
                }
            }
        }

        if (uacImageCampaignContents.isNotEmpty()) {
            imageContentCreateJobService.createUacImageContents(clientId, uacImageCampaignContents)
        }
        if (uacVideoCampaignContents.isNotEmpty()) {
            videoContentCreateJobService.createUacVideoContents(uacVideoCampaignContents)
        }
        if (uacHtml5CampaignContents.isNotEmpty()) {
            html5ContentCreateJobService.createUacHtml5Contents(uacHtml5CampaignContents)
        }

        return uacYdbDirectContentRepository.getDirectContentByIds(idToUacCampaignContent.keys)
            .filter { it.status == DirectContentStatus.CREATED }
    }

    override fun updateAdGroupBriefCatalogsAndGroups(
        brief: UacAdGroupBrief,
        catalogIds: List<Long>?,
        adGroupIds: Set<Long>
    ) {
        logger.error("Fatal error! Ad group brief is not supported in Ydb. Campaign id ${brief.campaignId}")
    }

    override fun updateAdGroupBriefGroupIds(
        brief: UacAdGroupBrief,
        adGroupIds: Set<Long>
    ) {
        logger.error("Fatal error!! Ad group brief is not supported in Ydb. Campaign id ${brief.campaignId}")
    }
}
