package ru.yandex.direct.jobs.updatestorecontent;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.mobilecontent.service.UpdateMobileContentService;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоба для регулярного обновления данных по мобильному контенту в таблице ppc.mobile_content
 * Для обновления выбираются записи, которые не обновлялись более 24 часов.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        //PRIORITY: обновляет БД директа по yt табличке созданной в предыдущей джобе
        tags = {DIRECT_PRIORITY_1, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: обновляет БД директа по yt табличке созданной в предыдущей джобе
        tags = {DIRECT_PRIORITY_1, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 600, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateStoreContentJob extends DirectShardedJob {

    private static final int CHUNK_SIZE = 1000;
    private static final Duration MAX_AGE_AVAILABLE = Duration.ofDays(1);
    private static final Duration MAX_AGE_UNAVAILABLE = Duration.ofSeconds(500);
    private static final long MAX_TRIES = 100;

    private final MobileContentRepository mobileContentRepository;
    private final UpdateMobileContentService updateMobileContentService;

    @Autowired
    public UpdateStoreContentJob(
            MobileContentRepository mobileContentRepository,
            UpdateMobileContentService updateMobileContentService) {
        this.mobileContentRepository = mobileContentRepository;
        this.updateMobileContentService = updateMobileContentService;
    }

    @Override
    public void execute() {
        for (OsType osType : OsType.values()) {
            AtomicLong counter = new AtomicLong(0);
            try (Stream<MobileContent> stream1 = mobileContentRepository
                    .getMobileContentOlderThan(getShard(), MAX_AGE_AVAILABLE, MAX_TRIES, true);
                 Stream<MobileContent> stream2 = mobileContentRepository
                         .getMobileContentOlderThan(getShard(), MAX_AGE_UNAVAILABLE, MAX_TRIES, false)
            ) {
                StreamEx.of(stream1)
                        .append(stream2)
                        .filter(mc -> mc.getOsType().equals(osType))
                        .groupRuns((a, b) -> counter.incrementAndGet() % CHUNK_SIZE != 0)
                        .forEach(chunk -> updateMobileContentService.processChunk(getShard(), osType, chunk));
            }
        }
    }
}
