package ru.yandex.direct.jobs.urlmonitoring;

import java.net.IDN;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.urlmonitoring.service.UrlMonitoringService;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.URL_MONITORING_ON_LB_AND_YT_ENABLED;
import static ru.yandex.direct.jobs.urlmonitoring.UrlMonitoringUtils.URL_PATTERN;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Из YT таблицы markov.[//home/yabs/url_monitoring/states]
 * выгружает новые домены в ppcdict.domains_dict
 * и обновляет "потухшие" домены в ppc.bs_dead_domains
 * <p>
 * Выгрузка каждые 3 минуты
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 25),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_SANTAMA,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 9),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@Hourglass(cronExpression = "0 0/3 * * * ?", needSchedule = ProductionOnly.class)
@Hourglass(cronExpression = "0 30 1/3 * * ?", needSchedule = NonProductionEnvironment.class)
@ParametersAreNonnullByDefault
public class ImportUrlMonitoringStatesFromYtJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ImportUrlMonitoringStatesFromYtJob.class);
    private static final int MAX_DOMAIN_LENGTH = Tables.DOMAINS_DICT.DOMAIN.getDataType().length();
    private static final String MAXIMUM_DOMAIN_LENGTH_EXCEEDED_MESSAGE =
            "Domain exceeds maximum length of " + MAX_DOMAIN_LENGTH;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DomainService domainService;
    private final UrlMonitoringService urlMonitoringService;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public ImportUrlMonitoringStatesFromYtJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DomainService domainService,
            UrlMonitoringService urlMonitoringService,
            SolomonPushClient solomonPushClient) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.domainService = domainService;
        this.urlMonitoringService = urlMonitoringService;
        this.solomonPushClient = solomonPushClient;
    }

    @Override
    public void execute() {
        String jobEnabledPropValue = ppcPropertiesSupport.get(URL_MONITORING_ON_LB_AND_YT_ENABLED.getName());
        boolean jobEnabled = Boolean.parseBoolean(jobEnabledPropValue);
        if (!jobEnabled) {
            logger.info("Job is not enabled. {}={}", URL_MONITORING_ON_LB_AND_YT_ENABLED, jobEnabledPropValue);
            return;
        }
        Set<String> deadDomains = getValidDeadDomains();

        Map<String, Long> domainToIdMap = domainService.refreshDeadDomains(deadDomains);
        logger.info("{} domains in bs_dead_domains", domainToIdMap.size());
        sendMetrics(deadDomains.size());
    }

    Set<String> getValidDeadDomains() {
        List<String> urlList = urlMonitoringService.importDeadDomains();
        int invalidUrlCount = 0;
        Set<String> domains = new HashSet<>();
        for (String url : urlList) {
            try {
                Matcher matcher = URL_PATTERN.matcher(url);
                Preconditions.checkArgument(matcher.matches(), "URL [%s] doesn't match pattern [%s]", URL_PATTERN);
                String rowDomain = matcher.group(1);
                List<String> domainEncodingVariants = Arrays.asList(IDN.toASCII(rowDomain), IDN.toUnicode(rowDomain));
                for (String domainVariant : domainEncodingVariants) {
                    checkArgument(domainVariant.length() <= MAX_DOMAIN_LENGTH,
                            MAXIMUM_DOMAIN_LENGTH_EXCEEDED_MESSAGE, domainVariant);
                    domains.add(domainVariant);
                }
            } catch (IllegalArgumentException e) {
                logger.warn("invalid url {}", url, e);
                invalidUrlCount++;
            }
        }
        logger.info("{} urls, {} domains, {} invalid urls", urlList.size(), domains.size(), invalidUrlCount);
        return domains;
    }

    void sendMetrics(int bsFullStateDeadDomainsCount) {
        var registry = SolomonUtils.newPushRegistry("flow", UrlMonitoringUtils.URL_MONITORING_SOLOMON_LABEL,"type", "domains");
        registry.gaugeInt64("bs_full_state_dead_domains_count").set(bsFullStateDeadDomainsCount);
        try {
            solomonPushClient.sendMetrics(registry);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }
}
