package ru.yandex.direct.jobs.urlmonitoring;

import java.net.IDN;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;
import java.util.regex.Matcher;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Provider;

import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.entity.urlmonitoring.model.UrlMonitoringNotificationStatistics;
import ru.yandex.direct.core.entity.urlmonitoring.service.UrlMonitoringService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.jobs.urlmonitoring.model.UrlMonitoringEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonPushClientException;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.json.JSonSplitter;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.URL_MONITORING_ON_LB_AND_YT_ENABLED;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.URL_MONITORING_ON_LB_DRY_RUN;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.URL_MONITORING_LB_SYNC_CONSUMER_PROVIDER;
import static ru.yandex.direct.jobs.urlmonitoring.UrlMonitoringUtils.URL_PATTERN;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Джоба, рассылающая нотификации про недоступные домены
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 25),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_SANTAMA,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 9),
        needCheck = TestingOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY})
@Hourglass(cronExpression = "0 0/3 * * * ?", needSchedule = ProductionOnly.class)
@Hourglass(cronExpression = "0 0/3 * * * ?", needSchedule = TestingOnly.class)
@ParametersAreNonnullByDefault
public class UrlMonitoringEventReceivingJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UrlMonitoringEventReceivingJob.class);

    // dead - переход state из 1 или 4 в 0
    private static final Predicate<UrlMonitoringEvent> DIED_EVENT_PREDICATE =
            e -> (Integer.valueOf(1).equals(e.getOldStatus()) || Integer.valueOf(4).equals(e.getOldStatus()))
                    && Integer.valueOf(0).equals(e.getNewStatus());
    // alive - переход state из 0 в 1
    private static final Predicate<UrlMonitoringEvent> RESURRECTED_EVENT_PREDICATE =
            e -> Integer.valueOf(0).equals(e.getOldStatus()) && Integer.valueOf(1).equals(e.getNewStatus());

    static final String DEAD_DOMAINS_KEY = "dead";
    static final String ALIVE_DOMAINS_KEY = "alive";
    // отображение названий state-ов на предикаты
    private static final ImmutableMap<String, Predicate<UrlMonitoringEvent>> EVENT_FILTER_PREDICATE_BY_STATE =
            ImmutableMap.of(ALIVE_DOMAINS_KEY, RESURRECTED_EVENT_PREDICATE, DEAD_DOMAINS_KEY, DIED_EVENT_PREDICATE);

    private static final int MAX_EVENTS_TO_READ_AT_TIME = 150_000;
    private static final Duration DEFAULT_MAX_TIME_TO_READ_EVENTS = Duration.ofSeconds(20);

    private static final String DIRECT_SOURCE_TYPE = "direct";

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final UrlMonitoringService urlMonitoringService;
    private final SolomonPushClient solomonPushClient;
    private final Provider<SyncConsumer> syncConsumerProvider;

    private boolean dryRun;

    @Autowired
    public UrlMonitoringEventReceivingJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            @Qualifier(URL_MONITORING_LB_SYNC_CONSUMER_PROVIDER) Provider<SyncConsumer> syncConsumerProvider,
            UrlMonitoringService urlMonitoringService,
            SolomonPushClient solomonPushClient) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.syncConsumerProvider = syncConsumerProvider;
        this.urlMonitoringService = urlMonitoringService;
        this.solomonPushClient = solomonPushClient;
    }


    @Override
    public void execute() {
        String jobEnabledPropValue = ppcPropertiesSupport.get(URL_MONITORING_ON_LB_AND_YT_ENABLED.getName());
        boolean jobEnabled = Boolean.parseBoolean(jobEnabledPropValue);
        String dryRunPropValue = ppcPropertiesSupport.get(URL_MONITORING_ON_LB_DRY_RUN.getName());
        dryRun = Boolean.parseBoolean(dryRunPropValue);
        if (!jobEnabled && !dryRun) {
            logger.info("Job is not enabled. {}={}, {}={}", URL_MONITORING_ON_LB_AND_YT_ENABLED, jobEnabledPropValue,
                    URL_MONITORING_ON_LB_DRY_RUN, dryRun);
            return;
        }
        readAndApplyEvents();
    }

    private void readAndApplyEvents()  {
        int eventsCount = 0;

        Stopwatch stopwatchRead = Stopwatch.createStarted();
        try (SyncConsumer syncConsumer = this.syncConsumerProvider.get()) {
            List<Long> cookies = new ArrayList<>();
            List<UrlMonitoringEvent> eventList = new ArrayList<>();
            final Duration maxTimeToReadEvents = ppcPropertiesSupport
                    .find(PpcPropertyEnum.URL_MONITORING_EVENTS_READ_TIMEOUT_SECONDS.getName())
                    .filter(StringUtils::isNumeric)
                    .map(Long::valueOf)
                    .map(Duration::ofSeconds)
                    .orElse(DEFAULT_MAX_TIME_TO_READ_EVENTS);
            for (ConsumerReadResponse response = Interrupts.failingGet(syncConsumer::read);
                 response != null
                         && eventsCount < MAX_EVENTS_TO_READ_AT_TIME
                         && stopwatchRead.elapsed().compareTo(maxTimeToReadEvents) < 0;
                 response = Interrupts.failingGet(syncConsumer::read)) {
                cookies.add(response.getCookie());

                List<MessageBatch> batches = response.getBatches();
                for (MessageBatch batch : batches) {
                    List<MessageData> batchMessageData = batch.getMessageData();
                    for (MessageData messageData : batchMessageData) {
                        byte[] bytes = messageData.getDecompressedData();
                        String strData = new String(bytes, StandardCharsets.UTF_8);
                        String[] jsonMessages = JSonSplitter.splitJsons(strData);
                        for (String jsonMessage : jsonMessages) {
                            eventsCount++;
                            try {
                                UrlMonitoringEvent event = JsonUtils.fromJson(jsonMessage, UrlMonitoringEvent.class);
                                if (event.getSources().contains(DIRECT_SOURCE_TYPE)) {
                                    eventList.add(event);
                                }
                            } catch (RuntimeException e) {
                                logger.error("Cannot parse url monitoring event json " + jsonMessage, e);
                            }
                        }
                    }
                }
            }
            stopwatchRead.stop();
            logger.info("Total/direct events received: {}/{}; total reads made: {}; read timeout (sec): {}",
                    eventsCount, eventList.size(), cookies.size(), maxTimeToReadEvents.getSeconds());

            applyEvents(eventList);

            if (!cookies.isEmpty()) {
                commit(cookies, syncConsumer);
            }
        } catch (RuntimeException | TimeoutException ex) {
            throw new RuntimeException("Error while reading url monitoring events from LogBroker", ex);
        }
    }

    void applyEvents(Collection<UrlMonitoringEvent> eventList) {
        int invalidDomainsCount = 0;
        int bsDeadDomainsCount = 0;
        int bsAliveDomainsCount = 0;
        int bsUnknownStateDomainsCount = 0;
        // отображение пары протокол+домен на событие мониторинга
        Map<Pair<String, String>, UrlMonitoringEvent> mergedEvents = new HashMap<>();

        for (UrlMonitoringEvent event : eventList) {
            try {
                // ожидается, что url в событии мониторинга обязательно содержит протокол (http:// или https://)
                String url = event.getUrl();
                Matcher matcher = URL_PATTERN.matcher(url);
                checkArgument(matcher.matches(), "URL [%s] doesn't match pattern [%s]", URL_PATTERN);
                String protocol = url.contains("://") ? url.split("://")[0] : null;
                String domain = matcher.group(1);
                checkArgument(protocol != null && domain != null, "Failed to parse protocol and domain from url [%s]", domain);
                List<String> domainEncodingVariants = Arrays.asList(IDN.toASCII(domain), IDN.toUnicode(domain));
                for (String domainVariant : domainEncodingVariants) {
                    mergedEvents.merge(
                            Pair.of(protocol, domainVariant), event,
                            (e1, e2) -> StreamEx.of(e1, e2)
                                    // интересует последнее по времени событие по каждому домену
                                    .maxByLong(UrlMonitoringEvent::getUpdated).get());
                }
                if (DIED_EVENT_PREDICATE.test(event)) {
                    bsDeadDomainsCount++;
                } else if (RESURRECTED_EVENT_PREDICATE.test(event)) {
                    bsAliveDomainsCount++;
                } else {
                    bsUnknownStateDomainsCount++;
                }
            } catch (RuntimeException ex) {
                invalidDomainsCount++;
                logger.error("Parsing url monitoring event message {} failed with {} ", JsonUtils.toJson(event), ex);
            }
        }

        // хэш типа {"dead": [{"left": "http", "right": "dead.domain1.com"}], {"left": "https", "right": "dead.domain2.ru"]], "alive": [{"left":""http", "right":"immortal.com"}]}
        // получается разбиением пар протокол+домен из полученного списка событий по направлению перехода состояния
        // dead - переход state из 1 в 0
        // alive - переход state из 0 в 1
        Map<String, Set<Pair<String, String>>> domainsByState = EntryStream.of(EVENT_FILTER_PREDICATE_BY_STATE)
                .mapValues(eventStatePredicate -> EntryStream.of(mergedEvents).filterValues(eventStatePredicate).keys().toSet())
                .filterValues(domains -> !domains.isEmpty())
                .toMap();

        // отдельные нотификации об умерших и оживших доменах
        Map<String, UrlMonitoringNotificationStatistics> statistics =
                urlMonitoringService.notifyUsersOnDomainsStateChange(domainsByState, dryRun);

        sendDataToSolomon(eventList.size(), invalidDomainsCount, bsDeadDomainsCount,
                bsAliveDomainsCount, bsUnknownStateDomainsCount, domainsByState, statistics);
    }

    private void commit(List<Long> cookies, SyncConsumer syncConsumer) throws TimeoutException {
        try {
            syncConsumer.commit(cookies);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    /**
     * Засылает метрики в соломон
     * @param fetchedDomainsCount        сколько доменов получили от БК
     * @param invalidDomainsCount        сколько из полученных от БК доменов не прошло валидацию
     * @param bsDeadDomainsCount         количество доменов от БК (ставших недоступными) после валидации и фильтрации
     * @param bsAliveDomainsCount        количество доменов от БК (ставших доступными) после валидации и фильтрации
     * @param bsUnknownStateDomainsCount количество доменов от БК c неопознанным статусом после валидации и фильтрации
     * @param domainsByState
     */
    private void sendDataToSolomon(int fetchedDomainsCount, int invalidDomainsCount,
                                    int bsDeadDomainsCount, int bsAliveDomainsCount, int bsUnknownStateDomainsCount,
                                    Map<String, Set<Pair<String, String>>> domainsByState,
                                    Map<String, UrlMonitoringNotificationStatistics> statisticsByStateMap) {

        var registry = SolomonUtils.newPushRegistry("flow", UrlMonitoringUtils.URL_MONITORING_SOLOMON_LABEL);
        var domainsSubregistry = registry.subRegistry("type", "domains");
        domainsSubregistry.gaugeInt64("fetched_domains_count").set(fetchedDomainsCount);
        domainsSubregistry.gaugeInt64("invalid_domains_count").set(invalidDomainsCount);
        domainsSubregistry.gaugeInt64("bs_dead_domains_count").set(bsDeadDomainsCount);
        domainsSubregistry.gaugeInt64("bs_alive_domains_count").set(bsAliveDomainsCount);
        domainsSubregistry.gaugeInt64("bs_unknown_state_domains_count").set(bsUnknownStateDomainsCount);
        domainsSubregistry.gaugeInt64("result_dead_domains_count").set(domainsByState.getOrDefault(DEAD_DOMAINS_KEY, emptySet()).size());
        domainsSubregistry.gaugeInt64("result_alive_domains_count").set(domainsByState.getOrDefault(ALIVE_DOMAINS_KEY, emptySet()).size());

        for (String state : statisticsByStateMap.keySet()) {
            UrlMonitoringNotificationStatistics stat = statisticsByStateMap.get(state);
            String statePrefix = state + "_notifications";
            var notificationsSubregistry = registry.subRegistry("type", statePrefix);
            notificationsSubregistry.gaugeInt64("db_camp_domain_pairs").set(stat.getCampDomainPairs());
            notificationsSubregistry.gaugeInt64("notified_camps").set(stat.getNotifiedCamps());
            notificationsSubregistry.gaugeInt64("notified_domains").set(stat.getNotifiedDomains());
            notificationsSubregistry.gaugeInt64("notifications_count").set(stat.getNotificationsCount());
            notificationsSubregistry.gaugeInt64("failed_notifications").set(stat.getFailedNotifications());
            notificationsSubregistry.gaugeInt64("sms_count").set(stat.getSmsCount());
        }
        try {
            solomonPushClient.sendMetrics(registry);
        } catch (SolomonPushClientException e) {
            logger.error("Got exception on sending metrics", e);
        }
    }

}
