package ru.yandex.direct.jobs.util.juggler;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.ansiblejuggler.PlaybookBuilder;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.jobs.util.juggler.checkinfo.JobWorkingCheckInfo;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.check.DirectNumericCheck;
import ru.yandex.direct.juggler.check.DirectNumericChecksBundle;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.checkinfo.AnnotationBasedJugglerCheckInfo;
import ru.yandex.direct.juggler.check.checkinfo.BaseJugglerCheckInfo;
import ru.yandex.direct.juggler.check.checkinfo.NumericCheckInfo;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.hourglass.HourglassJob;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.scheduler.support.ParametersSource;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Каталог всех juggler-проверок аннотированных {@link JugglerCheck}
 * <p>
 * Позволяет выгрузить их в playbook для обновления на мониторинговых серверах и получить juggler-события по объекту
 * джобы
 */
@Component
@ParametersAreNonnullByDefault
public class JobsAppJugglerChecksProvider {

    private static final String TOP_LEVEL_CHECK_TEMPLATE = "jobs.toplevel_%s.working.%s";

    private static final Comparator<BaseJugglerCheckInfo> CHECKS_COMPARATOR =
            Comparator.comparing(BaseJugglerCheckInfo::getServiceName);

    private final ApplicationContext appCtx;
    private final EnvironmentTypeProvider environmentTypeProvider;

    private Map<Class, JobWorkingCheckInfo> jobToCheck;
    private List<NumericCheckInfo> numericChecks;
    private List<BaseJugglerCheckInfo> customChecks;

    private final CheckInfoExtractor<HourglassJob, JobWorkingCheckInfo> jobExtractor;
    private final CheckInfoExtractor<DirectNumericCheck, NumericCheckInfo> numericExtractor;
    private final CheckInfoExtractor<DirectNumericChecksBundle, NumericCheckInfo> numericBundleExtractor;

    @Autowired
    public JobsAppJugglerChecksProvider(ShardHelper shardHelper,
                                        EnvironmentTypeProvider environmentTypeProvider,
                                        ApplicationContext appCtx) {
        this.appCtx = appCtx;
        this.environmentTypeProvider = environmentTypeProvider;

        BiFunction<DirectNumericCheck, JugglerCheck, NumericCheckInfo> numericCheckToCheckInfo =
                (check, annotation) -> new NumericCheckInfo(annotation, check,
                        environmentTypeProvider.get().getLegacyName());

        numericExtractor = new CheckInfoExtractor<>(numericCheckToCheckInfo.andThen(Collections::singleton));

        numericBundleExtractor = new CheckInfoExtractor<>(
                (numericChecksBundle, annotation) -> numericChecksBundle.getBundledChecks().stream()
                        .map(numericCheck -> numericCheckToCheckInfo.apply(numericCheck, annotation))
                        .collect(toList()));

        jobExtractor = new CheckInfoExtractor<>(
                (job, annotation) -> Collections.singleton(
                        new JobWorkingCheckInfo(annotation, job.getClass(), shardHelper.dbShards(),
                                environmentTypeProvider.get(),
                                jobClass -> {
                                    if (DirectParameterizedJob.isParameterized(jobClass)) {
                                        @Nullable ParameterizedBy parameterizedBy =
                                                jobClass.getAnnotation(ParameterizedBy.class);
                                        if (null != parameterizedBy) {
                                            Class<? extends ParametersSource> sourceClazz =
                                                    parameterizedBy.parametersSource();
                                            return DirectParameterizedJob.getAllParams(sourceClazz, appCtx);
                                        }
                                    }
                                    return emptyList();
                                })));
    }

    /**
     * Конструктор исключительно для тестирования
     */
    JobsAppJugglerChecksProvider(Map<Class, JobWorkingCheckInfo> jobToCheck,
                                 List<NumericCheckInfo> numericChecks) {
        //noinspection ConstantConditions
        this(null, null, null);

        this.jobToCheck = jobToCheck;
        this.numericChecks = numericChecks;
        this.customChecks = emptyList();
    }

    Map<Class, JobWorkingCheckInfo> getJobToCheck() {
        if (jobToCheck == null) {
            jobToCheck = StreamEx.ofValues(appCtx.getBeansOfType(HourglassJob.class))
                    .mapToEntry(jobExtractor::getCheckDescription)
                    .nonNullValues()
                    .mapKeys(j -> (Class) j.getClass())
                    .toMap();
        }
        return jobToCheck;
    }

    List<NumericCheckInfo> getNumericChecks() {
        if (numericChecks == null) {
            StreamEx<NumericCheckInfo> bundledInfos =
                    StreamEx.ofValues(appCtx.getBeansOfType(DirectNumericChecksBundle.class))
                            .map(numericBundleExtractor::getCheckDescriptions)
                            .flatMap(Collection::stream);

            numericChecks = StreamEx.ofValues(appCtx.getBeansOfType(DirectNumericCheck.class))
                    .map(numericExtractor::getCheckDescription)
                    .append(bundledInfos)
                    .nonNull()
                    .collect(toList());
        }
        return numericChecks;
    }

    private List<BaseJugglerCheckInfo> getCustomChecks() {
        if (customChecks == null) {
            customChecks = StreamEx.ofValues(appCtx.getBeansOfType(BaseJugglerCheckInfo.class))
                    .filter(this::evaluateEnvironmentCondition)
                    .toList();
        }
        return customChecks;
    }

    /**
     * Получить из директовской джобы объект джагглерного события, заполненный из полей этой джобы
     */
    @Nullable
    public JugglerEvent getEvent(BaseDirectJob job) {
        JobWorkingCheckInfo checkInfo = getJobToCheck().get(job.getClass());
        if (checkInfo == null) {
            return null;
        }

        if (job instanceof DirectShardedJob) {
            return new JugglerEvent(
                    checkInfo.shardedServiceName(((DirectShardedJob) job).getShard()),
                    job.getJugglerStatus(), job.getJugglerDescription());
        } else if (job instanceof DirectParameterizedJob) {
            return new JugglerEvent(
                    checkInfo.parameterizedServiceName(((DirectParameterizedJob) job).getParam()),
                    job.getJugglerStatus(), job.getJugglerDescription());
        } else {
            return new JugglerEvent(checkInfo.getServiceName(), job.getJugglerStatus(),
                    job.getJugglerDescription());
        }
    }

    /**
     * Добавить в генератор плейбуки проверки по задачам из каталога
     *
     * @param builder генератор плейбуки
     */
    public void addChecks(PlaybookBuilder builder) {
        getJobToCheck().values().stream().sorted(CHECKS_COMPARATOR).forEach(desc -> desc.addCheckToPlaybook(builder));
        getNumericChecks().stream().sorted(CHECKS_COMPARATOR).forEach(desc -> desc.addCheckToPlaybook(builder));
        getCustomChecks().stream().sorted(CHECKS_COMPARATOR).forEach(desc -> desc.addCheckToPlaybook(builder));

        addTopLevelChecks(builder);
    }

    /**
     * Добавляет верхнеуровневые агрегаты над агрегатами джоб.
     * Джобы агрегируются по тегам.
     */
    private void addTopLevelChecks(PlaybookBuilder builder) {
        // тег -> список job, которые размечены этим тегом
        Map<CheckTag, List<JobWorkingCheckInfo>> tagToCheckInfoList =
                StreamEx.ofValues(getJobToCheck())
                        // мапа: список тегов -> джоба
                        .mapToEntry(JobWorkingCheckInfo::getTags, identity())
                        .removeKeys(List::isEmpty)
                        // список пар тег + джоба
                        .flatMapKeyValue((topLevelTags, checkInfo) ->
                                mapList(topLevelTags, tag -> Pair.of(tag, checkInfo)).stream())
                        // тег -> джоба
                        .mapToEntry(Pair::getKey, Pair::getValue)
                        .grouping();

        EntryStream.of(tagToCheckInfoList)
                .sorted(Map.Entry.comparingByKey())
                .forKeyValue((checkTag, checkInfoList) -> {
                    List<String> services = mapList(checkInfoList, JobWorkingCheckInfo::getServiceName);
                    String topLevelService = String.format(TOP_LEVEL_CHECK_TEMPLATE,
                            checkTag.getName(),
                            environmentTypeProvider.get().getLegacyName());
                    ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck check =
                            builder.addTopLevelCheck(services, topLevelService);
                });
    }

    private boolean evaluateEnvironmentCondition(BaseJugglerCheckInfo info) {
        return appCtx.getBean(info.getEnvironmentConditionClass()).evaluate();
    }

    private class CheckInfoExtractor<T, R extends AnnotationBasedJugglerCheckInfo> {
        private BiFunction<T, JugglerCheck, Collection<R>> extractor;

        CheckInfoExtractor(BiFunction<T, JugglerCheck, Collection<R>> extractor) {
            this.extractor = extractor;
        }

        private Collection<R> convertToChecksInfo(Map.Entry<T, JugglerCheck> in) {
            return extractor.apply(in.getKey(), in.getValue());
        }

        List<R> getCheckDescriptions(T annotatedObject) {
            return StreamEx.of(annotatedObject)
                    .cross(annotatedObject.getClass().getAnnotationsByType(JugglerCheck.class))
                    .map(this::convertToChecksInfo)
                    .flatMap(Collection::stream)
                    .filter(JobsAppJugglerChecksProvider.this::evaluateEnvironmentCondition)
                    .collect(toList());
        }

        @Nullable
        R getCheckDescription(T annotatedObject) {
            List<R> descriptions = getCheckDescriptions(annotatedObject);

            if (descriptions.size() == 1) {
                return descriptions.get(0);
            } else if (descriptions.isEmpty()) {
                return null;
            }

            throw new IllegalStateException(
                    String.format("There can be only one effective check per environment. %s has %s",
                            annotatedObject.getClass().getSimpleName(), descriptions.size()));
        }
    }

}
