package ru.yandex.direct.jobs.util.juggler.checkinfo;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.direct.ansiblejuggler.PlaybookBuilder;
import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMeta;
import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMetaUrl;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.checkinfo.AnnotationBasedJugglerCheckInfo;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.scheduler.support.DirectShardedJob.isShardedJobClass;

/**
 * Контейнер с информацией, необходимой для создания шаблона juggler-проверки на работоспособность джобы
 */
public class JobWorkingCheckInfo extends AnnotationBasedJugglerCheckInfo {
    private static final String SHARDED_CHECK_TEMPLATE = "%s.%s_%s";

    private final Class<?> jobClass;
    private final EnvironmentType environmentType;
    private final boolean sharded;
    private final boolean parameterized;
    private final String service;
    private final Collection<Integer> dbShards;
    private final Collection<String> allParams;

    /**
     * appCtx необходим для получения значений параметра из параметризованных джобов
     */
    public JobWorkingCheckInfo(JugglerCheck annotation, Class<?> jobClass, Collection<Integer> dbShards,
                               EnvironmentType environmentType,
                               @Nullable Function<Class<?>, Collection<String>> allParamsProvider) {
        super(annotation);

        this.jobClass = jobClass;
        this.environmentType = environmentType;
        this.dbShards = dbShards;
        sharded = isShardedJobClass(jobClass);
        parameterized = DirectParameterizedJob.isParameterized(jobClass);
        if (parameterized) {
            checkNotNull(allParamsProvider);
            this.allParams = allParamsProvider.apply(jobClass);
        } else {
            this.allParams = null;
        }
        service = String.format("jobs.%s.working.%s", jobClass.getSimpleName(), environmentType.getLegacyName());
    }

    @Override
    public String getServiceName() {
        return service;
    }

    public String shardedServiceName(int shard) {
        if (!sharded) {
            throw new IllegalStateException("Job is not sharded");
        }
        return String.format(SHARDED_CHECK_TEMPLATE, service, DirectShardedJob.SHARD_PARAM, shard);
    }

    public String parameterizedServiceName(String param) {
        if (!parameterized) {
            throw new IllegalStateException("Job is not parameterized");
        }
        return String.format(SHARDED_CHECK_TEMPLATE, service, DirectParameterizedJob.PARAM_NAME, param)
                .replaceAll("/", ".");
    }

    @Override
    public void addCheckToPlaybook(PlaybookBuilder builder) {
        ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck jugglerCheck;
        if (sharded) {
            List<String> checks = dbShards.stream()
                    .map(this::shardedServiceName)
                    .collect(Collectors.toList());
            jugglerCheck = builder.addMultiServicesCheck(checks, service, getTTL());
        } else if (parameterized) {
            List<String> checks = allParams.stream()
                    .map(this::parameterizedServiceName)
                    .collect(Collectors.toList());
            if (checks.isEmpty()) {
                // Если нет ни одного возможного значения параметра для запуска джобы,
                // то и проверки заводить не нужно (так как джоба запускаться вообще не будет)
                return;
            }
            jugglerCheck = builder.addMultiServicesCheck(checks, service, getTTL());
        } else {
            jugglerCheck = builder.addSingleServiceCheck(service, getTTL());
        }

        addMeta(jugglerCheck);
        getNotifications().forEach(jugglerCheck::withNotification);
        getTags().forEach(t -> jugglerCheck.withTag(t.getName()));
    }

    private void addMeta(ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck jugglerCheck) {
        JugglerMeta meta = new JugglerMeta();
        jugglerCheck.withMeta(meta);

        addDefaultUrls(jugglerCheck);
        addCustomUrls(jugglerCheck);
    }

    private void addDefaultUrls(ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck jugglerCheck) {
        JugglerMetaUrl logviewerUrl = UrlFactory.jobLogviewerUrl(environmentType, jobClass);
        if (logviewerUrl != null) {
            jugglerCheck.getMeta().withUrl(logviewerUrl);
        }
        jugglerCheck.getMeta()
                .withUrl(UrlFactory.jobDocumentationUrl(jobClass))
                .withUrl(UrlFactory.jobCodeUrl(jobClass));
    }

    private void addCustomUrls(ru.yandex.direct.ansiblejuggler.model.checks.JugglerCheck jugglerCheck) {
        getUrls().forEach(jugglerCheck.getMeta()::withUrl);
    }
}
