package ru.yandex.direct.jobs.util.juggler.checkinfo;

import java.util.Map;

import javax.annotation.Nullable;

import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMetaUrl;
import ru.yandex.direct.ansiblejuggler.model.meta.JugglerMetaUrlType;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.jobs.interceptors.JobTracingInterceptor;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessorNotScheduled;

import static ru.yandex.direct.utils.JavaKotlinInteropUtils.isKotlinClass;

public class UrlFactory {

    private static final Map<EnvironmentType, String> ENV_TYPE_TO_DOMAIN_MAP =
            Map.of(
                    EnvironmentType.PRODUCTION, "direct.yandex.ru",
                    EnvironmentType.TESTING, "test-direct.yandex.ru",
                    EnvironmentType.DEVTEST, "8998.beta1.direct.yandex.ru",
                    EnvironmentType.DEV7, "8996.beta1.direct.yandex.ru");

    private static final String JOB_LOGVIEWER_URL_FORMAT =
            "https://%s/logviewer/#~(logType~'messages~form~" +
                    "(fields~(~'log_time~'host~'service~'method~'trace_id~'span_id~'prefix~'log_level~'class_name" +
                    "~'message)" +
                    "~conditions~(service~'direct.jobs~method~'%s)" +
                    "~limit~100~offset~0~reverseOrder~true~showTraceIdRelated~false))$";

    private static final String JOB_DOC_BASE_URL = "https://docs.yandex-team.ru/direct-dev/reference/jobs/list/";

    private static final String JOB_CODE_BASE_URL = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/" +
            "jobs/src/main/java/";
    private static final String ESS_PROCESSOR_CODE_BASE_URL = "https://a.yandex-team.ru/arc/trunk/arcadia/direct/" +
            "apps/event-sourcing-system/logicprocessor/src/main/java/";

    @Nullable
    public static JugglerMetaUrl jobLogviewerUrl(EnvironmentType environmentType, Class<?> jobClass) {
        String domain = ENV_TYPE_TO_DOMAIN_MAP.get(environmentType);
        if (domain == null) {
            return null;
        }

        String jobLogviewerMethod = JobTracingInterceptor.formatMethodName(jobClass);
        String logviewerUrl = String.format(JOB_LOGVIEWER_URL_FORMAT, domain, jobLogviewerMethod);
        return new JugglerMetaUrl("Логи", logviewerUrl, JugglerMetaUrlType.YASM_ALERT);
    }

    public static JugglerMetaUrl jobDocumentationUrl(Class<?> jobClass) {
        String jobDocUrl = JOB_DOC_BASE_URL + jobClass.getSimpleName();
        return new JugglerMetaUrl("Документация", jobDocUrl, JugglerMetaUrlType.YASM_ALERT);
    }

    public static JugglerMetaUrl jobCodeUrl(Class<?> jobClass) {
        String jobBaseUrl = getJobBaseUrl(jobClass);
        String jobFileSuffix = getJobFileSuffix(jobClass);
        String jobDocUrl = jobBaseUrl +
                jobClass.getCanonicalName().replace(".", "/") +
                jobFileSuffix;
        return new JugglerMetaUrl("Код", jobDocUrl, JugglerMetaUrlType.YASM_ALERT);
    }

    private static String getJobBaseUrl(Class<?> jobClass) {
        return isEssProcessor(jobClass) ? ESS_PROCESSOR_CODE_BASE_URL : JOB_CODE_BASE_URL;
    }

    private static String getJobFileSuffix(Class<?> jobClass) {
        return isKotlinClass(jobClass) ? ".kt" : ".java";
    }

    private static boolean isEssProcessor(Class<?> jobClass) {
        return BaseLogicProcessorNotScheduled.class.isAssignableFrom(jobClass);
    }
}
