package ru.yandex.direct.jobs.util.yql;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YqlRowMapper;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.tables.generated.YtBanners;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Обобщение типовой задачи "экспортировать что-нибудь YQL-запросом".
 * <p>
 * Исходная таблица - таблица, с которой читается аттрибут {@code upload_time}.
 * <p>
 * Целевая таблица - результирующая таблица с выгрузкой (если выгружается несколько, то одна из них).
 * Генерируется по домашней директории YtCluster, относительному пути-окружения и суффиксу пути, заданному в билдере.
 * После генерации на нее записывается аттрибут {@code upload_time} исходной таблицы, вычитанный при начале генерации.
 * <p>
 * Провайдер биндингов - метод, которому на вход передается объект с параметрами текущей генерации,
 * который должен вернуть массив объектов, подставляемых в YQL-запрос (вместо {@code ?})
 */
@ParametersAreNonnullByDefault
public final class CommonYqlExport {
    private static final String YQL_START_TIME_ATTRIBUTE = "yql_start_time";
    private static final String MIN_LAST_TIMESTAMP_ATTRIBUTE = "min_last_timestamp";
    private static final String YT_NODE_TYPE_ATTRIBUTE = "type";
    private final Logger logger;
    private final YtOperator ytOperator;
    private final String title;
    private final String query;
    private final YtTable sourceTable;
    private final YtTable destinationTable;
    @Nullable
    private final YtTable metadataTable;
    private final Function<Context, Object[]> bindingsProvider;
    @Nullable
    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;
    private final YtDynamicOperator ytDynamicOperator;
    private final String syncStatesTablePath;
    private final String exportRelativePath;

    /**
     * Билдер, хранящий в себе общие для разных кластеров параметры
     */
    public static class Builder {
        private static final YtTable DEFAULT_SOURCE_TABLE = new YtBanners();
        private static final Function<Context, Object[]> BINDINGS_PROVIDER_EXPORT_TABLE =
                parameters -> new Object[]{parameters.getDestinationTablePath()};
        private static final Function<Context, Object[]> BINDINGS_PROVIDER_EXPORT_TABLE_STRING =
                parameters -> new Object[]{parameters.getDestinationTablePathAsString()};
        private static final Function<Context, Object[]> EMPTY_BINDINGS_PROVIDER = ignored -> new Object[]{};

        private final Logger logger;
        private final YtProvider ytProvider;
        private final String query;
        private final String exportRelativePath;

        private YtTable sourceTable;
        private String metadataRelativePath;
        private Function<Context, Object[]> bindingsProvider;
        private YtSQLSyntaxVersion yqlSyntaxVersion;
        @Nullable
        private YtClusterFreshnessRepository ytClusterFreshnessRepository;
        private String title;

        /**
         * Констуктор билдера.
         * <p>
         * Умолчания:
         * Исходная таблица {@link #DEFAULT_SOURCE_TABLE}.
         * Провайдер биндингов {@link #EMPTY_BINDINGS_PROVIDER} - пустой, без подстановок.
         *
         * @param logger             логгер класса, в котором предполагается запускать экспорт;
         *                           из него же получается короткое имя класса, подставляемое в трейсинг
         * @param ytProvider         компонент для получения инстансов по работе с YT
         * @param queryResource      путь к ресурсу с YQL-запросом
         * @param exportRelativePath суффикс пути для генерируемой таблицы
         */
        public Builder(
                Logger logger,
                YtProvider ytProvider,
                String queryResource,
                String exportRelativePath) {
            this.logger = logger;
            this.ytProvider = ytProvider;
            this.query = LiveResourceFactory.get(queryResource).getContent();
            this.exportRelativePath = exportRelativePath;
            this.yqlSyntaxVersion = YtSQLSyntaxVersion.SQLv1;
        }

        /**
         * Переопределить исходную таблицу
         */
        public Builder withSourceTable(YtTable sourceTable) {
            this.sourceTable = sourceTable;
            return this;
        }

        public Builder withMetadataRelativePath(String metadataRelativePath) {
            this.metadataRelativePath = metadataRelativePath;
            return this;
        }

        /**
         * Переопределить провайдер биндингов
         */
        public Builder withBindingsProvider(Function<Context, Object[]> bindingsProvider) {
            this.bindingsProvider = bindingsProvider;
            return this;
        }

        /**
         * Задать провайдер биндингов, генерирующий один подстановочный параметр - равный пути целевой таблицы
         */
        public Builder withExportTablePathAsBindings() {
            this.bindingsProvider = BINDINGS_PROVIDER_EXPORT_TABLE;
            return this;
        }

        /**
         * Задать провайдер биндингов, генерирующий один подстановочный параметр - равный строке пути до целевой таблицы
         */
        public Builder withExportTablePathStringAsBindings() {
            this.bindingsProvider = BINDINGS_PROVIDER_EXPORT_TABLE_STRING;
            return this;
        }

        public Builder withYtClusterFreshnessRepository(YtClusterFreshnessRepository ytClusterFreshnessRepository) {
            this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;
            return this;
        }

        /**
         * Переопределить версию YQL-синтаксиса для запроса
         */
        public Builder withYqlSyntaxVersion(YtSQLSyntaxVersion yqlSyntaxVersion) {
            this.yqlSyntaxVersion = yqlSyntaxVersion;
            return this;
        }

        /**
         * Создать инстанс экспортера
         *
         * @param ytCluster кластер YT на котором будет выполняться экспорт
         */
        public CommonYqlExport build(YtCluster ytCluster) {
            return new CommonYqlExport(
                    this,
                    ytCluster,
                    nvl(sourceTable, DEFAULT_SOURCE_TABLE),
                    nvl(bindingsProvider, EMPTY_BINDINGS_PROVIDER),
                    ytClusterFreshnessRepository,
                    title,
                    exportRelativePath,
                    metadataRelativePath);
        }

        Logger getLogger() {
            return logger;
        }

        YtProvider getYtProvider() {
            return ytProvider;
        }

        YtSQLSyntaxVersion getYqlSyntaxVersion() {
            return yqlSyntaxVersion;
        }

        String getQuery() {
            return query;
        }

        public Builder withTitle(String title) {
            this.title = title;
            return this;
        }
    }

    /**
     * Конструктор
     *
     * @param exportRelativePath           относительный путь для результатов экспорта
     * @param metadataRelativePath         относительный путь для таблицы с метаданными
     * @param builder                      билдер, из него получаем те параметры, что final
     * @param ytCluster                    кластер YT для выполнения
     * @param sourceTable                  исходная таблица
     * @param bindingsProvider             провайдер биндингов
     * @param ytClusterFreshnessRepository
     */
    private CommonYqlExport(
            Builder builder,
            YtCluster ytCluster,
            YtTable sourceTable,
            Function<Context, Object[]> bindingsProvider,
            @Nullable YtClusterFreshnessRepository ytClusterFreshnessRepository,
            String title,
            String exportRelativePath,
            @Nullable String metadataRelativePath
    ) {
        String home = builder.getYtProvider().getClusterConfig(ytCluster).getHome();

        this.logger = builder.getLogger();
        this.ytOperator = builder.getYtProvider().getOperator(ytCluster, builder.getYqlSyntaxVersion());
        this.ytDynamicOperator = builder.getYtProvider().getDynamicOperator(ytCluster);
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;
        this.syncStatesTablePath = "//home/direct/mysql-sync/current/mysql-sync-states";
        this.query = builder.getQuery();

        this.sourceTable = sourceTable;
        this.bindingsProvider = bindingsProvider;

        this.destinationTable = new YtTable(YtPathUtil.generatePath(home, relativePart(), exportRelativePath));
        this.exportRelativePath = exportRelativePath;

        if (metadataRelativePath != null) {
            this.metadataTable = new YtTable(YtPathUtil.generatePath(home, relativePart(), metadataRelativePath));
        } else {
            this.metadataTable = null;
        }

        this.title = title;
    }

    /**
     * Контекст генерации.
     * Содержите в себе параметры, относящиеся к текущему выполнению {@link #generateIfNeeded()}
     */
    public class Context {
        private String sourceUploadTimeCached;

        private Context() {
            logger.trace("fetching upload time for {}", sourceTable.getPath());
            sourceUploadTimeCached = ytOperator.readTableUploadTime(sourceTable);
            logger.info("upload_time for {} is {}", sourceTable.getPath(), sourceUploadTimeCached);
        }

        /**
         * Получить путь до целевой таблицы в виде массива байт (в кодировке UTF_8).
         * Полезно для того, чтобы указать в биндинге.
         */
        public byte[] getDestinationTablePath() {
            return destinationTable.getPath().getBytes(StandardCharsets.UTF_8);
        }

        /**
         * Получить путь до целевой таблицы в виде строки.
         * Полезно, при подстановке таблицы в переменную, а не тело запроса.
         */
        public String getDestinationTablePathAsString() {
            return destinationTable.getPath();
        }

        /**
         * Получить аттрибут {@code upload_time} таблицы, заданной как исходная, на момент начала генерации.
         */
        public String getDbUploadTime() {
            return sourceUploadTimeCached;
        }

        /**
         * Получить дату выгрузки исходной таблицы.
         * Ожидается, что ее аттрибут {@code upload_time} задан в формате {@literal 2018-10-24T07:06:19Z}
         */
        public LocalDate getDbUploadDate() {
            String date = getDbUploadTime().substring(0, 10);
            return LocalDate.parse(date, ISO_LOCAL_DATE);
        }
    }

    /**
     * Устаревший метод. Старые джобы должны быть заменены общей джобой - PpcDataExportJob.
     * <p>
     * Запустить генерацию, если целевая таблица не существует или ее времени загрузки меньше, чем у исходной
     * <p>
     * Если генерация была запущена - по завершении записывает время выгрузки исходной таблицы в аттрибуты целевой.
     */
    @Deprecated
    public void generateIfNeeded() {
        Context ctx = new Context();

        if (ytOperator.exists(destinationTable)
                && ytOperator.tryReadTableUploadTime(destinationTable).orElse("").equals(ctx.getDbUploadTime())) {
            logger.info("table {} already exists and up-to-date", destinationTable.getPath());
        } else {
            generate(ctx, null);
            ytOperator.writeTableUploadTime(destinationTable, ctx.getDbUploadTime());
        }
    }

    public void generateIfNeeded(Duration deltaTime) {
        generateIfNeeded(deltaTime, null);
    }

    /**
     * Запустить генерацию, если целевая таблица не существует или текущее время больше, чем последнее время загруки
     * + {@code deltaTime}
     * <p>
     * Если генерация была запущена - по завершении записывает время окончание выгрузки в аттрибуты целевой.
     */
    public void generateIfNeeded(Duration deltaTime, @Nullable YqlRowMapper mapper) {

        var timeToUpdateTable = readTimeToUpdateTable(deltaTime);

        if (LocalDateTime.now().isAfter(timeToUpdateTable)) {
            Context ctx = new Context();
            String yqlStartTime = LocalDateTime.now().toString();
            Map<Integer, Long> shardToTimestamp =
                    ifNotNull(ytClusterFreshnessRepository,
                            r -> r.loadShardToTimestamp(ytDynamicOperator, syncStatesTablePath));

            generate(ctx, mapper);

            writeMetadata(yqlStartTime, shardToTimestamp);
        } else {
            logger.info("table {} already exists and up-to-date", destinationTable.getPath());
        }
    }

    private LocalDateTime readTimeToUpdateTable(Duration deltaTime) {
        var tableWithMetadata =
                ytOperator.exists(destinationTable) && isTableNode(destinationTable)
                        ? destinationTable
                        : metadataTable;

        LocalDateTime timeToUpdateTable;
        if (tableWithMetadata != null && ytOperator.exists(tableWithMetadata)) {
            timeToUpdateTable = ytOperator.tryReadTableUploadTime(tableWithMetadata)
                    .map(x -> LocalDateTime.parse(x).plusSeconds(deltaTime.getSeconds()))
                    .orElse(LocalDateTime.MIN);
        } else {
            timeToUpdateTable = LocalDateTime.MIN;
        }

        return timeToUpdateTable;
    }

    private boolean isTableNode(YtTable destinationTable) {
        try {
            var node = ytOperator.getYt().cypress().get(destinationTable.ypath(), Cf.set(YT_NODE_TYPE_ATTRIBUTE));
            return "table".equals(node.getAttribute(YT_NODE_TYPE_ATTRIBUTE).get().stringValue());
        } catch (RuntimeException ex) {
            logger.warn("Couldn't get destination table node type", ex);
            return false;
        }
    }

    /**
     * Метод сохраняет данные о результатах запуска, в том числе время окончания экспорта,
     * на основе которого будет вычислено время следующего запуска
     */
    private void writeMetadata(String yqlStartTime, @Nullable Map<Integer, Long> shardToTimestamp) {
        YtTable tableWithMetadata;

        var useDestTable = ytOperator.exists(destinationTable) && isTableNode(destinationTable);
        if (useDestTable) {
            tableWithMetadata = destinationTable;
        } else {
            if (metadataTable == null) {
                logger.warn("Cannot write metadata for {}", destinationTable.getPath());
                return;
            }

            if (!ytOperator.exists(metadataTable)) {
                ytOperator.getYt().cypress().create(metadataTable.ypath(), CypressNodeType.TABLE, true);
            }

            tableWithMetadata = metadataTable;
        }

        ytOperator.writeTableUploadTime(tableWithMetadata, LocalDateTime.now().toString());
        ytOperator.writeTableStringAttribute(tableWithMetadata, YQL_START_TIME_ATTRIBUTE, yqlStartTime);
        if (shardToTimestamp != null) {
            var minLastTimestamp = shardToTimestamp.values().stream()
                    .filter(Objects::nonNull)
                    .min(Long::compareTo)
                    .map(ts -> ts / 1000)
                    .map(Objects::toString)
                    .orElse(null);
            ytOperator.writeTableStringAttribute(tableWithMetadata, MIN_LAST_TIMESTAMP_ATTRIBUTE, minLastTimestamp);
        }
    }

    /**
     * Выполнить yql запрос с использованием своего маппера.
     * Используется, когда нужно обрабатывать полученные из запроса значения,
     * например, при запросе данных для выгрузки в соломон
     */
    public void generate(Context ctx, @Nullable YqlRowMapper mapper) {
        Object[] bindings = bindingsProvider.apply(ctx);
        logger.info("start generate");

        var yqlQuery = new YqlQuery(this.query, bindings).withTitle(title);
        if (mapper != null) {
            ytOperator.yqlQuery(yqlQuery, mapper);
        } else {
            ytOperator.yqlExecute(yqlQuery);
        }

        logger.info("generate finished");
    }
}
