package ru.yandex.direct.jobs.verifications;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Сверка сумм откруток, сгруппированных по кампаниям и дням между таблицой
 * //home/direct/export/balance/BillingOrderDomainsSplittedStat посчитанной Директом
 * и //home/yabs/stat/BillingOrderDomainsSplittedStat
 * <p>
 * Они не должны сильно расходиться, иначе агентствам будет неправильно расчитана премия.
 * https://st.yandex-team.ru/DIRECT-106298
 * <p>
 *
 * @see ru.yandex.direct.jobs.export.BillingOrderDomainsExporter
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_2})
@Hourglass(cronExpression = "0 59 1 * * ?", needSchedule = ProductionOnly.class)
public class BillingOrderDomainsSumCheckJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(BillingOrderDomainsSumCheckJob.class);

    private final YtProvider ytProvider;

    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("verifications/BillingOrderDomainsSumCheck.sql")
                    .readLines()
    );

    public BillingOrderDomainsSumCheckJob(YtProvider ytProvider) {
        this.ytProvider = ytProvider;
    }

    @Override
    public void execute() {
        YtOperator ytOperator = ytProvider.getOperator(YtCluster.HAHN, YtSQLSyntaxVersion.SQLv1);

        var now = ZonedDateTime.now(DateTimeUtils.MSK);
        var diffTable = String.format("//home/direct/tmp/andreymak/bod_by_cid_date_diff_%s",
                now.format(DateTimeFormatter.ISO_LOCAL_DATE)
        );
        var yesterdayTs = now.truncatedTo(ChronoUnit.DAYS).minusDays(1).toEpochSecond();
        Object[] params = StreamEx.of(diffTable, yesterdayTs).toArray();

        logger.info("Running YQL. Diff table is {}", diffTable);
        var stats = ytOperator.yqlQuery(YQL_QUERY, rs -> {
            var metadata = rs.getMetaData();
            List<String> statRows = new ArrayList<>();
            for (int i = 1; i <= metadata.getColumnCount(); ++i) {
                String colLabel = metadata.getColumnLabel(i);
                statRows.add(String.format("%s: %s", colLabel, rs.getString(colLabel)));
            }
            return statRows;
        }, params).get(0);
        logger.info("Finished YQL");

        logger.info("Stats: {}", String.join(", ", stats));
    }
}
