package ru.yandex.direct.jobs.verifications;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.http.client.HttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.jobs.export.BillingOrderDomainsExporter;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.utils.HttpClientUtils;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_ROBOT_ADS_AUDIT_TOKEN;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

@Service
@ParametersAreNonnullByDefault
public class BillingOrderQualityYqlRunner {

    private static final Logger logger = LoggerFactory.getLogger(BillingOrderQualityYqlRunner.class);

    private static final String BILLING_ORDER_DOMAINS_YQL_QUERY =
            "SELECT COUNT(Cost) AS `count`, SUM(Cost) AS cost FROM `?` WHERE Domain = \"\" AND ServiceID = 7";
    private static final String TICKET_HEADER =
            "Количество или стоимость откруток с пустым доменом превышает допустимое значение";
    private static final String TICKET_DESCRIPTION_TEMPLATE =
            "Кластер: %s, таблица: ((%s %s)), количество: %d, суммарная стоимость откруток: %d";
    private static final String ST_USER_AGENT = "Startrek Java Client";
    private static final String ST_API_URL = "https://st-api.yandex-team.ru";

    private static final Long DEFAULT_COST_THRESHOLD = 310000_000_000L;
    private static final Long DEFAULT_COUNT_THRESHOLD = 76000L;

    private final SolomonPushClient solomonPushClient;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public static class Metrics {

        private long count;
        private long cost;

        public Metrics(long count, long cost) {
            this.count = count;
            this.cost = cost;
        }

        public long getCount() {
            return count;
        }

        public long getCost() {
            return cost;
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }

    private final YtProvider provider;
    private final String startrekToken;
    private final HttpClient startrekHttpClient;

    private final DirectConfig directConfig;


    public BillingOrderQualityYqlRunner(SolomonPushClient solomonPushClient,
                                        PpcPropertiesSupport ppcPropertiesSupport, YtProvider provider,
                                        @Nullable @Qualifier(STARTREK_ROBOT_ADS_AUDIT_TOKEN) String startrekToken,
                                        DirectConfig directConfig) {
        this.solomonPushClient = solomonPushClient;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.provider = provider;
        this.startrekToken = startrekToken;
        this.directConfig = directConfig;
        startrekHttpClient = HttpClientUtils.https(
                TimeUnit.SECONDS.toMillis(5),
                TimeUnit.SECONDS.toMillis(5),
                10,
                ST_USER_AGENT
        );
    }

    public void run(YtCluster cluster) {
        var home = provider.getClusterConfig(cluster).getHome();
        var operator = provider.getOperator(cluster, YtSQLSyntaxVersion.SQLv1);

        verifyMissingDomains(cluster, home, operator, BillingOrderDomainsExporter.OUTPUT_TABLE_PATH);
        verifyMissingDomains(cluster, home, operator, BillingOrderDomainsExporter.OUTPUT_TABLE_PATH_SPLITTED_STAT);
    }

    private void verifyMissingDomains(YtCluster cluster, String home, YtOperator operator, String tableName) {
        var tablePath = YtPathUtil.generatePath(home, relativePart(), tableName);

        logger.info(
                "Running query calculating sum and count of records with empty domains {} with parameter {}",
                BILLING_ORDER_DOMAINS_YQL_QUERY,
                tablePath
        );
        operator
                .yqlQuery(BILLING_ORDER_DOMAINS_YQL_QUERY, this::mapToMetrics, tablePath)
                .stream()
                .findAny()
                .ifPresent(metrics -> handleMetrics(cluster, metrics, tablePath));
    }

    private void handleMetrics(YtCluster cluster, Metrics metrics, String tablePath) {
        var costThreshold = Optional
                .ofNullable(ppcPropertiesSupport.get(PpcPropertyNames.BILLING_ORDER_DOMAINS_COST_THRESHOLD).get())
                .orElse(DEFAULT_COST_THRESHOLD);
        logger.info("Cost threshold is {}", costThreshold);
        var countThreshold = Optional
                .ofNullable(ppcPropertiesSupport.get(PpcPropertyNames.BILLING_ORDER_DOMAINS_COUNT_THRESHOLD).get())
                .orElse(DEFAULT_COUNT_THRESHOLD);
        logger.info("Count threshold is {}", countThreshold);

        if (metrics.getCost() > costThreshold || metrics.getCount() > countThreshold) {
            createTicket(cluster, metrics, tablePath);
        }
        sendMetrics(cluster, metrics);
    }

    private void createTicket(YtCluster cluster, Metrics metrics, String tablePath) {
        var billingOrderDomainsMonitoringConfig = directConfig
                .getBranch("billing_order_domains_monitoring")
                .getBranch("create_ticket");

        var assignee = billingOrderDomainsMonitoringConfig.getString("assignee");
        var followers = billingOrderDomainsMonitoringConfig.getStringList("followers");
        var ticketQueue = billingOrderDomainsMonitoringConfig.getString("queue");
        var ticketTag = billingOrderDomainsMonitoringConfig.getString("tag");

        checkNotNull(startrekToken);
        var session = StartrekClientBuilder.newBuilder()
                .uri(ST_API_URL)
                .httpClient(startrekHttpClient)
                .build(startrekToken);
        var text = String.format(
                TICKET_DESCRIPTION_TEMPLATE,
                cluster.getName(),
                tablePath,
                tablePath,
                metrics.getCount(),
                metrics.getCost()
        );
        var builder = IssueCreate.builder()
                .queue(ticketQueue)
                .summary(TICKET_HEADER)
                .description(text)
                .type("task")
                .priority("critical")
                .tags(ticketTag)
                .assignee(assignee)
                .followers(followers.toArray(new String[]{}));
        var issueCreate = builder.build();
        var issue = session.issues().create(issueCreate);
        logger.info("Issue created: https://st.yandex-team.ru/{}", issue.getKey());
    }

    private void sendMetrics(YtCluster cluster, Metrics metrics) {
        logger.info("Sending metrics into solomon {}", metrics);
        var sensorsRegistry = SolomonUtils.newPushRegistry(
                "cluster",
                cluster.getName(),
                "flow",
                "billing_orders_without_domain"
        );
        sensorsRegistry.gaugeDouble("billing_orders_without_domain_count").set(metrics.getCount());
        sensorsRegistry.gaugeDouble("billing_orders_without_domain_cost").set(metrics.getCost());

        solomonPushClient.sendMetrics(sensorsRegistry);
    }

    private Metrics mapToMetrics(ResultSet rs) throws SQLException {
        var count = rs.getLong("count");
        var cost = rs.getLong("cost");
        return new Metrics(count, cost);
    }
}
