package ru.yandex.direct.jobs.verifications;

import java.net.InetAddress;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.HttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.BOID_CHECK_MONITORING_ENABLED;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.BOID_CHECK_MONITORING_STATE;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.BOID_CHECK_MONITORING_USE_TRACKER;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_HTTP_CLIENT_BEAN;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_ROBOT_ADS_AUDIT_TOKEN;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Джоба для мониторинга сверок боидов
 * Запускается два раза в день и выполняет YQL запросы (недолгие, минут по 7).
 * Если на каком-то из кластеров обнаруживаются расхождения, то создаётся тикет в стартреке
 * и туда прикладывается таблица с результатами.
 * Если алерт срабатывает второй или третий раз за день, то новый тикет не создаём -- дописываем в имеющийся.
 * Таблицы с результатами удаляются, если они пустые (расхождений нет).
 * Если же расхождения есть, то таблице устанавливается TTL=2 года.
 * Состояние последней проверки записывается в базу в PPC_PROPERTIES.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 24),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, YT, DIRECT_PRODUCT_TEAM})
@Hourglass(cronExpression = "0 0 8/12 * * ?", needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1, hours = 1, minutes = 30),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_2, YT})
@Hourglass(cronExpression = "0 0 8/12 * * ?", needSchedule = NonProductionEnvironment.class)
@ParametersAreNonnullByDefault
public class BoidCheckMonitoringJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(BoidCheckMonitoringJob.class);

    // По умолчанию джоба выключена, включается через проперти BOID_CHECK_MONITORING_ENABLED
    private static final String DEFAULT_BOID_CHECK_MONITORING_ENABLED = "0";

    // По умолчанию джоба не пишет в Стартрек, включается через проперти BOID_CHECK_MONITORING_USE_TRACKER
    private static final String DEFAULT_BOID_CHECK_MONITORING_USE_TRACKER = "0";

    private static final SimpleDateFormat DATETIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm");

    private static final SimpleDateFormat CHECKTIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");

    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    // Сюда складываем таблички с результатами проверок
    private static final String OUTPUT_PATH_BASE = "export/monitoring_boids";

    private static final String PRODUCTION_STARTREK_QUEUE = "DIRECTSUP";
    private static final String TEST_STARTREK_QUEUE = "TEST";

    private final YtProvider ytProvider;

    private List<YtCluster> ytClusters;

    // Может быть null, если запускается из теста
    @Nullable
    private final String startrekToken;
    private final HttpClient startrekHttpClient;

    private final PpcPropertiesSupport ppcPropertiesSupport;

    private static final String BOID_CHECK_MONITORING_YQL_QUERY =
            String.join("\n", new ClassPathResourceInputStreamSource("verifications/BoidCheckMonitoring.sql")
                    .readLines());

    @Autowired
    public BoidCheckMonitoringJob(
            YtProvider ytProvider,
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectExportYtClustersParametersSource parametersSource,
            @Nullable @Qualifier(STARTREK_ROBOT_ADS_AUDIT_TOKEN) String startrekToken,
            @Qualifier(STARTREK_HTTP_CLIENT_BEAN) HttpClient startrekHttpClient) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytClusters = parametersSource.getAllParamValues();
        this.startrekToken = startrekToken;
        this.startrekHttpClient = startrekHttpClient;
    }

    private boolean isEnabled() {
        String jobEnabledProperty = ppcPropertiesSupport
                .find(BOID_CHECK_MONITORING_ENABLED.getName())
                .filter(StringUtils::isNumeric)
                .map(String::valueOf)
                .orElse(DEFAULT_BOID_CHECK_MONITORING_ENABLED);
        logger.info("Loaded ppc_property {} = {}", BOID_CHECK_MONITORING_ENABLED.getName(), jobEnabledProperty);
        return "1".equals(jobEnabledProperty);
    }

    private boolean isStartrekEnabled() {
        String useTrackerProperty = ppcPropertiesSupport
                .find(BOID_CHECK_MONITORING_USE_TRACKER.getName())
                .filter(StringUtils::isNumeric)
                .map(String::valueOf)
                .orElse(DEFAULT_BOID_CHECK_MONITORING_USE_TRACKER);
        logger.info("Loaded ppc_property {} = {}", BOID_CHECK_MONITORING_USE_TRACKER.getName(), useTrackerProperty);
        return "1".equals(useTrackerProperty);
    }

    private YPath getOrCreateBaseDir(YtCluster ytCluster) {
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(ytCluster);
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        //
        String home = ytClusterConfig.getHome();
        YPath path = YPath.simple(YtPathUtil.generatePath(home, relativePart(), OUTPUT_PATH_BASE));
        if (!ytOperator.getYt().cypress().exists(path)) {
            ytOperator.getYt().cypress().create(path, CypressNodeType.MAP, true);
        }
        //
        return path;
    }

    @Override
    public void execute() {
        if (!isEnabled()) {
            logger.info("Job is not enabled, exiting");
            return;
        }

        boolean startrekEnabled = isStartrekEnabled() && startrekToken != null;
        logger.info("startrekEnabled = {}", startrekEnabled);

        // Фиксируем время проверки
        Date checkTime = new Date();

        logger.info("Starting YQL queries to {}", ytClusters.stream().map(YtCluster::getName).collect(joining(", ")));
        List<CompletableFuture<YqlResult>> futures = ytClusters.stream().map(ytCluster ->
                CompletableFuture.supplyAsync(() -> {
                    YPath baseDir = getOrCreateBaseDir(ytCluster);
                    YPath tablePath = baseDir.child(DATETIME_FORMAT.format(checkTime));

                    Object[] params = StreamEx.of(CHECKTIME_FORMAT.format(checkTime), tablePath.toString()).toArray();

                    class ResultRow {
                        int failed;
                        int total;
                    }

                    YtOperator ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);
                    ResultRow result = ytOperator.yqlQuery(BOID_CHECK_MONITORING_YQL_QUERY, rs -> {
                        ResultRow row = new ResultRow();
                        row.failed = rs.getInt("failed_clients_cnt");
                        row.total = rs.getInt("total_clients_cnt");
                        return row;
                    }, params).get(0);

                    YqlResult yqlResult = new YqlResult();
                    yqlResult.ytCluster = ytCluster;
                    yqlResult.countFailed = result.failed;
                    yqlResult.countOk = result.total - result.failed;
                    yqlResult.tablePath = tablePath;
                    return yqlResult;
                })).collect(toList());

        // Запрос должен выполниться хотя бы для одного из кластеров
        List<YqlResult> results = new ArrayList<>();
        for (var future : futures) {
            try {
                results.add(future.get());
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("YQL execution was interrupted", e);
            } catch (ExecutionException e) {
                logger.error("YQL execution was failed", e);
            }
        }
        if (results.isEmpty()) {
            // Не удалось получить результат ни с одного из кластеров -- выходим с ошибкой
            // Если несколько запусков подряд ситуация будет повторяться -- сработает алерт
            throw new RuntimeException("No success YQL execution on any YT cluster");
        }

        JobState prevJobState = loadJobState(ppcPropertiesSupport);
        checkNotNull(prevJobState.getClusterStates());
        logger.info("Loaded prevJobState = {}", toJson(prevJobState));

        Map<YtCluster, YqlResult> resultsMap = results.stream().collect(toMap(r -> r.ytCluster, identity()));
        JobState jobState = JobState.emptyJobState();
        for (YtCluster ytCluster : ytClusters) {
            if (resultsMap.containsKey(ytCluster)) {
                //
                YqlResult yqlResult = resultsMap.get(ytCluster);
                if (yqlResult.countFailed == 0) {
                    // Удаляем таблицу
                    logger.info("No failed clients, removing table '{}'", yqlResult.tablePath);
                    removeTableQuietly(ytCluster, yqlResult.tablePath);
                } else {
                    // Устанавливаем TTL=2 года для этой таблицы
                    // Чтобы при необходимости можно было пойти и посмотреть исторические данные
                    GregorianCalendar gregorianCalendar = new GregorianCalendar();
                    gregorianCalendar.setTime(checkTime);
                    gregorianCalendar.add(Calendar.YEAR, 2);
                    logger.info("Set expiration_time = {} on '{}'", gregorianCalendar.getTime(), yqlResult.tablePath);
                    ytProvider.getOperator(ytCluster).getYt().cypress().set(
                            yqlResult.tablePath.attribute(YtUtils.EXPIRATION_TIME_ATTR),
                            gregorianCalendar.getTimeInMillis()
                    );
                }

                // Обновляем состояние джобы для этого кластера
                jobState.getClusterStates().put(
                        ytCluster.getName(),
                        new JobClusterState()
                                .withLastCheckTimestamp(checkTime)
                                .withLastCheckCountOk(yqlResult.countOk)
                                .withLastCheckCountFailed(yqlResult.countFailed)
                );
            } else {
                // Не удалось получить результат в этом кластере -- в этом случае
                // просто оставим старый результат (если он есть)
                if (prevJobState.getClusterStates().containsKey(ytCluster.getName())) {
                    jobState.getClusterStates().put(ytCluster.getName(),
                            prevJobState.getClusterStates().get(ytCluster.getName()));
                }
            }
        }

        // Если есть хотя бы 1 кластер, на котором запрос вернул ненулевое кол-во расхождений, пишем это в тикет
        if (resultsMap.values().stream().anyMatch(r -> r.countFailed != 0)) {
            String text = composeRobotAlertMessage(resultsMap);
            //
            if (prevJobState.getLastTicketTimestamp() != null
                    && !StringUtils.isBlank(prevJobState.getLastTicket())
                    && isSameDay(prevJobState.getLastTicketTimestamp(), checkTime)) {
                // Дописываем комментарий в уже имеющийся тикет
                logger.info("Adding {} comment to https://st.yandex-team.ru/{}: \n{}",
                        startrekEnabled ? "" : "(not real, startrek is disabled)",
                        prevJobState.getLastTicket(), text);
                if (startrekEnabled) {
                    addTicketComment(prevJobState.getLastTicket(), text);
                }
                // Информацию о тикете не меняем
                jobState.withLastTicket(prevJobState.getLastTicket());
                jobState.withLastTicketTimestamp(prevJobState.getLastTicketTimestamp());
            } else {
                // Создаём новый тикет
                String summary = String.format("Расхождения в BOID %s", DATE_FORMAT.format(checkTime));
                logger.info("Creating {} startrek ticket:\n{}\n{}",
                        startrekEnabled ? "" : "(not real, startrek is disabled)",
                        summary, text);
                if (startrekEnabled) {
                    String ticketKey = createTicket(summary, text);
                    jobState.withLastTicket(ticketKey);
                    jobState.withLastTicketTimestamp(new Date());
                }
            }
        } else {
            logger.info("Failed clients were not found");
        }

        logger.info("Saving new jobState = {}", toJson(jobState));
        saveJobState(jobState);
    }

    private String createTicket(String summary, String text) {
        checkNotNull(startrekToken);
        //
        Session session = StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .httpClient(startrekHttpClient)
                .build(startrekToken);
        //
        String queue;
        String tag;
        ComponentRef componentRef;
        if (Environment.getCached().isProduction()) {
            queue = PRODUCTION_STARTREK_QUEUE;
            tag = "direct-boids";
            componentRef = session.queues().get(PRODUCTION_STARTREK_QUEUE).getComponents().stream()
                    .filter(c -> c.getDisplay().equals("audit_direct_boid"))
                    .findFirst()
                    .orElse(null);
        } else {
            queue = TEST_STARTREK_QUEUE;
            tag = "direct-boids-test";
            componentRef = null;
        }
        //
        IssueCreate.Builder builder = IssueCreate.builder()
                .queue(queue)
                .summary(summary)
                .description(text)
                .type("task")
                .tags(tag);
        if (componentRef != null) {
            builder.components(componentRef);
        }
        IssueCreate issueCreate = builder.build();
        //
        Issue issue = session.issues().create(issueCreate);
        logger.info("Issue created: https://st.yandex-team.ru/{}", issue.getKey());
        //
        return issue.getKey();
    }

    private void addTicketComment(String ticketKey, String text) {
        checkNotNull(startrekToken);
        //
        Session session = StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .httpClient(startrekHttpClient)
                .build(startrekToken);

        Issue issue = session.issues().get(ticketKey);
        issue.comment(text);
    }

    private boolean isSameDay(Date a, Date b) {
        GregorianCalendar calendarA = new GregorianCalendar();
        GregorianCalendar calendarB = new GregorianCalendar();
        calendarA.setTime(a);
        calendarB.setTime(b);
        return calendarA.get(Calendar.YEAR) == calendarB.get(Calendar.YEAR)
                && calendarA.get(Calendar.DAY_OF_YEAR) == calendarB.get(Calendar.DAY_OF_YEAR);
    }

    private String composeRobotAlertMessage(Map<YtCluster, YqlResult> resultsMap) {
        String message = resultsMap.values().stream()
                .filter(r -> r.countFailed != 0)
                .map(yqlResult -> {
                    StringBuilder sb = new StringBuilder();
                    sb.append("Обнаружены клиенты с расхождениями в BOID'ах на кластере ")
                            .append(yqlResult.ytCluster).append(": ").append(yqlResult.countFailed).append("\n\n")
                            .append("((https://yt.yandex-team.ru/")
                            .append(yqlResult.ytCluster.getName())
                            .append("/navigation?path=")
                            .append(URLEncoder.encode(yqlResult.tablePath.toString(), StandardCharsets.UTF_8))
                            .append(" Таблица с результатами))");
                    return sb.toString();
                })
                .collect(joining("\n\n\n"));
        String signature = "((https://wiki.yandex-team.ru/users/aliho/projects/direct/sverki/ Информация о проверке))" +
                "\n\n" + "BoidCheckMonitoringJob @ " + getLocalHostname();
        return message + "\n\n----\n\n" + signature;
    }

    private String getLocalHostname() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            return "unknown";
        }
    }

    private void removeTableQuietly(YtCluster ytCluster, YPath path) {
        try {
            ytProvider.getOperator(ytCluster).getYt().cypress().remove(path);
        } catch (RuntimeException e) {
            // Некритичная ошибка
            logger.error(String.format("Can't remove the table %s", path), e);
        }
    }

    public static JobState loadJobState(PpcPropertiesSupport ppcPropertiesSupport) {
        Optional<String> propertyOptional = ppcPropertiesSupport.find(BOID_CHECK_MONITORING_STATE.getName());
        if (propertyOptional.isPresent()) {
            try {
                return fromJson(propertyOptional.get(), JobState.class);
            } catch (IllegalArgumentException e) {
                logger.error(String.format("Can't deserialize job state from '%s'", propertyOptional.get()), e);
                return JobState.emptyJobState();
            }
        }
        return JobState.emptyJobState();
    }

    private void saveJobState(JobState jobState) {
        ppcPropertiesSupport.set(BOID_CHECK_MONITORING_STATE.getName(), toJson(jobState));
    }

    private static class YqlResult {
        YtCluster ytCluster;
        int countOk;
        int countFailed;
        YPath tablePath;
    }
}
