package ru.yandex.direct.jobs.verifications;

import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.monlib.metrics.labels.Labels;

/**
 * Считывает из БД и отправляет в Solomon метрики, которые записаны джобой
 * {@link BoidCheckMonitoringJob}
 * Значения обновляются раз в 5 минут, чтобы не слишком грузить ppcdict запросами.
 */
public class BoidCheckMonitoringMetricsProvider {
    private static final Logger logger = LoggerFactory.getLogger(BoidCheckMonitoringMetricsProvider.class);
    private static final int DELAY_PERIOD_MS = 30_000;
    private static final int REFRESH_PERIOD_MS = 5 * 60_000;

    private final List<YtCluster> ytClusters;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final Timer timer;
    private final Map<YtCluster, Metrics> values = new ConcurrentHashMap<>();
    private volatile boolean sensorsRegistered = false;

    private static class Metrics {
        int okCount;
        int failedCount;

        Metrics(int okCount, int failedCount) {
            this.okCount = okCount;
            this.failedCount = failedCount;
        }

        static Metrics emptyMetrics() {
            return new Metrics(0, 0);
        }
    }

    @Autowired
    public BoidCheckMonitoringMetricsProvider(DirectExportYtClustersParametersSource parametersSource,
                                              PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytClusters = parametersSource.getAllParamValues();
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.timer = new Timer("boidCheckMonitoringMetricsProvider-timer", true);
    }

    private void reloadValuesFromDb() {
        JobState jobState = BoidCheckMonitoringJob.loadJobState(ppcPropertiesSupport);
        for (YtCluster ytCluster : ytClusters) {
            if (jobState.getClusterStates().containsKey(ytCluster.getName())) {
                values.put(ytCluster, new Metrics(
                        jobState.getClusterStates().get(ytCluster.getName()).getLastCheckCountOk(),
                        jobState.getClusterStates().get(ytCluster.getName()).getLastCheckCountFailed()
                ));
            } else {
                values.put(ytCluster, Metrics.emptyMetrics());
            }
        }
    }

    @PostConstruct
    public void postConstruct() {
        // Инициализируем периодическое обновление значений
        timer.schedule(
                new TimerTask() {
                    @Override
                    public void run() {
                        try {
                            reloadValuesFromDb();
                            if (!sensorsRegistered) {
                                registerSensors();
                                sensorsRegistered = true;
                            }
                        } catch (Exception e) {
                            logger.error("Unexpected exception in timer thread", e);
                        }
                    }
                }, DELAY_PERIOD_MS, REFRESH_PERIOD_MS
        );
    }

    private void registerSensors() {
        // Регистрируем сенсоры
        for (YtCluster ytCluster : ytClusters) {
            SolomonUtils.SOLOMON_REGISTRY.lazyGaugeInt64("boids_clients_ok_count",
                    Labels.of("yt_cluster", ytCluster.getName()),
                    () -> values.getOrDefault(ytCluster, Metrics.emptyMetrics()).okCount);
            SolomonUtils.SOLOMON_REGISTRY.lazyGaugeInt64("boids_clients_failed_count",
                    Labels.of("yt_cluster", ytCluster.getName()),
                    () -> values.getOrDefault(ytCluster, Metrics.emptyMetrics()).failedCount);
        }
    }

    @PreDestroy
    public void preDestroy() {
        timer.cancel();
    }
}
