package ru.yandex.direct.jobs.videostatxlsreport;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.stream.Collectors.toList;

class VideoStatDataProcessor {
    List<String> header;
    List<Map<String, String>> data;

    private CampaignRepository campaignRepository;

    private BannerCommonRepository bannerCommonRepository;

    private ShardHelper shardHelper;

    private ClientRepository clientRepository;

    private Long clientId;

    // временно переводим на русский язык захардкоженным словарем
    private Map<String, String> headerTranslation;

    VideoStatDataProcessor(CampaignRepository campaignRepository, BannerCommonRepository bannerCommonRepository,
                           ShardHelper shardHelper, ClientRepository clientRepository, Long clientId) {
        this.campaignRepository = campaignRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.clientId = clientId;
        this.headerTranslation = ImmutableMap.<String, String>builder()
                .put("UpdateTime", "Дата")
                .put("OrderID", "Номер кампании")
                .put("BannerID", "Баннер")
                .put("CampaignName", "Название")
                .put("GoalID", "Цель")
                .put("VdCategory", "Аудитория")
                .put("Shows", "Показы")
                .put("Clicks", "Клики")
                .put("VdShows", "Показы ВД")
                .put("VdClicks", "Клики ВД")
                .put("GoalsNum", "Достижения цели")
                .put("Cost", "Расход")
                .put("ConversionRate", "Конверсия")
                .put("AvgClickCost", "Средняя цена клика")
                .put("AvgGoalCost", "Средняя цена цели")

                .build();
    }

    private Double getDouble(Map<String, String> values, String key) {
        if (!values.containsKey(key)) {
            return 0.0;
        }
        try {
            return Double.valueOf(values.getOrDefault(key, "0.0"));
        } catch (Exception e) {
            return 0.0;
        }
    }

    public List<String> getTranslatedHeader() {
        return header.stream().map(s -> headerTranslation.getOrDefault(s, s)).collect(toList());
    }

    void process(VideoReportData reportData) {
        data = new ArrayList<>();

        int clientShard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
        Currency clientCurr = Currencies.getCurrency(clientRepository.getWorkCurrency(clientShard,
                ClientId.fromLong(clientId)).getLiteral());

        List<String> headerOrder = new ArrayList<>();
        headerOrder.add("GoalID");
        headerOrder.add("OrderID");
        headerOrder.add("CampaignName");
        headerOrder.add("BannerID");
        headerOrder.add("UpdateTime");
        headerOrder.add("VdCategory");
        headerOrder.add("VdShows");
        headerOrder.add("VdClicks");
        headerOrder.add("Shows");
        headerOrder.add("Clicks");
        headerOrder.add("Cost");
        headerOrder.add("ConversionRate");
        headerOrder.add("AvgClickCost");
        headerOrder.add("AvgGoalCost");
        headerOrder.add("GoalsNum");

        header = new ArrayList<>();
        Set<String> headerSet = new HashSet<>(reportData.header);
        if (headerSet.contains("Cost")) {
            headerSet.add("AvgClickCost");
            headerSet.add("AvgGoalCost");
        }
        if (headerSet.contains("OrderID")) {
            headerSet.add("CampaignName");
        }

        header = headerOrder.stream().filter(headerSet::contains).collect(toList());

        Set<Long> orderIds = new HashSet<>();
        Multimap<Long, Long> orderBannerIds = ArrayListMultimap.create();

        for (List<String> row : reportData.data) {
            Map<String, String> rowMap = new HashMap<>();
            StreamEx.of(reportData.header).zipWith(StreamEx.of(row)).forKeyValue((key, val) -> {
                if (isNumericKey(key) && val.equals("")) {
                    val = "0";
                }
                if (key.equals("OrderID")) {
                    orderIds.add(Long.valueOf(val));
                }
                rowMap.put(key, val);
            });
            if (rowMap.containsKey("OrderID") && rowMap.containsKey("BannerID")) {
                Long orderId = Long.valueOf(rowMap.get("OrderID"));
                Long bannerId = Long.valueOf(rowMap.get("BannerID"));
                orderBannerIds.put(orderId, bannerId);
            }
            data.add(rowMap);
        }

        Map<Long, Long> orderIdToCid = new HashMap<>();
        Map<Long, Long> bannerIdToBid = new HashMap<>();
        Map<Long, CampaignSimple> campaignInfo = new HashMap<>();
        shardHelper.groupByShard(orderIds, ShardKey.ORDER_ID).forEach((shard, chunk) -> {
            Map<Long, Long> cids = campaignRepository.getCidsForOrderIds(shard, chunk);
            orderIdToCid.putAll(cids);
            Map<Long, CampaignSimple> chunkInfo = campaignRepository.getCampaignsSimple(shard, cids.values());
            campaignInfo.putAll(chunkInfo);
            List<Long> chunkBannerIds =
                    chunk.stream().flatMap(oid -> orderBannerIds.get(oid).stream()).collect(toList());
            if (!chunkBannerIds.isEmpty()) {
                bannerIdToBid.putAll(bannerCommonRepository.getBidsForBannerIds(chunkBannerIds, shard));
            }
        });

        for (Map<String, String> row : data) {
            if (row.containsKey("OrderID")) {
                try {
                    Long orderID = Long.valueOf(row.get("OrderID"));
                    Long cid = orderIdToCid.get(orderID);
                    row.put("OrderID", cid.toString());
                    row.put("CampaignName", campaignInfo.get(cid).getName());
                } catch (Exception e) {
                    row.put("OrderID", "-");
                    row.put("CampaignName", "-");
                }
            }

            if (row.containsKey("BannerID")) {
                Long bannerId = Long.valueOf(row.get("BannerID"));
                try {
                    row.put("BannerID", bannerIdToBid.get(bannerId).toString());
                } catch (NullPointerException e) {
                    row.put("BannerID", "-");
                }
            }

            if (row.containsKey("Clicks") && row.containsKey("Cost")) {
                Double cost = getDouble(row, "Cost");
                Double clicks = getDouble(row, "Clicks");
                Double avgClickCost = 0.0;
                if (clicks > 0) {
                    avgClickCost = cost / clicks;
                }
                row.put("AvgClickCost", avgClickCost.toString());
            }

            if (row.containsKey("GoalsNum") && row.containsKey("Cost")) {
                Double cost = getDouble(row, "Cost");
                Double goalsNum = getDouble(row, "GoalsNum");
                Double avgGoalCost = 0.0;
                if (goalsNum > 0) {
                    avgGoalCost = cost / goalsNum;
                }
                row.put("AvgGoalCost", avgGoalCost.toString());
            }

            if (row.containsKey("ConversionRate")) {
                Double conversionRate = getDouble(row, "ConversionRate");
                row.put("ConversionRate", String.format("%.2f", conversionRate));
            }

            if (row.containsKey("VdCategory")) {
                String val = row.get("VdCategory");
                if (val == null || val.equals("")) {
                    val = "0";
                }
                row.put("VdCategory", val.equals("0") ? "Не видели ВД" : "Видели ВД");
                if (val.equals("0")) {
                    row.put("VdShows", "-");
                    row.put("VdClicks", "-");
                }
            }

            formatMoney(row, "Cost", clientCurr);
            formatMoney(row, "AvgGoalCost", clientCurr);
            formatMoney(row, "AvgClickCost", clientCurr);
        }
    }

    boolean isNumericKey(String key) {
        Set<String> numericKeys = ImmutableSet.of("VdShows", "VdClicks", "Shows", "Clicks", "Cost",
                "ConversionRate", "GoalsNum", "AvgClickCost", "AvgGoalCost");
        return numericKeys.contains(key);
    }

    private void formatMoney(Map<String, String> row, String key, Currency clientCurr) {
        if (!row.containsKey(key)) {
            return;
        }
        String moneyStr = row.get(key);
        if (moneyStr == null || moneyStr.equals("")) {
            return;
        }
        Money money = Money.valueOf(moneyStr, clientCurr.getCode()).divide(1000000);
        row.put(key, money.roundToCentDown().bigDecimalValue().toString());
    }
}
