package ru.yandex.direct.jobs.videostatxlsreport;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.asynchttp.ParsableStringRequest;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.mdsfile.model.MdsFileSaveRequest;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.dbqueue.DbQueueJobType;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.asynchttp.ParsableStringRequest.DEFAULT_REQUEST_ID;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/*
 * curl -d '{"order_by":[{"dir":"asc","field":"UpdateTime"},{"field":"GoalID"}],\
 * "order_ids":["2713079","6337924","8423359","11952963","11952974","11953132","15550474","15550475","16131409"],\
 * "filters_pre":{"GoalID":{"ne":["3228496","2559811"]}},\
 * "date_from":"2018-02-01","date_to":"2018-02-10",\
 * "group_by":["UpdateTime","OrderID","BannerID","GoalID","VdCategory"],\
 * "countable_fields":["Shows","Clicks","GoalsNum"]}'
 * -H "Content-Type: application/json"
 * https://ikanazirskiy-direct.common-int.yandex-team.ru/direct-export-stat/postview/v1/report
 * <p>
 * <p>
 * {
 * "status": 0,
 * "header": ["UpdateTime", "OrderID", "BannerID", "GoalID", "VdCategory", "Shows", "Clicks", "GoalsNum"],
 * "total": 1000,
 * "query": "use hahn;\n$filename = Re2::Capture(\"[^/]+$\");\nselect UpdateTime, OrderID, BannerID, GoalID, VdCategory, sum(Shows) as Shows, sum(Clicks) as Clicks, sum(GoalReachesA2) as GoalsN
 * um\nfrom range(\"//home/yabs/stat/offline/direct-video-postview/prestable/auto_video_direct_postview_report/v2\", \"2018-02-01\", \"2018-02-10\")\nwhere OrderID in (2713079, 6337924, 8423359
 * , 11952963, 11952974, 11953132, 15550474, 15550475, 16131409)\nand GoalID not in (3228496, 2559811)\ngroup by $filename(TablePath())._0 as UpdateTime, OrderID, BannerID, GoalID, VdCategory\n
 * order by UpdateTime, GoalID\n",
 * "data": [
 * ["2018-02-05", 11952963, 2820541860, null, 0, 5, 0, null],
 * ["2018-02-05", 2713079, 4388823879, null, 0, 64, 0, null],
 * ......
 * ]
 * }
 */

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3), tags = {DIRECT_PRIORITY_2, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 1-59/10 * * * ?", needSchedule = TypicalEnvironment.class)
public class VideoStatXLSReport extends DirectShardedJob {

    private static final int JOB_RETRIES = 1;

    private static final Logger logger = LoggerFactory.getLogger(VideoStatXLSReport.class);

    private String reportUrlPostclick;
    private String reportUrlPostview;

    private ParallelFetcherFactory parallelFetcherFactory;

    @Autowired
    private DbQueueService dbQueueService;

    @Autowired
    private CampaignRepository campaignRepository;

    @Autowired
    private BannerCommonRepository bannerCommonRepository;

    @Autowired
    private ClientRepository clientRepository;

    @Autowired
    private ShardHelper shardHelper;

    @Autowired
    private MdsFileService mdsFileService;

    private static final DbQueueJobType<VideoReportArgs, VideoReportResult> JOB_TYPE =
            new DbQueueJobType<>("video_stat_report", VideoReportArgs.class, VideoReportResult.class);

    @Autowired
    public VideoStatXLSReport(AsyncHttpClient asyncHttpClient,
                              @Value("${video_stat_url_postclick}") String postclickUrl,
                              @Value("${video_stat_url_postview}") String postviewUrl) {
        parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient,
                new FetcherSettings().withConnectTimeout(Duration.ofMinutes(10))
                        .withGlobalTimeout(Duration.ofMinutes(10)).withRequestTimeout(Duration.ofMinutes(10)));
        reportUrlPostclick = postclickUrl;
        reportUrlPostview = postviewUrl;
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class VideoReportResult {
        @JsonProperty
        private String error;

        @JsonProperty
        private String path;

        VideoReportResult withError(String error) {
            this.error = error;
            return this;
        }

        VideoReportResult withPath(String path) {
            this.path = path;
            return this;
        }
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class VideoReportArgs {
        @JsonProperty
        private VideoReportParams params;

        @JsonProperty("report_type")
        private String reportType;

        @JsonProperty("report_name")
        private String reportName;
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class VideoReportParams {

        @JsonProperty("order_by")
        private List<OrderBy> orderBy;

        @JsonProperty("order_ids")
        private List<String> orderIds;

        @JsonProperty("filters_pre")
        private Map<String, Map<String, List<String>>> filtersPre;

        @JsonProperty("date_from")
        private String dateFrom;

        @JsonProperty("date_to")
        private String dateTo;

        @JsonProperty("group_by")
        private List<String> groupBy;

        @JsonProperty("countable_fields")
        private List<String> countableFields;


        String getRequestBody() {
            return toJson(this);
        }

        @JsonInclude(JsonInclude.Include.NON_NULL)
        private static class OrderBy {
            @JsonProperty("dir")
            private String direction;

            @JsonProperty
            private String field;
        }
    }

    private String getStatDataUrl(String reportType) {
        if (reportType == null || reportType.equals("") || reportType.equals("postclick")) {
            return reportUrlPostclick;
        } else if (reportType.equals("postview")) {
            return reportUrlPostview;
        } else {
            throw new IllegalArgumentException("Invalid report type " + reportType);
        }
    }

    private VideoReportData fetchVideoReportData(VideoReportArgs videoReportArgs) {
        String url = getStatDataUrl(videoReportArgs.reportType);
        try (ParallelFetcher<String> fetcher = parallelFetcherFactory
                .getParallelFetcherWithRequestTimeout(Duration.ofMinutes(10));
             TraceProfile ignore = Trace.current().profile("postview:report")) {
            RequestBuilder builder = new RequestBuilder().setUrl(url).setBody(videoReportArgs.params.getRequestBody())
                    .addHeader("Content-Type", "application/json").setMethod("POST").setRequestTimeout(6000000);
//            logger.debug(videoReportParams.getRequestBody());
            ParsableRequest<String> request = new ParsableStringRequest(DEFAULT_REQUEST_ID, builder.build());
            String reportJson = fetcher.executeWithErrorsProcessing(request).getSuccess();
            logger.debug(reportJson);
            return fromJson(reportJson, VideoReportData.class);
        } catch (Exception e) {
            logger.warn(e.getMessage());
            throw e;
        }
    }

    private <T> void xlsWriteRow(Row row, Collection<T> data, XSSFCellStyle cellStyle) {
        int i = 0;
        for (T val : data) {
            Cell cell = row.createCell(i++);
            cell.setCellStyle(cellStyle);
            cell.setCellValue((String) val);
        }
    }

    private void xlsWriteRow(Row dataRow, Map<String, String> dataLine, VideoStatDataProcessor statDataProcessor,
                             XSSFCellStyle cellStyle) {
        int i = 0;
        for (String h : statDataProcessor.header) {
            Cell cell = dataRow.createCell(i++);
            cell.setCellStyle(cellStyle);
            if (statDataProcessor.isNumericKey(h)) {
                try {
                    cell.setCellValue(Double.valueOf(dataLine.get(h)));
                } catch (NumberFormatException e) {
                    cell.setCellValue(dataLine.get(h));
                }
            } else {
                cell.setCellValue(dataLine.get(h));
            }
        }
    }

    private XSSFWorkbook buildXSLX(VideoReportData videoReportData, Long clientId) {
        XSSFWorkbook result = new XSSFWorkbook();
        Sheet sheet = result.createSheet();
        int rownum = 0;
        Row header = sheet.createRow(rownum++);
        Cell cell = header.createCell(0);
        cell.setCellValue("Статистика по кампаниям");
        Row columnsHeader = sheet.createRow(rownum++);
        sheet.createFreezePane(0, 2);

        VideoStatDataProcessor statData =
                new VideoStatDataProcessor(campaignRepository, bannerCommonRepository, shardHelper, clientRepository,
                        clientId);
        statData.process(videoReportData);

        XSSFCellStyle headerStyle = result.createCellStyle();
        XSSFFont headerFont = result.createFont();
        headerFont.setFontHeightInPoints((short) 14);
        headerFont.setBold(true);
        headerStyle.setFont(headerFont);

        XSSFFont dataLineFont = result.createFont();
        dataLineFont.setFontHeightInPoints((short) 14);

        XSSFCellStyle oddLineStyle = result.createCellStyle();
        XSSFColor lineColor = new XSSFColor(java.awt.Color.lightGray);
        oddLineStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        oddLineStyle.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.getIndex());
        oddLineStyle.setFont(dataLineFont);

        XSSFCellStyle evenLineStyle = result.createCellStyle();
        evenLineStyle.setFont(dataLineFont);

        xlsWriteRow(columnsHeader, statData.getTranslatedHeader(), headerStyle);

        for (Map<String, String> dataLine : statData.data) {
            Row dataRow = sheet.createRow(rownum++);
            XSSFCellStyle rowStyle = evenLineStyle;
            if (rownum % 2 == 1) {
                rowStyle = oddLineStyle;
            }
            xlsWriteRow(dataRow, dataLine, statData, rowStyle);
        }
        for (int i = 0; i < statData.header.size(); ++i) {
            sheet.autoSizeColumn(i);
        }
        return result;
    }

    @Override
    public void execute() {
        logger.info("START");
        while (dbQueueService
                .grabAndProcessJob(getShard(), JOB_TYPE, this::fetchAndGenerateReport, JOB_RETRIES, this::onError)) {
            try {
                Thread.sleep(Duration.ofSeconds(1).toMillis());
            } catch (Exception e) {
                logger.error(e.getMessage());
                break;
            }
        }
        logger.info("FINISH");
    }

    private VideoReportResult fetchAndGenerateReport(DbQueueJob<VideoReportArgs, VideoReportResult> job) {
        logger.debug("begin job # " + job.getId().toString());
        logger.debug(toJson(job.getArgs()));
        VideoReportData videoReportData = fetchVideoReportData(job.getArgs());
        if (videoReportData.status != 0) {
            logger.error("Report generation error: status = " + videoReportData.status);
            return new VideoReportResult().withError("Report generation error: status = " + videoReportData.status);
        }
        XSSFWorkbook xssfWorkbook = buildXSLX(videoReportData, job.getClientId().asLong());
        VideoReportResult result = new VideoReportResult();
        try {
            String path = storeXLSX(xssfWorkbook, job.getArgs().reportName, job.getClientId().asLong());
            return result.withPath(path);
        } catch (Exception e) {
            logger.error(e.getMessage());
            return result.withError(e.getMessage());
        }
    }

    private VideoReportResult onError(DbQueueJob<VideoReportArgs, VideoReportResult> job, String error) {
        return new VideoReportResult().withError(error);
    }

    private String storeXLSX(XSSFWorkbook xlsx, String reportName, Long clientId) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        // FileOutputStream fstream = new FileOutputStream(reportName + ".xlsx");
        xlsx.write(out);
        // xlsx.write(fstream);
        MdsFileSaveRequest saveRequest = new MdsFileSaveRequest(MdsStorageType.OFFLINE_STAT_REPORTS, out.toByteArray())
                .withCustomName(reportName);
        List<MdsFileSaveRequest> saveResult = mdsFileService.saveMdsFiles(singletonList(saveRequest), clientId);
        if (saveResult.size() != 1) {
            throw new RuntimeException("save result expected to be of length 1");
        }

        MdsFileSaveRequest result = saveResult.get(0);

        logger.debug("saved to mds: hash=" + result.getMd5Hash() + ", group=" + result.getMdsMetadata().getMdsKey()
                + ", name = " + reportName);
        return reportName;
    }

}
