package ru.yandex.direct.jobs.walletswarnings;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.validator.routines.EmailValidator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.eventlog.model.DaysLeftNotificationType;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.notification.LocaleResolver;
import ru.yandex.direct.core.entity.notification.repository.SmsQueueRepository;
import ru.yandex.direct.core.entity.statistics.model.Period;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.util.mail.EmailUtils;
import ru.yandex.direct.jobs.util.mail.MailEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderException;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.TimeProvider;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.DEFAULT_MONEY_WARNING_VALUE;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.STATUS_MAIL_NO_MAIL_SEND;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.STATUS_MAIL_ONE_DAY_WARN_SEND;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignRepository.STATUS_MAIL_THREE_DAYS_WARN_SEND;
import static ru.yandex.direct.core.entity.eventlog.model.DaysLeftNotificationType.ONE_DAY_REMAIN;
import static ru.yandex.direct.core.entity.eventlog.model.DaysLeftNotificationType.THREE_DAYS_REMAIN;
import static ru.yandex.direct.feature.FeatureName.NEW_WALLET_WARNINGS_ENABLED;
import static ru.yandex.direct.feature.FeatureName.SEND_SMS_FOR_NEW_WALLET_WARNINGS;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба отправляет клиентам письма и sms о том, что у них на общем счете осталось денег меньше чем:
 * 3 средних расхода за день
 * 1 средний расход за день
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        notifications = @OnChangeNotification(method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        ),
        needCheck = ProductionOnly.class,
        tags = {GROUP_INTERNAL_SYSTEMS}
)
@Hourglass(periodInSeconds = 2400, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class WalletsWarningsSenderJob extends DirectShardedJob {

    private static final long WEEK = 7;
    private static final int WALLETS_CHUNK_SIZE = 300;

    private static final Map<DaysLeftNotificationType, Long> NOTIFICATION_TYPE_TO_NEW_STATUS_MAIL = Map.of(
            THREE_DAYS_REMAIN, STATUS_MAIL_THREE_DAYS_WARN_SEND,
            ONE_DAY_REMAIN, STATUS_MAIL_ONE_DAY_WARN_SEND);

    private static final EmailValidator emailValidator = EmailValidator.getInstance();
    private static final Logger logger = LoggerFactory.getLogger(WalletsWarningsSenderJob.class);
    static final String LOGIN = "login";
    static final String CLIENT_ID = "ClientID";
    private final YandexSenderClient senderClient;
    private final OrderStatService orderStatService;
    private final CampaignService campaignService;
    private final CampaignRepository campaignRepository;
    private final SmsQueueRepository smsQueueRepository;
    private final UserService userService;
    private final FeatureService featureService;
    private final TranslationService translationService;
    private final EventLogService eventLogService;
    private final WalletsWarningsMailTemplateResolver walletsWarningsMailTemplateResolver;
    private final TimeProvider timeProvider = new TimeProvider();

    @Autowired
    public WalletsWarningsSenderJob(YandexSenderClient senderClient, OrderStatService orderStatService,
                                    CampaignService campaignService, CampaignRepository campaignRepository,
                                    UserService userService,
                                    FeatureService featureService,
                                    SmsQueueRepository smsQueueRepository,
                                    TranslationService translationService,
                                    EventLogService eventLogService,
                                    WalletsWarningsMailTemplateResolver walletsWarningsMailTemplateResolver) {
        this.orderStatService = orderStatService;
        this.senderClient = senderClient;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.userService = userService;
        this.featureService = featureService;
        this.smsQueueRepository = smsQueueRepository;
        this.translationService = translationService;
        this.eventLogService = eventLogService;
        this.walletsWarningsMailTemplateResolver = walletsWarningsMailTemplateResolver;
    }

    /**
     * Конструктор только для тестов. Используется для указания шарда
     */
    public WalletsWarningsSenderJob(int shard, YandexSenderClient senderClient, OrderStatService orderStatService,
                                    CampaignService campaignService, CampaignRepository campaignRepository,
                                    UserService userService,
                                    FeatureService featureService,
                                    SmsQueueRepository smsQueueRepository,
                                    TranslationService translationService,
                                    EventLogService eventLogService,
                                    WalletsWarningsMailTemplateResolver walletsWarningsMailTemplateResolver) {
        super(shard);
        this.orderStatService = orderStatService;
        this.senderClient = senderClient;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.userService = userService;
        this.featureService = featureService;
        this.eventLogService = eventLogService;
        this.smsQueueRepository = smsQueueRepository;
        this.translationService = translationService;
        this.walletsWarningsMailTemplateResolver = walletsWarningsMailTemplateResolver;
    }

    private static void logEvent(MailEvent event) {
        logger.info(JsonUtils.toJson(event));
    }

    @Override
    public void execute() {
        int shard = getShard();
        List<WalletCampaign> wallets = new ArrayList<>(campaignRepository.getWalletsForWarnSend(shard));
        logger.info("Got {} wallets", wallets.size());
        for (List<WalletCampaign> walletsChunk : ListUtils.partition(wallets, WALLETS_CHUNK_SIZE)) {
            logger.debug("Processing {} wallets chunk...", walletsChunk.size());
            // фильтруем кошельки - письма отправляем только при включенной у клиента фиче NEW_WALLET_WARNINGS_ENABLED
            Set<ClientId> clientIds = listToSet(walletsChunk, w -> ClientId.fromLong(w.getClientId()));
            Map<ClientId, Boolean> clientsFeatureStatus = featureService
                    .isEnabledForClientIdsOnlyFromDb(clientIds, NEW_WALLET_WARNINGS_ENABLED.getName());
            Map<ClientId, Boolean> clientIdsWithSendSmsPermission = featureService
                    .isEnabledForClientIdsOnlyFromDb(clientIds, SEND_SMS_FOR_NEW_WALLET_WARNINGS.getName());
            List<WalletCampaign> walletsWithEnabledFeature =
                    filterList(walletsChunk, r -> clientsFeatureStatus.get(ClientId.fromLong(r.getClientId())));
            logger.info("Got {} wallets with enabled feature", walletsWithEnabledFeature.size());
            if (walletsWithEnabledFeature.isEmpty()) {
                // пропускаем итерацию если нет клиентов со включенной фичей
                continue;
            }

            var walletCampaignIds = listToSet(walletsWithEnabledFeature, WalletCampaign::getId);
            var walletsWithCampaigns = campaignRepository
                    .getWalletsWithCampaignsByWalletCampaignIds(shard, walletCampaignIds, false);
            var campaigns = listToMap(campaignRepository.getCampaigns(shard, walletCampaignIds), Campaign::getId);
            var walletsRestMoney = campaignService.getWalletsRestMoneyByWalletCampaignIds(shard, walletCampaignIds);

            var userIdsWithWallets = mapList(walletsWithEnabledFeature, WalletCampaign::getUserId);
            Map<Long, User> uidToUser = listToMap(userService.massGetUser(userIdsWithWallets), User::getId);

            Map<Long, String> walletIdToEmail = campaignRepository.getEmailByCampaignIds(shard, walletCampaignIds);

            for (WalletCampaign walletCampaign : walletsWithEnabledFeature) {
                Long walletCampaignId = walletCampaign.getId();
                logger.debug("Processing wallet {}...", walletCampaignId);
                LocalDateTime now = timeProvider.now();
                List<Period> periodList = singletonList(new Period(
                        "week", now.minus(WEEK, ChronoUnit.DAYS).toLocalDate(), now.toLocalDate()));
                var moneyOutLimit = Money.valueOf(
                        Currencies.getCurrency(walletCampaign.getCurrency()).getMoneyOutLimit(),
                        walletCampaign.getCurrency());
                var orderIds = mapList(walletsWithCampaigns.getCampaignsBoundTo(walletCampaign), WalletCampaign::getOrderId);
                var spentMoney = orderStatService.getOrdersSumSpent(orderIds, periodList, walletCampaign.getCurrency())
                        .get("week");
                if (spentMoney.lessThanOrEqual(Money.valueOf(Currencies.EPSILON, spentMoney.getCurrencyCode()))) {
                    // сразу пропускаем кошельки которые не тратили деньги за последнюю неделю
                    continue;
                }
                var avgByOneDay = spentMoney.divide(WEEK);
                Money avgByThreeDays = avgByOneDay.multiply(BigDecimal.valueOf(3));

                var walletRestMoney = walletsRestMoney.get(walletCampaignId);
                logger.debug("Wallet {}: walletRestMoney={} {}, avgByOneDay={}, avgByThreeDays={}",
                        walletCampaignId, walletRestMoney.getRest(),
                        walletRestMoney.getRest().getCurrencyCode().name(), avgByOneDay, avgByThreeDays);
                var campaign = campaigns.get(walletCampaignId);
                boolean isDefaultMoneyWarningLimit = campaign
                        .getMoneyWarningValue().equals(DEFAULT_MONEY_WARNING_VALUE);
                if (isDefaultMoneyWarningLimit && walletRestMoney.getRest().greaterThan(moneyOutLimit)) {
                    var statusMail = campaign.getStatusMail().longValue();
                    var user = uidToUser.get(walletCampaign.getUserId());
                    if (walletRestMoney.getRest().lessThanOrEqual(avgByOneDay)
                            && (statusMail == STATUS_MAIL_NO_MAIL_SEND
                            || statusMail == STATUS_MAIL_THREE_DAYS_WARN_SEND)) {
                        //отправить письмо про один день
                        if (sendMail(shard, user, walletCampaign, ONE_DAY_REMAIN, walletIdToEmail)) {
                            sendSms(shard, user, walletCampaign, ONE_DAY_REMAIN,
                                    campaign.getSmsTime(), clientIdsWithSendSmsPermission);
                            saveToEventLog(walletCampaign, ONE_DAY_REMAIN,
                                    walletRestMoney.getRest());
                        }
                    } else if (walletRestMoney.getRest().lessThanOrEqual(avgByThreeDays)
                            && statusMail == STATUS_MAIL_NO_MAIL_SEND) {
                        //отправить письмо про три дня
                        if (sendMail(shard, user, walletCampaign, THREE_DAYS_REMAIN, walletIdToEmail)) {
                            sendSms(shard, user, walletCampaign, THREE_DAYS_REMAIN,
                                    campaign.getSmsTime(), clientIdsWithSendSmsPermission);
                            saveToEventLog(walletCampaign, THREE_DAYS_REMAIN,
                                    walletRestMoney.getRest());
                        }
                    }
                }
            }
        }
    }

    private boolean sendMail(int shard, User user, WalletCampaign walletCampaign,
                          DaysLeftNotificationType notificationType, Map<Long, String> walletIdToEmail)  {
        String walletEmail = walletIdToEmail.getOrDefault(walletCampaign.getId(), "");
        String email = EmailUtils.getValidEmail(user, walletEmail);
        if (email == null) {
            logEvent(MailEvent.badEmail(user.getClientId(), user, null));
            statusUpdate(shard, walletCampaign, notificationType);
            return true;
        }
        String campSlug = walletsWarningsMailTemplateResolver.resolveTemplateId(user.getLang(), notificationType);
        if (campSlug == null || campSlug.isEmpty()) {
            logEvent(MailEvent.emptySlug(user.getClientId(), user, email));
            statusUpdate(shard, walletCampaign, notificationType);
            return true;
        }
        YandexSenderTemplateParams templateParams = new YandexSenderTemplateParams.Builder()
                .withCampaignSlug(campSlug)
                .withToEmail(email)
                .withAsync(Boolean.TRUE)
                .withArgs(Map.of(LOGIN, user.getLogin(), CLIENT_ID, user.getClientId().toString()))
                .build();
        try {
            if (senderClient.sendTemplate(templateParams, YandexSenderClient::isInvalidToEmail)) {
                logEvent(MailEvent.mailSent(notificationType, user.getClientId(), user, campSlug));
            } else {
                logEvent(MailEvent.badEmail(user.getClientId(), user, email));
            }
            statusUpdate(shard, walletCampaign, notificationType);
        } catch (YandexSenderException e){
            logEvent(MailEvent.senderError(
                    notificationType, user.getClientId(), user, campSlug, e.getMessage()));
            return false;
        }
        return true;
    }

    /**
     * Обновляем статус о том, что письмо отправлено.
     * Статус может обновиться на отправленный и в том случае, когда email'а нет или письмо не ушло, но ошибки не было.
     * Статус не меняется только в случае YandexSenderException.
     * @param notificationType тип отправленного письма
     */
    private void statusUpdate(int shard, WalletCampaign walletCampaign, DaysLeftNotificationType notificationType){
        Long newStatusMail = NOTIFICATION_TYPE_TO_NEW_STATUS_MAIL.get(notificationType);
        Long oldStatusMail = walletCampaign.getStatusMail().longValue();
        campaignRepository.setStatusMail(
                shard, singletonList(walletCampaign.getId()), newStatusMail, oldStatusMail);
    }

    private void sendSms(int shard, User user,
                         WalletCampaign walletCampaign,
                         DaysLeftNotificationType notificationType,
                         TimeInterval timeInterval,
                         Map<ClientId, Boolean> clientIdsWithSendSmsPermission) {
        if (!Boolean.TRUE.equals(clientIdsWithSendSmsPermission.get(ClientId.fromLong(walletCampaign.getClientId())))) {
            return;
        }

        var message = translationService
                .translate(notificationType == ONE_DAY_REMAIN
                                ? WalletsWarningsTranslations.INSTANCE.oneDayRemain()
                                : WalletsWarningsTranslations.INSTANCE.treeDaysRemain(),
                        LocaleResolver.getLocaleByLanguageWithFallback(user.getLang()));

        smsQueueRepository.addToSmsQueue(shard, user.getUid(), walletCampaign.getId(), message,
                SmsFlag.ACTIVE_ORDERS_MONEY_WARNING_SMS, timeInterval);
    }

    // cохранить событие о том, что на счете осталось средств на 1/3 дня
    private void saveToEventLog(WalletCampaign walletCampaign,
                                DaysLeftNotificationType notificationType,
                                Money restMoney) {
        eventLogService.addMoneyWarningForDaysEventLog(walletCampaign.getId(),
                walletCampaign.getClientId(),
                notificationType,
                restMoney);
    }
}
