package ru.yandex.direct.jobs.warnclientdomains;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.core.entity.banner.model.BidGroup;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.client.model.ClientDomainStripped;
import ru.yandex.direct.core.entity.client.repository.ClientDomainsStrippedRepository;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.NewDomainMailNotification;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.rbac.ClientPerminfo;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.FieldExtremums;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

/**
 * Отправка уведомлений по новым доменам на клиентских аккаунтах
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 1, minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS})
@Hourglass(periodInSeconds = 1200, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class JobSendWarnClientDomains extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(JobSendWarnClientDomains.class);

    private static final String LAST_RECORD_ID_TEMPLATE = "PrepareClientDomains_last_send_record_id_shard_%s";
    private static final String LAST_RECORD_TIME_TEMPLATE = "PrepareClientDomains_last_send_time_shard_%s";
    static final DateTimeFormatter DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    private static final LocalDateTime DEFAULT_RECORD_DATE_TIME = LocalDateTime.of(2000, 1, 1, 0, 0, 0, 0);
    /**
     * Минимальная задержка между отправками писем
     */
    private static final Duration DELAY_INTERVAL = Duration.ofMinutes(20);
    /**
     * Выбирать домены с датой появления не позднее, чем N часов назад
     */
    private static final Duration HOURS_TO_ANALYSE = Duration.ofHours(12);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ClientDomainsStrippedRepository clientDomainsStrippedRepository;
    private final DomainRepository domainRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final RbacService rbacService;
    private final PpcRbac ppcRbac;
    private final UserService userService;
    private final NotificationService notificationService;

    @Autowired
    public JobSendWarnClientDomains(PpcPropertiesSupport ppcPropertiesSupport,
                                    ClientDomainsStrippedRepository clientDomainsStrippedRepository,
                                    DomainRepository domainRepository,
                                    BannerRelationsRepository bannerRelationsRepository,
                                    RbacService rbacService, PpcRbac ppcRbac,
                                    UserService userService,
                                    NotificationService notificationService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.clientDomainsStrippedRepository = clientDomainsStrippedRepository;
        this.domainRepository = domainRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.rbacService = rbacService;
        this.ppcRbac = ppcRbac;
        this.userService = userService;
        this.notificationService = notificationService;
    }

    JobSendWarnClientDomains(int shard,
                             PpcPropertiesSupport ppcPropertiesSupport,
                             ClientDomainsStrippedRepository clientDomainsStrippedRepository,
                             DomainRepository domainRepository,
                             BannerRelationsRepository bannerRelationsRepository,
                             RbacService rbacService, PpcRbac ppcRbac,
                             UserService userService,
                             NotificationService notificationService
    ) {
        super(shard);
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.clientDomainsStrippedRepository = clientDomainsStrippedRepository;
        this.domainRepository = domainRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.rbacService = rbacService;
        this.ppcRbac = ppcRbac;
        this.userService = userService;
        this.notificationService = notificationService;
    }

    @Override
    public void execute() {
        makeJob();
    }

    private void makeJob() {
        DomainsBundle recentDomains = getRecentData(LocalDateTime.now());

        if (recentDomains.getDomainsToClient().isEmpty()) {
            logger.info("No new domains found");
        } else {
            logger.info("Found {} domains for notification", recentDomains.getDomainsToClient().size());
            sendNotifications(recentDomains);
            updateProperties(recentDomains);
        }
    }

    private void sendNotifications(DomainsBundle recentDomains) {
        Collection<NewDomainMailNotification> notifications = prepareNotifications(recentDomains.getDomainsToClient());
        for (NewDomainMailNotification notification : notifications) {
            notificationService.addNotification(notification);
        }
    }

    void updateProperties(DomainsBundle recentDomains) {
        Long maxRecordId = recentDomains.getMaxRecordId();
        LocalDateTime maxLogTime = recentDomains.getMaxLogtime();
        logger.info("Updating last_send_record_id: {}, last_send_time: {}", maxRecordId, maxLogTime);
        ppcPropertiesSupport.get(getPropertyIdName()).set(maxRecordId);
        ppcPropertiesSupport.get(getPropertyTimeName()).set(maxLogTime);
    }

    DomainsBundle getRecentData(LocalDateTime now) {
        long lastRecordId = ppcPropertiesSupport.get(getPropertyIdName()).getOrDefault(0L);

        LocalDateTime lastRecordTime =
                ppcPropertiesSupport.get(getPropertyTimeName()).getOrDefault(DEFAULT_RECORD_DATE_TIME);

        logger.info("Starting sending mails with last_send_record_id: {}, last_send_time: {}", lastRecordId,
                lastRecordTime);

        if (lastRecordTime.plus(DELAY_INTERVAL).isAfter(now)) {
            return new DomainsBundle(Collections.emptyList());
        } else {
            Collection<ClientDomainStripped> domains =
                    clientDomainsStrippedRepository.getRecent(getShard(), lastRecordId,
                            now.minus(HOURS_TO_ANALYSE));
            return new DomainsBundle(domains);
        }
    }

    Collection<NewDomainMailNotification> prepareNotifications(
            Map<Long, Set<ClientDomainStripped>> domainsToClient) {
        Collection<NewDomainMailNotification> notifications = new ArrayList<>();

        // Перебираем клиентов
        for (Map.Entry<Long, Set<ClientDomainStripped>> entry : domainsToClient.entrySet()) {
            ClientId clientId = ClientId.fromLong(entry.getKey());

            Optional<ClientPerminfo> clientPerminfo = ppcRbac.getClientPerminfo(clientId);
            if (!clientPerminfo.isPresent()) {
                logger.error("Client with id {} not found", clientId);
                continue;
            }

            Set<Long> agencyUids = StreamEx.of(clientPerminfo.get().agencyUids())
                    .filter(CommonUtils::isValidId)
                    .toSet();
            List<User> agencies = userService.massGetUser(agencyUids)
                    .stream().filter(a -> a.getOpts().contains("notify_about_new_domains"))
                    .collect(Collectors.toList());

            if (agencies.isEmpty()) {
                continue;
            }

            List<Long> domainIds = mapList(entry.getValue(), ClientDomainStripped::getDomainId);
            List<Domain> domains = domainRepository.getDomainsByIdsFromDict(domainIds);

            // Список инвертированных доменов по клиенту
            List<String> reversedDomains = mapList(domains, Domain::getReverseDomain);

            Collection<BidGroup> bidGroups = bannerRelationsRepository.getBidGroups(getShard(),
                    reversedDomains, rbacService.getClientRepresentativesUids(clientId));

            Collection<Long> clientRepresentativesUids = rbacService.getClientRepresentativesUids(clientId);

            if (bidGroups.isEmpty()) {
                List<String> fioList = mapList(userService.massGetUser(Set.copyOf(clientRepresentativesUids)),
                        User::getFio);
                List<String> domainList = mapList(domains, Domain::getDomain);
                Set<List<String>> allPossibleDomainAndFioPairs = Sets.cartesianProduct(Set.copyOf(domainList),
                        Set.copyOf(fioList));
                bidGroups = mapSet(allPossibleDomainAndFioPairs,
                        pair -> new BidGroup().withDomain(pair.get(0)).withFio(pair.get(1)));
            }
            Collection<BidGroup> finalBidGroups = bidGroups;
            agencies.forEach(agency -> notifications.add(new NewDomainMailNotification()
                    .withBidGroups(finalBidGroups).withAgencyUserId(agency.getUid()).withFio(agency.getFio())));
        }

        return notifications;
    }

    PpcPropertyName<LocalDateTime> getPropertyTimeName() {
        return new PpcPropertyName<>(String.format(LAST_RECORD_TIME_TEMPLATE, getShard()),
                PpcPropertyType.LOCAL_DATE_TIME);
    }

    PpcPropertyName<Long> getPropertyIdName() {
        return new PpcPropertyName<>(String.format(LAST_RECORD_ID_TEMPLATE, getShard()), PpcPropertyType.LONG);
    }

    /**
     * Класс-контейнер с информацией о пачке данных. Содержит словарь доменов и максимальные значения recordId/logtime
     */
    static class DomainsBundle {
        private final Map<Long, Set<ClientDomainStripped>> domainsToClient;
        private final FieldExtremums<ClientDomainStripped, Long> recordIdExtremums;
        private final FieldExtremums<ClientDomainStripped, LocalDateTime> logtimeExtremums;

        private DomainsBundle(Collection<ClientDomainStripped> domains) {
            recordIdExtremums = new FieldExtremums<>(ClientDomainStripped::getRecordId);
            logtimeExtremums = new FieldExtremums<>(ClientDomainStripped::getLogtime);
            domainsToClient = domains.stream().peek(recordIdExtremums).peek(logtimeExtremums)
                    // Преобразование к Map<Long, Set> делается для удобства сравнения при тестировании
                    .collect(groupingBy(ClientDomainStripped::getClientId, toSet()));
        }

        Map<Long, Set<ClientDomainStripped>> getDomainsToClient() {
            return domainsToClient;
        }

        Long getMaxRecordId() {
            return recordIdExtremums.getMax();
        }

        LocalDateTime getMaxLogtime() {
            return logtimeExtremums.getMax();
        }
    }
}
