package ru.yandex.direct.jobs.warnplace;

import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.warnplace.repository.WarnplaceRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Очищает таблицу {@link ru.yandex.direct.dbschema.ppc.tables.Warnplace}
 * от записей старше {@value BORDER_DAYS} дней
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4),
        tags = {DIRECT_PRIORITY_2, GROUP_INTERNAL_SYSTEMS, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 8 22 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class WarnplaceCleaner extends DirectShardedJob {

    private static final int BORDER_DAYS = 7;

    private static final Logger logger = LoggerFactory.getLogger(WarnplaceCleaner.class);

    private final WarnplaceRepository warnplaceRepository;

    @Autowired
    public WarnplaceCleaner(WarnplaceRepository warnplaceRepository) {
        this.warnplaceRepository = warnplaceRepository;
    }

    @Override
    public void execute() {
        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(BORDER_DAYS);
        logger.info("delete outdated records from border time {}", borderDateTime);
        int deletedRows = warnplaceRepository.deleteRecordsOlderThan(getShard(), borderDateTime);
        logger.info("deleted {} records", deletedRows);
    }
}
