package ru.yandex.direct.jobs.xiva;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.xiva.XivaPushTypeInfo;
import ru.yandex.direct.core.entity.xiva.XivaPushesQueueService;
import ru.yandex.direct.core.entity.xiva.model.XivaPushesQueueItem;
import ru.yandex.direct.core.entity.xiva.model.XivaPushesQueuePushType;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.xiva.client.XivaClient;
import ru.yandex.direct.xiva.client.model.Push;
import ru.yandex.direct.xiva.client.model.Recipient;
import ru.yandex.direct.xiva.client.model.RecipientUser;
import ru.yandex.direct.xiva.client.model.SendStatus;
import ru.yandex.direct.xiva.client.model.SendStatusList;
import ru.yandex.direct.xiva.client.model.SendStatusSingleOrList;


@JugglerCheck(
        needCheck = ProductionOnly.class,
        ttl = @JugglerCheck.Duration(minutes = 10),
        notifications = @OnChangeNotification(
                method = NotificationMethod.TELEGRAM,
                recipient = NotificationRecipient.LOGIN_BUHTER,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 1)
public class SendPushesJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(SendPushesJob.class);

    private static final int BATCH_SIZE = 10000;

    private XivaPushesQueueService xivaPushesQueueService;
    private XivaClient xivaClient;
    private int shard;

    @Autowired
    public SendPushesJob(XivaPushesQueueService xivaPushesQueueService, XivaClient xivaClient) {
        this.xivaPushesQueueService = xivaPushesQueueService;
        this.xivaClient = xivaClient;
    }

    SendPushesJob(int shard, XivaPushesQueueService xivaPushesQueueService, XivaClient xivaClient) {
        super(shard);
        this.xivaPushesQueueService = xivaPushesQueueService;
        this.xivaClient = xivaClient;
    }

    @Override
    public void execute() {
        shard = getShard();
        List<XivaPushesQueueItem> queueItems = xivaPushesQueueService.getPushes(shard, BATCH_SIZE);
        List<XivaPushesQueueItem> toDelete = extractOldPushes(queueItems);
        if (!toDelete.isEmpty()) {
            logger.info("Found {} outdated pushes", toDelete.size());
        }

        HashMap<XivaPushesQueuePushType, List<Long>> groupsToSend = groupByPushType(queueItems);
        for (XivaPushesQueuePushType pushType : groupsToSend.keySet()) {
            Push push = new Push(pushType.toString(), pushType.toString());
            List<Long> clients = groupsToSend.get(pushType);
            List<Recipient> recipients = getRecipients(clients);
            List<SendStatusSingleOrList> statuses = xivaClient.sendBatch(recipients, push, null);
            if (statuses != null) {
                toDelete.addAll(getSuccessfullySentPushes(pushType, clients, statuses));
            }
        }

        xivaPushesQueueService.deletePushesFromQueue(shard, toDelete);
    }

    private List<XivaPushesQueueItem> extractOldPushes(List<XivaPushesQueueItem> queueItems) {
        List<XivaPushesQueueItem> old = new LinkedList<>();
        LocalDateTime now = LocalDateTime.now();
        for (Iterator<XivaPushesQueueItem> iterator = queueItems.iterator(); iterator.hasNext();) {
            XivaPushesQueueItem item = iterator.next();
            int ttl = XivaPushTypeInfo.getTTL(item.getPushType());
            if (item.getAddTime().isBefore(now.minusSeconds(ttl))) {
                old.add(item);
                iterator.remove();
            }
        }
        return old;
    }

    private HashMap<XivaPushesQueuePushType, List<Long>> groupByPushType(List<XivaPushesQueueItem> items) {
        HashMap<XivaPushesQueuePushType, List<Long>> groups = new HashMap();
        for (XivaPushesQueueItem item : items) {
            //тут раскидываем по типам пушей = батчам
            if (!groups.containsKey(item.getPushType())) {
                groups.put(item.getPushType(), new LinkedList<>());
            }
            groups.get(item.getPushType()).add(item.getClientId());
        }
        return groups;
    }

    private List<Recipient> getRecipients(List<Long> clients) {
        List<Recipient> recipients = new LinkedList<>();
        for (Long client : clients) {
            recipients.add(new RecipientUser(client.toString()));
        }
        return recipients;
    }

    private List<XivaPushesQueueItem> getSuccessfullySentPushes(
            XivaPushesQueuePushType pushType, List<Long> clients, List<SendStatusSingleOrList> statuses) {
        List<XivaPushesQueueItem> successful = new LinkedList<>();
        // проверяем надо ли пробовать переотправить
        // если не надо, отправляем пуш на удаление из очереди
        for (var i = 0; i < statuses.size(); i++) {
            SendStatusSingleOrList statusSingleOrList = statuses.get(i);
            var needResending = false;
            if (statusSingleOrList.isSingle()) {
                SendStatus status = (SendStatus) statusSingleOrList;
                needResending = checkStatusIfNeedResending(status, clients.get(i));
            }
            if (statusSingleOrList.isList()) {
                SendStatusList list = (SendStatusList) statusSingleOrList;
                for (SendStatus status : list.getList()) {
                    needResending = needResending || checkStatusIfNeedResending(status, clients.get(i));
                }
            }
            if (!needResending) {
                successful.add(
                        new XivaPushesQueueItem()
                                .withPushType(pushType)
                                .withClientId(clients.get(i))
                );
            }
        }
        return successful;
    }

    /**
     * Определяем, нужно ли перепослать пуш. Есть статусы "не ок", при которых бесполезно
     * пробовать перепослать, в этом случае пуш останется в очереди пока не устреет.
     * <p>
     * Ошибочные статусы пытаемся залогировать.
     *
     * @see <a href="https://console.push.yandex-team.ru/#api-reference-batch-send">Описание статусов</a>
     * @see XivaPushTypeInfo#getTTL(XivaPushesQueuePushType)
     */
    private Boolean checkStatusIfNeedResending(SendStatus status, Long clientId) {
        if (status.getCode() == 200) {
            return false;
        }
        if (status.getCode() < 400) {
            // Статусы вроде как успешные, но до получателя пуш не дошёл
            // Например, подписки отсутствуют или сообщение отклонено фильтром подписки
            logger.warn("Warn on sending push to client {}, code {}", clientId, status.getCode());
            return false;
        }
        logger.info("Cannot send push to client {}, error 400, cause {}", clientId, status.getBody());
        return true;
    }
}
