package ru.yandex.direct.jobs.xlshistory;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.mdsfile.model.MdsStorageType;
import ru.yandex.direct.core.entity.mdsfile.service.MdsFileService;
import ru.yandex.direct.core.entity.xlshistory.repository.XlsHistoryRepository;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassStretchPeriod;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Чистит историю загруженных/выгруженных через xls кампаний.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 3),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS}
)
// Запускается ежедневно в 3:35 (смещена от ppcClearXlsHistory.pl)
@Hourglass(cronExpression = "0 35 3 * * ?", needSchedule = TypicalEnvironment.class)
@HourglassStretchPeriod()
@ParametersAreNonnullByDefault
class ClearXlsJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(ClearXlsJob.class);

    private static final int XLS_EXPIRES_DAYS = 90;
    private static final int LIMIT_XLS_HISTORY_ROWS = 1000;
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(3.0);

    private final XlsHistoryRepository xlsHistoryRepository;
    private final MdsFileService mdsFileService;

    @Autowired
    public ClearXlsJob(XlsHistoryRepository xlsHistoryRepository, MdsFileService mdsFileService) {
        this.xlsHistoryRepository = xlsHistoryRepository;
        this.mdsFileService = mdsFileService;
    }

    ClearXlsJob(int shard, XlsHistoryRepository xlsHistoryRepository, MdsFileService mdsFileService) {
        super(shard);
        this.xlsHistoryRepository = xlsHistoryRepository;
        this.mdsFileService = mdsFileService;
    }

    @Override
    public void execute() {

        LocalDateTime borderDateTime = LocalDateTime.now().minusDays(XLS_EXPIRES_DAYS);

        int countXlsHistoryDeleted = 0;
        // макс. 1000 итераций для предотвращения бесконечного цикла (на всякий случай)
        int counter = 1000;
        while (counter-- > 0) {
            List<Long> xlsHistoryIds = xlsHistoryRepository.getIdsByLogdateLessThan(getShard(), borderDateTime,
                    LIMIT_XLS_HISTORY_ROWS);
            if (xlsHistoryIds.isEmpty()) {
                break;
            }
            countXlsHistoryDeleted += RELAXED_WORKER.callAndRelax(() ->
                    xlsHistoryRepository.deleteById(getShard(), xlsHistoryIds));
        }
        logger.info("deleted {} outdated rows from xls_history by logdate less than {}", countXlsHistoryDeleted,
                borderDateTime);

        mdsFileService.deleteOldMdsFiles(getShard(), borderDateTime, MdsStorageType.XLS_HISTORY);
    }
}
