package ru.yandex.direct.jobs.ydbreplication;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.jooq.EnumType;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.util.RepositoryUtils;

@Service
@ParametersAreNonnullByDefault
public class YdbMapperService {

    private final ZoneId zoneId;

    public YdbMapperService(ZoneId zoneId) {
        this.zoneId = zoneId;
    }

    public Value convertToLocalDateTime(Object value) {
        LocalDateTime localDateTimeValue = (LocalDateTime) value;
        Instant instant = localDateTimeValue.atZone(zoneId).toInstant();
        return PrimitiveValue.datetime(instant);
    }

    public OptionalValue convertToOptionalBoolean(Boolean boolValue) {
        OptionalType optionalType = OptionalType.of(PrimitiveType.bool());
        if (boolValue == null) {
            return optionalType.emptyValue();
        }
        return optionalType.newValue(PrimitiveValue.bool(boolValue));
    }

    public OptionalValue convertToOptionalUint64(Long longValue) {
        OptionalType optionalType = OptionalType.of(PrimitiveType.uint64());
        if (longValue == null) {
            return optionalType.emptyValue();
        }
        return optionalType.newValue(PrimitiveValue.uint64(longValue));
    }

    public OptionalValue convertToOptionalUint32(Long longValue) {
        OptionalType optionalType = OptionalType.of(PrimitiveType.uint32());
        if (longValue == null) {
            return optionalType.emptyValue();
        }
        Integer intValue = RepositoryUtils.intFromLong(longValue);
        return optionalType.newValue(PrimitiveValue.uint32(intValue));
    }

    public OptionalValue convertToOptionalUtf8(EnumType parType) {
        OptionalType optionalType = OptionalType.of(PrimitiveType.utf8());
        if (parType == null) {
            return optionalType.emptyValue();
        }
        String parTypeString = parType.getLiteral();
        return optionalType.newValue(PrimitiveValue.utf8(parTypeString));
    }

    public OptionalValue convertToOptionalDouble(Double floatValue) {
        OptionalType optionalType = OptionalType.of(PrimitiveType.float64());
        if (floatValue == null) {
            return optionalType.emptyValue();
        }
        return optionalType.newValue(PrimitiveValue.float64(floatValue));
    }
}
