package ru.yandex.direct.jobs.yt;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtUtils.isClusterAvailable;

/**
 * Template for job that parses table with logs by executing YQL-query and creates static yt tables with results.
 * Inserts into new table are supposed to be done by YQL-query.
 * Each run of this job processes one table with logs.
 * Each table with logs is read only once and won't be re-read again.
 * List of clusters is fixed in the configuration file.
 * The cluster to read logs table is chosen independently of last processed log time.
 */
public abstract class ExecuteYQLAndExportDataJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(ExecuteYQLAndExportDataJob.class);

    private final YtProvider ytProvider;
    private final PpcProperty<String> clusterProperty;
    private final List<YtCluster> validClusters;
    private final PpcProperty<LocalDateTime> lastProcessedLogTimeProperty;
    private final String targetTablesFolder;
    private final String yqlQuery;

    protected ExecuteYQLAndExportDataJob(
            DirectConfig directConfig,
            YtProvider ytProvider,
            PpcProperty<String> clusterProperty,
            PpcProperty<LocalDateTime> lastProcessedLogTimeProperty,
            String yqlPath
    ) {
        DirectConfig config = directConfig.getBranch(getConfigBranch());

        this.targetTablesFolder = config.getString(getOutputTablesFolder());
        this.validClusters = getClusters(config);

        this.ytProvider = ytProvider;

        this.clusterProperty = clusterProperty;
        this.lastProcessedLogTimeProperty = lastProcessedLogTimeProperty;
        yqlQuery = String.join("\n",
                new ClassPathResourceInputStreamSource(yqlPath).readLines()
        );
    }

    @Override
    public void execute() {
        YtCluster cluster = getCluster();

        LocalDateTime lastProcessedLogTime = lastProcessedLogTimeProperty.get();
        LocalDateTime logTimeToProcess = getLogTimeToProcess(lastProcessedLogTime);
        if (logTimeToProcess == null) {
            logger.info("No work to do");
            return;
        }

        boolean success = runQuery(cluster, logTimeToProcess);

        if (success) {
            lastProcessedLogTimeProperty.cas(lastProcessedLogTime, logTimeToProcess);
        }
    }

    protected YtCluster getCluster() {
        String clusterRaw = clusterProperty.get();
        YtCluster cluster = YtCluster.parse(clusterRaw);

        if (!validClusters.contains(cluster)) {
            throw new IllegalArgumentException(String.format("%s is not a valid cluster", clusterRaw));
        }

        if (!isClusterAvailable(ytProvider, cluster)) {
            logger.warn("Cluster {} is not available", cluster);
            cluster = switchCluster(cluster);
        }

        return cluster;
    }

    /**
     * В первый раз переменная lastProcessedLogTime должна устанавливаться вручную через переменную окружения
     */
    @Nullable
    protected LocalDateTime getLogTimeToProcess(LocalDateTime lastProcessedLogTime) {
        if (lastProcessedLogTime == null) {
            logger.warn("No last processed log time");
            return null;
        }

        LocalDateTime logTimeToProcess = lastProcessedLogTime.plusDays(1);

        if (logTimeToProcess.isAfter(LocalDateTime.now())) {
            logger.info("Log time to process is later than now");
            return null;
        }

        return logTimeToProcess;
    }

    protected boolean runQuery(YtCluster cluster, LocalDateTime logTimeToProcess) {
        YtOperator ytOperator = ytProvider.getOperator(cluster, YtSQLSyntaxVersion.SQLv1);
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(cluster);

        String logDateToProcess = logTimeToProcess.toLocalDate().toString();
        List<String> sourceTables = StreamEx.of(getSourceTablesPathTemplate())
                .map(t -> String.format(t, logDateToProcess))
                .toList();
        String tableToExportData = YtPathUtil.generatePath(
                ytClusterConfig.getHome(), relativePart(), targetTablesFolder, logTimeToProcess.toString()
        );

        logger.info("Trying to process tables {} in cluster {} to table {}", sourceTables, cluster, tableToExportData);

        boolean sourceTablesAreReady =
                StreamEx.of(sourceTables).allMatch(sourceTable -> checkSourceTable(ytOperator, sourceTable));
        if (!sourceTablesAreReady) {
            logger.warn("Source tables is not ready");
            return false;
        }

        Object[] args = StreamEx.of(sourceTables)
                .append(tableToExportData)
                .toArray(Object[]::new);
        ytOperator.yqlExecute(yqlQuery, args);
        return true;
    }

    protected boolean checkSourceTable(YtOperator ytOperator, String sourceTable) {
        boolean tableExists = ytOperator.getYt().cypress().exists(YPath.simple(sourceTable));
        if (!tableExists) {
            logger.warn("Source table {} does not exist", sourceTable);
            return false;
        }

        long tableSize = ytOperator.readTableRowCount(new YtTable(sourceTable));
        if (tableSize == 0) {
            logger.error("Source table {} is empty", sourceTable);
            return false;
        }

        return true;
    }

    protected YtCluster switchCluster(YtCluster current) {
        int ind = validClusters.indexOf(current);
        YtCluster newCluster = validClusters.get((ind + 1) % validClusters.size());
        clusterProperty.set(newCluster.getName());
        return newCluster;
    }

    protected List<YtCluster> getClusters(DirectConfig config) {
        return mapList(config.getStringList("clusters"), YtCluster::parse);
    }

    protected abstract List<String> getSourceTablesPathTemplate();

    protected abstract String getConfigBranch();

    protected abstract String getOutputTablesFolder();
}
