package ru.yandex.direct.jobs.yt;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Stopwatch;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.transfermanagerutils.TransferManager;
import ru.yandex.direct.transfermanagerutils.TransferManagerConfig;
import ru.yandex.direct.transfermanagerutils.TransferManagerJobConfig;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.attributes.OptimizeForAttr;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.common.YtTimestamp;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.specs.MergeMode;
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static java.util.Comparator.comparing;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.MYSQL_YT_SYNC_BACKUP_JOB_ENABLED;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.DEFAULT_YT_CLUSTER;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;
import static ru.yandex.direct.ytwrapper.YtUtils.getRecursiveUsageByMediums;
import static ru.yandex.direct.ytwrapper.YtUtils.getRemainingSpaceByMediums;

/**
 * Выполняет бакап таблиц синхронизатора mysql->yt
 * Что делаем:
 * 1. Выбираем кластер с наиболее свежими данными
 * 2. Операцией merge выполняем копирование mysql-sync-states в статическую таблицу.
 * Гарантий на согласованность между ключами в таком способе копирования нет, но там на каждый шард лежит ровно по
 * одной записи.
 * При этом синхронизатор может что-то дописывать в таблицы с содержимым, но gtid_executed уже не будут изменены,
 * и бакап будет содержать все данные на момент копирования первой таблицы плюс некоторые изменения после.
 * При восстановлении бакапа нас это устраивает, так как мы будем на бекапе прогонять бинлоги после этой точки.
 * 3. Для каждой таблицы с содержимым фиксируем last_commit_timestamp и ожидаем флаша этой транзакции на диск.
 * Это происходит тогда, когда unflushed_timestamp становится строго больше зафиксированного last_commit_timestamp.
 * 4. Выполняем копирование всех таблиц операциями merge. На выходе получаем набор статических таблиц.
 * 5. Заказываем transfer на hahn в директорию с TTL=30 days.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 72), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_2, YT,
        DIRECT_PRODUCT_TEAM})
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 48), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_2, YT})
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = NonProductionEnvironment.class)
@ParametersAreNonnullByDefault
public class MysqlYtSyncBackupJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(MysqlYtSyncBackupJob.class);

    /**
     * Директория, которая будет использована в кластерах для складывания туда бакапов
     */
    private static final String BACKUP_DIRECTORY = "mysql-sync-backup";

    private static final Duration WAIT_FOR_FLUSH_TIMEOUT = Duration.ofMinutes(30);

    private final MysqlYtSyncBackupConfig jobConfig;
    private final DirectYtDynamicConfig dynamicConfig;
    private final YtProvider ytProvider;
    private final YtCluster dstYtCluster;
    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;
    private final TransferManager transferManager;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public MysqlYtSyncBackupJob(
            MysqlYtSyncBackupConfig jobConfig,
            DirectYtDynamicConfig dynamicConfig,
            YtProvider ytProvider,
            @Qualifier(DEFAULT_YT_CLUSTER) YtCluster dstYtCluster,
            YtClusterFreshnessRepository ytClusterFreshnessRepository,
            TransferManagerConfig transferManagerConfig,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.jobConfig = jobConfig;
        this.dynamicConfig = dynamicConfig;
        this.ytProvider = ytProvider;
        this.dstYtCluster = dstYtCluster;
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;
        this.transferManager = new TransferManager(transferManagerConfig);
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    /**
     * Возвращает true, если на аккаунте, в котором размещена home-директория, есть достаточно
     * свободного места (1.5 * размер директории с данными синхронизатора).
     * false в противном случае
     */
    private boolean isEnoughSpace(YtCluster cluster) {
        String home = ytProvider.getClusterConfig(cluster).getHome();
        Yt yt = ytProvider.getOperator(cluster).getYt();
        String account = YtUtils.getYtAccount(yt, YPath.simple(home));
        YPath basedir = YPath.simple(dynamicConfig.tables().direct().baseDirectoryPath());
        String medium = jobConfig.getPrimaryMedium();
        Map<String, Long> recursiveUsageByMediums = getRecursiveUsageByMediums(yt, basedir);
        Map<String, Long> remainingSpaceByMediums = getRemainingSpaceByMediums(yt, account);
        if (!remainingSpaceByMediums.containsKey(medium)) {
            logger.error("Can't get remaining space for cluster {} on medium = {}", cluster, medium);
            return false;
        }
        long used = recursiveUsageByMediums.getOrDefault(medium, 0L);
        long remaining = remainingSpaceByMediums.get(medium);
        if (remaining < used * 3 / 2) {
            logger.info("Remaining space on cluster {} is {} and it's not enough to safe backup (need 1.5 * {} bytes)",
                    cluster, remaining, used);
            return false;
        }
        logger.info("Remaining space on cluster {}: {}, it's OK", cluster, remaining);
        return true;
    }

    private YtCluster selectSrcYtCluster() {
        Map<YtCluster, YtDynamicOperator> clusterToDynamicOperator = dynamicConfig.getClusters().stream()
                .collect(Collectors.toMap(identity(), ytProvider::getDynamicOperator));
        if (clusterToDynamicOperator.isEmpty()) {
            logger.error("No YT dyntables clusters declared in config, nothing to do");
            throw new RuntimeException("No YT dyntables clusters found");
        }

        // Определяем свежесть данных в кластерах
        List<Pair<YtCluster, Long>> clustersWithTs = new ArrayList<>();
        for (YtCluster ytCluster : clusterToDynamicOperator.keySet()) {
            YtDynamicOperator ytDynamicOperator = clusterToDynamicOperator.get(ytCluster);
            Map<Integer, Long> shardToTimestamp = ytClusterFreshnessRepository.loadShardToTimestamp(ytDynamicOperator);
            if (shardToTimestamp == null) {
                logger.info("Can't load freshness for cluster " + ytCluster);
                continue;
            }
            Long minTimestamp = shardToTimestamp.values().stream().min(Long::compare).orElseThrow(AssertionError::new);
            logger.info("Cluster {} has timestamp {}", ytCluster, minTimestamp);
            if (!isEnoughSpace(ytCluster)) {
                logger.info("Skip cluster {} because not enough space for safe backup", ytCluster);
                continue;
            }
            clustersWithTs.add(Pair.of(ytCluster, minTimestamp));
        }
        // Берём самый свежий кластер
        YtCluster srcYtCluster =
                clustersWithTs.stream().max(comparing(Pair::getRight)).orElseThrow(() -> {
                    logger.error("Need at least one suitable cluster to start backup, got 0");
                    throw new RuntimeException("No cluster suitable for backup found");
                }).getKey();

        checkState(srcYtCluster != dstYtCluster, "srcYtCluster and dstYtCluster can't be the same");
        logger.info("Found {} active replicas, starting backup using {} cluster", clustersWithTs.size(), srcYtCluster);

        return srcYtCluster;
    }

    private boolean isEnabled() {
        String jobEnabledProperty = ppcPropertiesSupport.get(MYSQL_YT_SYNC_BACKUP_JOB_ENABLED.getName());
        logger.info("Loaded ppc_property {} = {}", MYSQL_YT_SYNC_BACKUP_JOB_ENABLED.getName(), jobEnabledProperty);
        return "1".equals(jobEnabledProperty);
    }

    @Override
    public void execute() {
        logger.info("Started job with config={}", jobConfig);
        //
        if (isEnabled()) {
            // Выбираем кластер, с которого будем снимать бекап
            YtCluster srcYtCluster = selectSrcYtCluster();

            // Подготавливаем директории на src и dst. Лучше сделать это сразу, чтобы джоба упала быстрее,
            // если на создание директорий нет прав. А не обнаружить это, когда некоторая работа уже будет сделана
            String srcDirectory = prepareBackupDirectory(srcYtCluster, jobConfig.getSrcClusterCopyTtl());
            String dstDirectory = prepareBackupDirectory(dstYtCluster, jobConfig.getDstClusterCopyTtl());

            try {
                // Сначала копируем таблицу mysql-sync-states (через merge)
                String statesTablePath = copySyncStatesTable(srcYtCluster, srcDirectory);

                // Ждём, что все изменения, которые были на момент копирования mysql-sync-states, доедут до дисков
                waitForContentTablesFlushed(srcYtCluster);

                // Копируем их сразу в статические таблицы через merge
                ArrayList<String> staticTablesRelative = copyContentTables(srcYtCluster, srcDirectory);

                // Отправляем на dst кластер
                staticTablesRelative.add(statesTablePath);
                transfer(srcYtCluster, dstYtCluster, srcDirectory, dstDirectory, staticTablesRelative);
            } finally {
                // Удаляем копию на src-кластере, чтобы не занимать место (места не очень много)
                deleteDirectory(srcYtCluster, srcDirectory);
            }
        }
    }

    private void deleteDirectory(YtCluster srcYtCluster, String srcDirectory) {
        Yt yt = ytProvider.getOperator(srcYtCluster).getYt();
        logger.info("Directory {} will be deleted", srcDirectory);
        yt.cypress().remove(Optional.empty(), false, YPath.simple(srcDirectory), true, true);
        logger.info("Deleted {} OK", srcDirectory);
    }

    private String copySyncStatesTable(YtCluster srcYtCluster, String srcDirectory) {
        String baseDir = dynamicConfig.tables().direct().baseDirectoryPath();
        String table = makeRelative(dynamicConfig.tables().direct().syncStatesTablePath(), baseDir);
        return copyTableUsingMerge(srcYtCluster, baseDir, srcDirectory, table, extractSyncVersion(srcYtCluster));
    }

    /**
     * Выделяет строку с версией данных из пути, на который указывает симлинк current
     */
    private String extractSyncVersion(YtCluster srcYtCluster) {
        String baseDir = dynamicConfig.tables().direct().baseDirectoryPath();
        Yt yt = ytProvider.getOperator(srcYtCluster).getYt();
        String targetPath = yt.cypress().get(YPath.simple(baseDir + "&"), Cf.wrap(singleton("target_path")))
                .getAttribute("target_path").get().stringValue();
        return targetPath.substring(targetPath.lastIndexOf("/") + 1);
    }

    private static class TableTimeInfo {
        /**
         * Минимальный таймстемп, начиная с которого данные находятся в памяти и не сброшены на диск.
         * Все ts, меньшие unflushed_timestamp, уже должны быть в чанках на диске.
         */
        YtTimestamp unflushedTimestamp;
        /**
         * Таймстемп крайнего коммита в таблицу. По идее, это таймстемп самых свежих данных, и обычно он ≥ всех других
         * таймстемпов (кроме моментов, когда все данные сброшены на диск, в этом случае, по всей видимости,
         * unflushed_timestamp получает значение = крайний ts + 1, и становится больше last_commit_timestamp).
         */
        YtTimestamp lastCommitTimestamp;
        /**
         * Информационный атрибут, показывает, сколько ms прошло с момента, когда в памяти появились новые данные,
         * которые ещё не были сброшены на диск.
         */
        long flushLagTime;

        TableTimeInfo(YtTimestamp unflushedTimestamp, YtTimestamp lastCommitTimestamp, long flushLagTime) {
            this.unflushedTimestamp = unflushedTimestamp;
            this.lastCommitTimestamp = lastCommitTimestamp;
            this.flushLagTime = flushLagTime;
        }

        @Override
        public String toString() {
            return "TableTimeInfo{" +
                    "unflushedTimestamp=" + unflushedTimestamp +
                    ", lastCommitTimestamp=" + lastCommitTimestamp +
                    ", flushLagTime=" + flushLagTime +
                    '}';
        }
    }

    private Map<String, TableTimeInfo> getTablesTimestamps(YtCluster srcYtCluster, String baseDir,
                                                           List<String> tablesPathsRelative) {
        Map<String, TableTimeInfo> result = new HashMap<>();
        Cypress cypress = ytProvider.getOperator(srcYtCluster).getYt().cypress();
        for (String tablePathRelative : tablesPathsRelative) {
            YPath dynamicTable = YPath.simple(baseDir + "/" + tablePathRelative);
            YTreeNode node = cypress.get(dynamicTable,
                    Cf.set("unflushed_timestamp", "last_commit_timestamp", "flush_lag_time"));
            result.put(tablePathRelative, new TableTimeInfo(
                    YtTimestamp.valueOf(node.getAttribute("unflushed_timestamp").get().longValue()),
                    YtTimestamp.valueOf(node.getAttribute("last_commit_timestamp").get().longValue()),
                    node.getAttribute("flush_lag_time").get().longValue()
            ));
        }
        return result;
    }

    private void waitForContentTablesFlushed(YtCluster srcYtCluster) {
        List<String> tablesPathsRelative = getContentTablesPathsRelative(srcYtCluster);
        String baseDir = dynamicConfig.tables().direct().baseDirectoryPath();
        Map<String, TableTimeInfo> beforeWait = getTablesTimestamps(srcYtCluster, baseDir, tablesPathsRelative);
        logger.info("Got content tables timestamps before waiting: {}", beforeWait);

        Stopwatch stopwatch = Stopwatch.createStarted();
        List<String> unflushed;
        do {
            logger.info("Waiting 30 sec");
            Interrupts.failingRun(() -> Thread.sleep(30_000));
            //
            Map<String, TableTimeInfo> afterWait = getTablesTimestamps(srcYtCluster, baseDir, tablesPathsRelative);
            logger.info("Got content tables timestamps: {}", afterWait);
            unflushed = EntryStream.of(beforeWait)
                    .filter(entry -> {
                        String table = entry.getKey();
                        TableTimeInfo before = entry.getValue();
                        TableTimeInfo after = afterWait.get(table);
                        return !(after.unflushedTimestamp.getValue() > before.lastCommitTimestamp.getValue());
                    }).keys().toList();
            logger.info("Unflushed tables: {} ({} total, elapsed {} seconds)",
                    unflushed, unflushed.size(), stopwatch.elapsed(TimeUnit.SECONDS));
        } while (!unflushed.isEmpty() && stopwatch.elapsed().compareTo(WAIT_FOR_FLUSH_TIMEOUT) < 0);
        if (!unflushed.isEmpty()) {
            logger.error("Timed out waiting disk flush for tables {}", unflushed);
            throw new RuntimeException("Timed out waiting disk flush for tables");
        }
    }

    // Определяет, какие таблицы с содержимым нужно бекапить, возвращает relative-пути относительно базовой директории
    private List<String> getContentTablesPathsRelative(YtCluster ytCluster) {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        String baseDir = dynamicConfig.tables().direct().baseDirectoryPath();
        String contentDir = dynamicConfig.tables().direct().baseContentDirectoryPath();
        List<YTreeStringNode> list = ytOperator.getYt().cypress().list(YPath.simple(contentDir), Cf.set("type"));
        logger.info("Got list {}: {}", contentDir, list);
        return list.stream()
                .filter(node -> "table".equals(node.getAttribute("type").map(YTreeNode::stringValue).orElse(null)))
                .map(YTreeStringNode::getValue)
                // Делаем пути относительными
                .map(table -> makeRelative(contentDir + "/" + table, baseDir))
                .collect(toList());
    }

    private String copyTableUsingMerge(YtCluster srcYtCluster, String fromDirectory, String toDirectory,
                                       String dynamicTableRelative, String syncVersion) {
        YtOperator ytOperator = ytProvider.getOperator(srcYtCluster);

        String staticTableRelative = dynamicTableRelative + "_static";
        YPath dynamicTable = YPath.simple(fromDirectory + "/" + dynamicTableRelative);
        YPath staticTable = YPath.simple(toDirectory + "/" + staticTableRelative);

        // Создаём таблицу с такой же схемой (перезаписываем, если таблица уже есть)
        if (ytOperator.getYt().cypress().exists(staticTable)) {
            logger.info("Removing existing table {}", staticTable);
            ytOperator.getYt().cypress().remove(staticTable);
        }
        YTreeNode dynamicNode = ytOperator.getYt().cypress().get(dynamicTable, Cf.hashSet(YtUtils.SCHEMA_ATTR, "key_columns"));
        logger.info("Creating table {}", staticTable);
        ytOperator.getYt().cypress().create(staticTable, CypressNodeType.TABLE,
                true, false, Cf.map(
                        YtUtils.SCHEMA_ATTR, dynamicNode.getAttribute(YtUtils.SCHEMA_ATTR).orElseThrow(() -> new NoSuchElementException("No schema")),
                        "primary_medium", YTree.stringNode(jobConfig.getPrimaryMedium()),
                        // Так меньше места занимает
                        "optimize_for", YTree.stringNode(OptimizeForAttr.SCAN.getText()),
                        // Записываем в атрибут таблицы версию данных, с которых эта таблица была скопирована
                        "sync_version", YTree.stringNode(syncVersion)
                ));

        // Выполняем merge туда
        logger.info("Merging {} to {}", dynamicTable, staticTable);
        ListF<String> mergeBy = Cf.wrap(dynamicNode.getAttribute("key_columns").orElseThrow(() -> new NoSuchElementException("No key_columns"))
                .asList().stream().map(YTreeNode::stringValue).collect(toList()));
        GUID operationId = ytOperator.getYt().operations().merge(
                MergeSpec.builder()
                        .setMergeMode(MergeMode.SORTED)
                        .setCombineChunks(true)
                        .setMergeBy(mergeBy)
                        .setInputTables(Collections.singletonList(dynamicTable))
                        .setOutputTable(staticTable)
                        .build()
        );
        ytOperator.getYt().operations().getOperation(operationId).awaitAndThrowIfNotSuccess(Duration.ofMinutes(60));
        logger.info("Merged {} to {} OK", dynamicTable, staticTable);

        return staticTableRelative;
    }

    /**
     * Переливает данные из динамических таблиц в статические.
     * Это необходимо для копирования и последующего трансфера между датацентрами
     * (transfer manager всё равно отказывается возить динамические таблицы см тикет TM-131)
     * Статические таблицы создаются с суффиксами '_static'
     * Возвращает список относительных путей полученных статических таблиц.
     */
    private ArrayList<String> copyContentTables(YtCluster srcYtCluster, String srcDirectory) {
        ArrayList<String> staticTablesRelative = new ArrayList<>();
        List<String> tablesPathsRelative = getContentTablesPathsRelative(srcYtCluster);
        String baseDir = dynamicConfig.tables().direct().baseDirectoryPath();
        String syncVersion = extractSyncVersion(srcYtCluster);
        for (String table : tablesPathsRelative) {
            staticTablesRelative.add(copyTableUsingMerge(srcYtCluster, baseDir, srcDirectory, table, syncVersion));
        }
        return staticTablesRelative;
    }

    /**
     * Подготавливает директорию в указанном кластере YT для хранения копии данных.
     * Если директория уже существует, она очищается.
     * У директории устанавливается expiration_time по указанному ttl
     * <p>
     * В продакшене это будет путь вида //home/direct/mysql-sync-backup/2019-02-27
     * В тестинге //home/direct/test/ppc_test/mysql-sync-backup/2019-02-27
     * В devtest //home/direct/tmp/ppc_devtest/mysql-sync-backup/2019-02-27
     * В development //home/direct/tmp/elwood_development/mysql-sync-backup/2019-02-27
     */
    private String prepareBackupDirectory(YtCluster ytCluster, Duration ttl) {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        String home = ytProvider.getClusterConfig(ytOperator.getCluster()).getHome();
        String today = LocalDate.now().format(DateTimeFormatter.ISO_LOCAL_DATE);
        String todayDir = YtPathUtil.generatePath(home, relativePart(), BACKUP_DIRECTORY, today);
        YPath todayDirPath = YPath.simple(todayDir);
        if (ytOperator.getYt().cypress().exists(todayDirPath)) {
            logger.info("Directory {} already exists and will be deleted", todayDirPath);
            ytOperator.getYt().cypress().remove(todayDirPath);
            logger.info("Directory {} deleted OK", todayDirPath);
        }
        logger.info("Creating directory {} with TTL={}", todayDirPath, ttl);
        ytOperator.getYt().cypress().create(todayDirPath, CypressNodeType.MAP, true);
        ytOperator.getYt().cypress().set(
                YPath.simple(todayDir + "/@" + YtUtils.EXPIRATION_TIME_ATTR),
                Duration.ofMillis(new Date().getTime()).plus(ttl).toMillis()
        );
        ytOperator.getYt().cypress().create(todayDirPath.child("combined"), CypressNodeType.MAP, true);
        logger.info("Directory {} created OK", todayDirPath);
        return todayDir;
    }

    private String makeRelative(String fullPath, String baseDir) {
        checkState(fullPath.startsWith(baseDir));
        checkState(fullPath.length() > baseDir.length());
        return fullPath.substring(baseDir.length() + 1);
    }

    private void transfer(YtCluster srcYtCluster, YtCluster dstYtCluster,
                          String srcDirectory, String dstDirectory, List<String> tablesRelative) {
        Map<String, String> srcToDstTables = new HashMap<>();
        for (String tableRelative : tablesRelative) {
            srcToDstTables.put(srcDirectory + "/" + tableRelative, dstDirectory + "/" + tableRelative);
        }
        logger.info("Starting transfering tables {}", srcToDstTables);

        List<TransferManagerJobConfig> jobConfigs = EntryStream.of(srcToDstTables)
                .mapKeyValue((srcTable, dstTable) -> new TransferManagerJobConfig()
                        .withInputCluster(srcYtCluster.getName())
                        .withOutputCluster(dstYtCluster.getName())
                        .withInputTable(srcTable)
                        .withOutputTable(dstTable))
                .toList();
        List<String> taskIds = transferManager.queryTransferManager(jobConfigs);
        Map<String, Boolean> results = transferManager.await(taskIds, Duration.ofMinutes(60));
        if (results.containsValue(Boolean.FALSE)) {
            logger.error("Transferring FAILED: {}", results);
            throw new RuntimeException(
                    String.format("Transferring failed: only %d of %d tasks completed",
                            results.values().stream().filter(Boolean.TRUE::equals).count(),
                            results.values().size()
                    ));
        }

        logger.info("Transferring finished OK");
    }
}
