package ru.yandex.direct.jobs.yt;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.jobs.yt.model.TableData;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.direct.ytwrapper.model.YtTableRow;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtUtils.LAST_UPDATE_TIME_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.isClusterAvailable;
import static ru.yandex.inside.yt.kosher.tables.YTableEntryTypes.YSON;

public abstract class PopulateDynTableJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(PopulateDynTableJob.class);

    protected static final int CHUNK_SIZE = 10_000;

    protected final YtProvider ytProvider;
    protected final PpcProperty<Boolean> jobEnabledProperty;

    protected final Map<YtCluster, String> clusterToTablesPath;
    protected final String targetTableFullPath;
    protected final List<YtCluster> clusters;

    public PopulateDynTableJob(
            DirectConfig directConfig,
            YtProvider ytProvider,
            PpcProperty<Boolean> jobEnabledProperty
    ) {
        DirectConfig config = directConfig.getBranch(getConfigBranch());

        this.clusters = mapList(config.getStringList("clusters"), YtCluster::parse);

        this.ytProvider = ytProvider;

        String staticTablesFolder = config.getString(getParsedLogFolder());
        this.clusterToTablesPath = Collections.unmodifiableMap(
                getClusterToTablesPath(ytProvider, clusters, staticTablesFolder)
        );

        this.targetTableFullPath = YtPathUtil.generatePath(
                ytProvider.getClusterConfig(getTargetTableCluster()).getHome(), relativePart(),
                getTargetTable()
        );

        this.jobEnabledProperty = jobEnabledProperty;
    }

    @Override
    public void execute() {
        if (!jobEnabledProperty.getOrDefault(false)) {
            logger.info("Job is not enabled");
            return;
        }

        TableData tableData = findTableToProcess();
        if (tableData != null) {
            processTable(tableData);
        } else {
            logger.info("No table to process");
        }
    }

    @Nullable
    protected TableData findTableToProcess() {
        EntryStream<String, YtCluster> allTables = EntryStream.empty();

        for (YtCluster cluster : clusters) {
            Cypress cypress = ytProvider.get(cluster).cypress();
            YPath tablesYpath = YPath.simple(clusterToTablesPath.get(cluster));

            if (!isClusterAvailable(ytProvider, cluster) || !cypress.exists(tablesYpath)) {
                logger.error("Unable to get tables from cluster {}", cluster);
                continue;
            }

            ListF<String> allTablesInCluster = Cf.wrap(cypress.list(tablesYpath)).map(YTreeNode::stringValue);
            allTables = allTables.append(
                    StreamEx.of(allTablesInCluster).mapToEntry(o -> cluster)
            );
        }

        var minTableWithCluster = allTables.filterKeys(table -> parseDate(table) != null)
                .sortedBy(entry -> parseDate(entry.getKey()))
                .findFirst();

        if (minTableWithCluster.isEmpty()) {
            return null;
        }

        return new TableData(
                minTableWithCluster.get().getKey(),
                minTableWithCluster.get().getValue()
        );
    }

    /**
     * Вычитывает чанки из таблицы {@param tableData} и удаляет ее
     */
    protected void processTable(TableData tableData) {

        String tableName = tableData.getTableName();
        YtCluster cluster = tableData.getCluster();

        YtOperator ytOperator = ytProvider.getOperator(cluster);

        YtTable sourceTable = new YtTable(clusterToTablesPath.get(cluster) + "/" + tableName);
        YPath targetTableYpath = YPath.simple(targetTableFullPath);

        logger.info("Going to write chunks from table {} in cluster {}", sourceTable, cluster);

        Long tableTimestamp = LocalDateTime.parse(tableName).toEpochSecond(ZoneOffset.UTC);

        Long rowsCount = ytOperator.readTableRowCount(sourceTable);

        long chunkBegin = 0L;
        while (chunkBegin < rowsCount) {
            long chunkEnd = Math.min(chunkBegin + CHUNK_SIZE, rowsCount);
            writeChunk(ytOperator, targetTableYpath, sourceTable, chunkBegin, chunkEnd, tableTimestamp);
            chunkBegin += CHUNK_SIZE;
        }

        setLastUpdateTimeAttribute(targetTableYpath);
        ytOperator.getYt().cypress().remove(sourceTable.ypath());
    }

    private void setLastUpdateTimeAttribute(YPath tableYpath) {
        YtOperator ytOperator = ytProvider.getOperator(getTargetTableCluster());
        ytOperator.writeTableStringAttribute(tableYpath, LAST_UPDATE_TIME_ATTR, ZonedDateTime.now().toString());
    }

    private void writeChunk(YtOperator ytOperator, YPath toTableYpath, YtTable fromTable,
                            long chunkBegin, long chunkEnd, Long tableTimestamp) {
        try {
            List<YTreeMapNode> rows = new ArrayList<>();

            ytOperator.readTableByRowRange(
                    fromTable, ytTableRow -> rows.add(ytTableRow.getData()), new YtTableRow(),
                    chunkBegin, chunkEnd
            );

            List<YTreeMapNode> rowsToWrite = mapList(rows, row -> convertRow(row, tableTimestamp));

            ytProvider.get(getTargetTableCluster()).tables()
                    .insertRows(toTableYpath, true, false, false,
                            YSON, Cf.wrap(rowsToWrite).iterator());

        } catch (Exception e) {
            logger.error("Got error while writing chunk from table {}[{}, {}]:", fromTable, chunkBegin, chunkEnd, e);
            throw new RuntimeException(e);
        }
    }

    @Nullable
    private LocalDateTime parseDate(String s) {
        LocalDateTime res = null;
        try {
            res = LocalDateTime.parse(s);
        } catch (DateTimeParseException ignored) {
        }
        return res;
    }

    protected abstract String getParsedLogFolder();

    protected Map<YtCluster, String> getClusterToTablesPath(
            YtProvider ytProvider, List<YtCluster> clusters, String staticTablesFolder
    ) {
        EnumMap<YtCluster, String> clusterToTables = new EnumMap<>(YtCluster.class);

        for (YtCluster cluster : clusters) {
            String path = YtPathUtil.generatePath(
                    ytProvider.getClusterConfig(cluster).getHome(), relativePart(),
                    staticTablesFolder
            );
            clusterToTables.put(cluster, path);
        }
        return clusterToTables;
    }

    protected abstract YtCluster getTargetTableCluster();

    protected abstract String getConfigBranch();

    protected abstract String getTargetTable();

    protected abstract YTreeMapNode convertRow(YTreeMapNode node, Long timestamp);
}
