package ru.yandex.direct.jobs.yt.audit;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.http.client.HttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_COLUMNS_EXCLUDE;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_DEST_PATH;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_TABLES_EXCLUDE;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_TABLES_INCLUDE;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_TRACKER_QUEUE;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_USE_TRACKER;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.YT_CHECKSUM_JOB_REPORTING_STATE;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_HTTP_CLIENT_BEAN;
import static ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_ROBOT_ADS_AUDIT_TOKEN;
import static ru.yandex.direct.jobs.yt.audit.YtChecksumMysqlRepository.removeShard;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Выполняет сверку данных между mysql и выгрузкой mysql2yt-full
 * <p>
 * Джоба запускается для указанного источника (имени базы, например ppc:5 и приёмника - кластера YT)
 * За одно выполнение джоба выбирает одну таблицу, которую нужно проверить, и проверяет её.
 * Если проверка была успешной (вне зависимости от кол-ва найденных расхождений),
 * то в таблице yt_checksum сохранится информация о дате последней проверки.
 * И при следующем запуске джоба возьмётся за следующую таблицу.
 * Если все таблицы проверены менее чем сутки назад, джоба завершается сразу же.
 */
@Hourglass(cronExpression = "0 0/3 * * * ?", needSchedule = ProductionOnly.class)
@ParameterizedBy(parametersSource = MysqlYtCheckTaskParametersSource.class)
@ParametersAreNonnullByDefault
public class MysqlYtChecksumJob extends DirectParameterizedJob<MysqlYtCheckTask> {
    private final static Logger logger = LoggerFactory.getLogger(MysqlYtChecksumJob.class);

    private static final boolean DEFAULT_YT_CHECKSUM_USE_TRACKER = true;
    private static final String DEFAULT_YT_CHECKSUM_TRACKER_QUEUE = "TEST";

    private static final Duration LOCK_TIMEOUT = Duration.ofSeconds(120);

    private final MysqlYtCheckTaskParametersSource parametersSource;
    private final PpcPropertiesSupport propertiesSupport;
    private final DatabaseWrapperProvider databaseWrapperProvider;
    private final YtProvider ytProvider;
    private final YtChecksumMysqlRepository mysqlRepository;
    private final CuratorFrameworkProvider curatorFrameworkProvider;
    // Может быть null, если запускается из теста
    @Nullable
    private final String startrekToken;
    private HttpClient startrekHttpClient;

    public static final Duration TABLE_CHECK_INTERVAL = Duration.ofHours(24);


    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    @Autowired
    public MysqlYtChecksumJob(MysqlYtCheckTaskParametersSource parametersSource,
                              PpcPropertiesSupport propertiesSupport,
                              DatabaseWrapperProvider databaseWrapperProvider,
                              YtProvider ytProvider,
                              CuratorFrameworkProvider curatorFrameworkProvider,
                              @Nullable @Qualifier(STARTREK_ROBOT_ADS_AUDIT_TOKEN) String startrekToken,
                              @Qualifier(STARTREK_HTTP_CLIENT_BEAN) HttpClient startrekHttpClient) {
        this.parametersSource = parametersSource;
        this.propertiesSupport = propertiesSupport;
        this.databaseWrapperProvider = databaseWrapperProvider;
        this.ytProvider = ytProvider;
        this.curatorFrameworkProvider = curatorFrameworkProvider;
        this.startrekToken = startrekToken;
        this.startrekHttpClient = startrekHttpClient;
        this.mysqlRepository = new YtChecksumMysqlRepository(databaseWrapperProvider);
    }

    @Override
    public void execute() {
        String threadName = Thread.currentThread().getName();
        try {
            MysqlYtCheckTask task = parametersSource.convertStringToParam(getParam());
            String dbName = task.getDbname();
            YtCluster ytCluster = task.getYtCluster();

            // Чтобы в поле prefix в логвьювере было написано, к какой таске это относится
            Thread.currentThread().setName(dbName + "/" + ytCluster.getName());

            execute(dbName, ytCluster);
            //
        } catch (RuntimeException e) {
            logger.error("Unhandled exception, job will abort", e);
            throw e;
        } finally {
            Thread.currentThread().setName(threadName);
        }
    }

    private void execute(String dbName, YtCluster ytCluster) {
        var tablesInclude = getTablesList(YT_CHECKSUM_TABLES_INCLUDE, dbName);
        var destPath = propertiesSupport.get(YT_CHECKSUM_DEST_PATH).getOrDefault("//home/direct/mysql-sync/current");
        var useTracker = propertiesSupport.get(YT_CHECKSUM_USE_TRACKER).getOrDefault(DEFAULT_YT_CHECKSUM_USE_TRACKER);
        useTracker &= startrekToken != null;
        var queue = propertiesSupport.get(YT_CHECKSUM_TRACKER_QUEUE).getOrDefault(DEFAULT_YT_CHECKSUM_TRACKER_QUEUE);
        var tablesExclude = new HashSet<>(getTablesList(YT_CHECKSUM_TABLES_EXCLUDE, dbName));
        logger.info("Got cluster={}, path={}, tablesInclude={}, tablesExclude={}, useTracker={}, queue={}",
                ytCluster, destPath, toJson(tablesInclude), toJson(tablesExclude), useTracker, queue);

        var lastFinished = mysqlRepository.getLastFinishedIterations(dbName, ytCluster, tablesInclude);
        logger.info("Got last finished iterations from: {}", toJson(lastFinished));

        var freshnessThreshold = LocalDateTime.now().minus(TABLE_CHECK_INTERVAL);
        var freshTables = lastFinished.stream()
                .filter(iteration -> iteration.getFinishTimestamp().isAfter(freshnessThreshold))
                .map(IterationInfo::getTable)
                .map(TableChecker::parseTable)
                .collect(Collectors.toSet());
        logger.info("Fresh tables (last check was after {}): {}", freshnessThreshold, toJson(freshTables));

        var tablesNeedCheck = filterList(tablesInclude, t -> !tablesExclude.contains(t) && !freshTables.contains(t));
        logger.info("Next tables to check ({} tables): {}", tablesNeedCheck.size(), toJson(tablesNeedCheck));

        if (tablesNeedCheck.isEmpty()) {
            logger.info("Nothing to do");
            return;
        }
        var table = tablesNeedCheck.get(0);

        // Номер следующей итерации определяем как max() + 1 или 0, если таблица ещё не проверялась
        var nextIteration = Optional.ofNullable(mysqlRepository.getMaxIteration(dbName, ytCluster, table))
                .map(v -> v + 1)
                .orElse(0);

        logger.info("Will check table {}", table);
        var params = new TableCheckParams();
        params.setDb(dbName);
        params.setTable(table);
        params.setDestYtCluster(ytCluster);
        params.setDestYtPath(destPath + "/" + dbName);
        params.setIteration(nextIteration);
        params.setExcludeColumns(getExcludeColumnsFor(dbName, table));

        var ytRepository = new YtChecksumYtRepository(ytProvider);
        var tableChecker = new TableChecker(databaseWrapperProvider, params, ytRepository);
        var result = tableChecker.processTable();

        if (result.isFinished() && !result.getFailedRows().isEmpty() && useTracker) {
            writeToTracker(params, result, queue);
        }
    }

    private void writeToTracker(TableCheckParams params, TableCheckResult result, String queue) {
        var lockName = MysqlYtChecksumJob.class.getSimpleName() + "_" + Environment.getCached() + "_lock";
        logger.info("Waiting for ZK lock to track results");
        try (var ignored = curatorFrameworkProvider.getLock(lockName, LOCK_TIMEOUT, null)) {
            JobReportingState reportingState = propertiesSupport.find(YT_CHECKSUM_JOB_REPORTING_STATE.getName())
                    .map(json -> fromJson(json, JobReportingState.class))
                    .orElse(JobReportingState.empty());
            var now = new Date();
            // Если сегодня ещё не было создано тикетов
            if (reportingState.getTicket() == null || !isSameDay(reportingState.getTicketTimestamp(), now)) {
                // Создать тикет
                var ticket = createTicket(queue, composeTicketSummary(now), composeTicketMessage(params, result));
                // Сохранить стейт
                propertiesSupport.set(YT_CHECKSUM_JOB_REPORTING_STATE.getName(), toJson(new JobReportingState(ticket,
                        now)));
            } else {
                // Дописать коммент в имеющийся
                addTicketComment(reportingState.getTicket(), composeTicketMessage(params, result));
            }
        } catch (Exception e) {
            logger.error("Can't write to tracker", e);
            throw new RuntimeException(e);
        }
    }

    private String createTicket(String queue, String summary, String text) {
        checkNotNull(startrekToken);
        //
        Session session = StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .httpClient(startrekHttpClient)
                .build(startrekToken);
        //
        IssueCreate.Builder builder = IssueCreate.builder()
                .queue(queue)
                .summary(summary)
                .description(text)
                .type("task");
        IssueCreate issueCreate = builder.build();
        //
        Issue issue = session.issues().create(issueCreate);
        logger.info("Issue created: https://st.yandex-team.ru/{}", issue.getKey());
        //
        return issue.getKey();
    }

    private void addTicketComment(String ticketKey, String text) {
        checkNotNull(startrekToken);
        //
        Session session = StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .httpClient(startrekHttpClient)
                .build(startrekToken);

        Issue issue = session.issues().get(ticketKey);
        issue.comment(text);
        logger.info("Added comment to issue https://st.yandex-team.ru/{}", ticketKey);
    }

    private String composeTicketSummary(Date now) {
        return String.format("Расхождения в mysql2yt-full %s", DATE_FORMAT.format(now));
    }

    private String composeTicketMessage(TableCheckParams params, TableCheckResult result) {
        var sb = new StringBuilder();
        var src = String.format("%s/%s", params.getDb(), params.getTable());
        var dst = String.format("%s:%s/straight/%s", params.getDestYtCluster(), params.getDestYtPath(),
                params.getTable());
        sb.append("Обнаружены расхождения между\n");
        sb.append(src).append(" и\n").append(dst).append("\n\n");
        sb.append("<{Развернуть\n");
        sb.append(
                result.getFailedRows().stream()
                        .map(row -> "mysql\n%%\n" + row.getThisStr() + "\n%%\nyt\n%%\n" + row.getYtStr() + "\n%%\n")
                        .collect(Collectors.joining("\n----\n"))
        );
        sb.append("\n}>");
        sb.append("\n\n----\n\n");
        sb.append(MysqlYtChecksumJob.class.getSimpleName()).append("@").append(getLocalHostname());
        return sb.toString();
    }

    private String getLocalHostname() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            return "unknown";
        }
    }

    private boolean isSameDay(Date a, Date b) {
        GregorianCalendar calendarA = new GregorianCalendar();
        GregorianCalendar calendarB = new GregorianCalendar();
        calendarA.setTime(a);
        calendarB.setTime(b);
        return calendarA.get(Calendar.YEAR) == calendarB.get(Calendar.YEAR)
                && calendarA.get(Calendar.DAY_OF_YEAR) == calendarB.get(Calendar.DAY_OF_YEAR);
    }

    private List<String> getTablesList(PpcPropertyName<List<String>> ppcProperty, String dbName) {
        List<String> prefixedList = propertiesSupport.get(ppcProperty).getOrDefault(emptyList());
        return mysqlRepository.getTablesList(prefixedList, dbName);
    }

    private List<String> getExcludeColumnsFor(String dbName, String table) {
        var allExcludeColumns = propertiesSupport.get(YT_CHECKSUM_COLUMNS_EXCLUDE).getOrDefault(emptyList());
        logger.info("Loaded YT_CHECKSUM_COLUMNS_EXCLUDE={}", toJson(allExcludeColumns));
        var prefix = removeShard(dbName) + ":" + table + ".";
        return allExcludeColumns.stream()
                .map(String::strip)
                .filter(v -> v.startsWith(prefix))
                .map(v -> v.substring(prefix.length()))
                .collect(toList());
    }

    private static class JobReportingState {
        @Nullable
        @JsonProperty("ticket")
        private String ticket;
        @JsonProperty("ticket_timestamp")
        @JsonFormat(shape = JsonFormat.Shape.STRING, pattern = "yyyy-MM-dd'T'HH:mm:ss.SSSXXX")
        @Nullable
        private Date ticketTimestamp;

        public JobReportingState() {
        }

        public JobReportingState(@Nullable String ticket, @Nullable Date ticketTimestamp) {
            this.ticket = ticket;
            this.ticketTimestamp = ticketTimestamp;
        }

        @Nullable
        public String getTicket() {
            return ticket;
        }

        public void setTicket(@Nullable String ticket) {
            this.ticket = ticket;
        }

        @Nullable
        public Date getTicketTimestamp() {
            return ticketTimestamp;
        }

        public void setTicketTimestamp(@Nullable Date ticketTimestamp) {
            this.ticketTimestamp = ticketTimestamp;
        }

        public static JobReportingState empty() {
            return new JobReportingState();
        }
    }
}
