package ru.yandex.direct.jobs.yt.audit;

import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_TABLES_EXCLUDE;
import static ru.yandex.direct.common.db.PpcPropertyNames.YT_CHECKSUM_TABLES_INCLUDE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Собирает метрики по кол-ву проверенных/непроверенных таблиц в mysql.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_2})
@Hourglass(cronExpression = "0 0/5 * * * ?")
@ParametersAreNonnullByDefault
public class MysqlYtMetricsCollectorJob extends DirectJob {
    private final MysqlYtCheckTaskParametersSource mysqlYtCheckTasksSource;
    private final PpcPropertiesSupport propertiesSupport;
    private final SolomonPushClient solomonPushClient;
    private final YtChecksumMysqlRepository mysqlRepository;

    @Autowired
    public MysqlYtMetricsCollectorJob(MysqlYtCheckTaskParametersSource mysqlYtCheckTasksSource,
                                      PpcPropertiesSupport ppcPropertiesSupport,
                                      DatabaseWrapperProvider databaseWrapperProvider,
                                      SolomonPushClient solomonPushClient) {
        this.mysqlYtCheckTasksSource = mysqlYtCheckTasksSource;
        this.propertiesSupport = ppcPropertiesSupport;
        this.solomonPushClient = solomonPushClient;
        this.mysqlRepository = new YtChecksumMysqlRepository(databaseWrapperProvider);
    }

    @Override
    public void execute() {
        List<String> include = propertiesSupport.get(YT_CHECKSUM_TABLES_INCLUDE).getOrDefault(emptyList());
        List<String> exclude = propertiesSupport.get(YT_CHECKSUM_TABLES_EXCLUDE).getOrDefault(emptyList());

        MetricRegistry registry = SolomonUtils.newPushRegistry("flow", "yt_checksum");
        LocalDateTime threshold = LocalDateTime.now().minus(MysqlYtChecksumJob.TABLE_CHECK_INTERVAL);
        for (MysqlYtCheckTask task : mysqlYtCheckTasksSource.getAllParamValues()) {
            Set<String> tablesInclude = new HashSet<>(mysqlRepository.getTablesList(include, task.getDbname()));
            Set<String> tablesExclude = new HashSet<>(mysqlRepository.getTablesList(exclude, task.getDbname()));
            Set<String> needVerify = Sets.difference(tablesInclude, tablesExclude);

            Set<String> verified = mysqlRepository.getCheckedTables(task.getDbname(), task.getYtCluster(), threshold);

            int needVerifyCount = needVerify.size();
            int verifiedCount = Sets.intersection(needVerify, verified).size();
            int unverifiedCount = Sets.difference(needVerify, verified).size();

            Labels labels = Labels.of(
                    "yt_cluster", task.getYtCluster().getName(),
                    "dbname", task.getDbname()
            );
            registry.gaugeInt64("yt_checksum_tables_cnt", labels).set(needVerifyCount);
            registry.gaugeInt64("yt_checksum_tables_verified_cnt", labels).set(verifiedCount);
            registry.gaugeInt64("yt_checksum_tables_unverified_cnt", labels).set(unverifiedCount);
        }
        solomonPushClient.sendMetrics(registry);
    }
}
